<?php
// This file is part of mod_openbook for Moodle - http://moodle.org/
//
// It is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// It is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Handles file uploads by students!
 *
 * @package       mod_openbook
 * @author        University of Geneva, E-Learning Team
 * @author        Academic Moodle Cooperation {@link http://www.academic-moodle-cooperation.org}
 * @copyright     2025 University of Geneva {@link http://www.unige.ch}
 * @license       http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once('../../config.php');
require_once($CFG->dirroot . '/mod/openbook/locallib.php');
require_once($CFG->dirroot . '/mod/openbook/upload_form_teacher.php');

$cmid = required_param('cmid', PARAM_INT); // Course Module ID.
$id = optional_param('id', 0, PARAM_INT); // EntryID.

if (!$cm = get_coursemodule_from_id('openbook', $cmid)) {
    throw new \moodle_exception('invalidcoursemodule');
}

if (!$course = $DB->get_record('course', ['id' => $cm->course])) {
    throw new \moodle_exception('coursemisconf');
}

require_login($course, false, $cm);

$context = context_module::instance($cm->id);

require_capability('mod/openbook:upload', $context);

$openbook = new openbook($cm, $course, $context);

$url = new moodle_url('/mod/openbook/upload_teacher.php', ['cmid' => $cm->id]);
if (!empty($id)) {
    $url->param('id', $id);
}
$PAGE->set_url($url);
$PAGE->add_body_class('limitedwidth');

if (!has_capability('mod/openbook:uploadcommonteacherfile', $openbook->get_context())) {
    redirect(new moodle_url('/mod/openbook/view.php', ['id' => $cm->id]));
}

$entry = new stdClass();
$entry->id = $USER->id;

$entry->definition = '';
$entry->definitionformat = FORMAT_HTML;

$maxfiles = $openbook->get_instance()->maxfiles;
$maxbytes = $openbook->get_instance()->maxbytes;

$acceptedfiletypes = $openbook->get_accepted_types();

$definitionoptions = [
        'trusttext' => true,
        'subdirs' => false,
        'maxfiles' => $maxfiles,
        'maxbytes' => $maxbytes,
        'context' => $context,
        'accepted_types' => $acceptedfiletypes,
];
$attachmentoptions = [
        'subdirs' => false,
        'maxfiles' => $maxfiles,
        'maxbytes' => $maxbytes,
        'accepted_types' => $acceptedfiletypes,
];

$entry = file_prepare_standard_editor(
    $entry,
    'definition',
    $definitionoptions,
    $context,
    'mod_openbook',
    'entry',
    $entry->id,
);

$entry = file_prepare_standard_filemanager(
    $entry,
    'commonteacherfiles',
    $attachmentoptions,
    $context,
    'mod_openbook',
    'commonteacherfiles',
    $entry->id
);

$entry->cmid = $cm->id;

// Create a new form object (found in lib.php).
$mform = new mod_openbook_upload_form_teacher(null, [
        'current' => $entry,
        'cm' => $cm,
        'openbook' => $openbook,
        'definitionoptions' => $definitionoptions,
        'attachmentoptions' => $attachmentoptions,
]);

if ($mform->is_cancelled()) {
    redirect(new moodle_url('/mod/openbook/view.php', ['id' => $cm->id]));
} else if ($data = $mform->get_data()) {
    $entry = file_postupdate_standard_editor(
        $entry,
        'definition',
        $definitionoptions,
        $context,
        'mod_openbook',
        'entry',
        $entry->id
    );

    $entry = file_postupdate_standard_filemanager(
        $entry,
        'commonteacherfiles',
        $attachmentoptions,
        $context,
        'mod_openbook',
        'commonteacherfiles',
        $entry->id
    );
    $sid = $openbook->get_instance()->id;
    $filearea = 'commonteacherfiles';

    $fs = get_file_storage();

    $files = $fs->get_area_files($context->id, 'mod_openbook', $filearea, $sid, 'timemodified', false);

    $values = [];
    foreach ($files as $file) {
        $values[] = $file->get_id();
    }

    $filescount = count($values);
    $rows = $DB->get_records('openbook_file', ['openbook' => $openbook->get_instance()->id,
        'userid' => $USER->id]);

    // Find new files and store in db.
    foreach ($files as $file) {
        $found = false;

        foreach ($rows as $row) {
            if ($row->fileid == $file->get_id()) {
                $found = true;
            }
        }

        if (!$found) {
            $dataobject = new stdClass();
            $dataobject->openbook = $openbook->get_instance()->id;
            $dataobject->userid = $USER->id;
            $dataobject->timecreated = $file->get_timecreated();
            $dataobject->fileid = $file->get_id();
            $dataobject->studentapproval = 0;
            $dataobject->teacherapproval = 0;
            $dataobject->filename = $file->get_filename();
            $dataobject->type = OPENBOOK_MODE_UPLOAD;
            if (has_capability('mod/openbook:uploadcommonteacherfile', $context)) {
                $dataobject->commonteacherfile = true;
            } else {
                $dataobject->commonteacherfile = false;
            }

            $dataobject->id = $DB->insert_record('openbook_file', $dataobject);

            if ($openbook->get_instance()->notifyfilechange != 0) {
                openbook::send_notification_filechange($cm, $dataobject, null, $openbook);
            }

            \mod_openbook\event\openbook_file_uploaded::create_from_object($cm, $dataobject)->trigger();
        }
    }

    // Find deleted files and update db.
    foreach ($rows as $idx => $row) {
        $found = false;
        foreach ($files as $file) {
            if ($file->get_id() == $row->fileid) {
                $found = true;
                continue;
            }
        }

        if (!$found) {
            $dataobject = $DB->get_record('openbook_file', ['id' => $row->id]);
            \mod_openbook\event\openbook_file_deleted::create_from_object($cm, $dataobject)->trigger();
            $DB->delete_records('openbook_file', ['id' => $row->id]);
        }
    }

    // Update completion status - if filescount == 0 => activity not completed, else => activity completed.

    $completion = new completion_info($course);
    if ($completion->is_enabled($cm) && $openbook->get_instance()->completionupload) {
        if ($filescount == 0) {
            $completion->update_state($cm, COMPLETION_INCOMPLETE, $USER->id);
        } else {
            $completion->update_state($cm, COMPLETION_COMPLETE, $USER->id);
        }
    }
    openbook::send_all_pending_notifications();
    redirect(new moodle_url('/mod/openbook/view.php', ['id' => $cm->id]));
}

// Load existing files into draft area.

echo $OUTPUT->header();

$mform->display();

echo $OUTPUT->footer();
