<?php
// This file is part of mod_publication for Moodle - http://moodle.org/
//
// It is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// It is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Instance settings form.
 *
 * @package       mod_publication
 * @author        Philipp Hager
 * @author        Andreas Windbichler
 * @copyright     2014 Academic Moodle Cooperation {@link http://www.academic-moodle-cooperation.org}
 * @license       http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

global $CFG;

require_once($CFG->dirroot . '/course/moodleform_mod.php');
require_once($CFG->dirroot . '/mod/publication/locallib.php');

/**
 * Form for creating and editing mod_publication instances
 *
 * @package       mod_publication
 * @author        Philipp Hager
 * @author        Andreas Windbichler
 * @copyright     2014 Academic Moodle Cooperation {@link http://www.academic-moodle-cooperation.org}
 * @license       http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class mod_publication_mod_form extends moodleform_mod {
    /**
     * @var array List of assignment IDs that have team submissions enabled.
     */
    private $teamassigns;

    /**
     * Define this form - called by the parent constructor
     */
    public function definition() {
        global $DB, $CFG, $COURSE, $PAGE;

        $mform = $this->_form;
        $mform->addElement('header', 'general', get_string('general', 'form'));

        // Name.
        $mform->addElement('text', 'name', get_string('name', 'publication'), ['size' => '64']);
        if (!empty($CFG->formatstringstriptags)) {
            $mform->setType('name', PARAM_TEXT);
        } else {
            $mform->setType('name', PARAM_CLEANHTML);
        }
        $mform->addRule('name', null, 'required', null, 'client');

        // Adding the standard "intro" and "introformat" fields!
        $this->standard_intro_elements();

        // Publication specific elements.
        $mform->addElement('header', 'submissionsettings', get_string('submissionsettings', 'publication'));
        $mform->setExpanded('submissionsettings');

        if (isset($this->current->id) && $this->current->id != "") {
            $filecount = $DB->count_records('publication_file', ['publication' => $this->current->id]);
        } else {
            $filecount = 0;
        }

        $disabled = [];
        if ($filecount > 0) {
            $disabled['disabled'] = 'disabled';
        }

        $modearray = [];
        $modearray[] =& $mform->createElement(
            'radio',
            'mode',
            '',
            get_string('modeupload', 'publication'),
            PUBLICATION_MODE_UPLOAD,
            $disabled
        );
        $modearray[] =& $mform->createElement(
            'radio',
            'mode',
            '',
            get_string('modeimport', 'publication'),
            PUBLICATION_MODE_IMPORT,
            $disabled
        );
        $mform->addGroup($modearray, 'modegrp', get_string('mode', 'publication'), [' '], false);
        $mform->addHelpButton('modegrp', 'mode', 'publication');
        if ($filecount === 0) {
            $mform->addRule('modegrp', null, 'required', null, 'client');
        }

        // Publication mode import specific elements.
        $choices = [];
        $choices[-1] = get_string('choose', 'publication');
        $assigninstances = $DB->get_records('assign', ['course' => $COURSE->id], 'name ASC');
        $module = $DB->get_record('modules', ['name' => 'assign']);
        $select = $mform->createElement('select', 'importfrom', get_string('assignment', 'publication'), $choices, $disabled);
        $notteamassigns = [-1];
        $teamassigns = [];
        foreach ($assigninstances as $assigninstance) {
            $cm = $DB->get_record('course_modules', ['module' => $module->id, 'instance' => $assigninstance->id]);
            if ($cm->deletioninprogress == 1) {
                continue;
            }
            if (!$assigninstance->teamsubmission) {
                $notteamassigns[] = $assigninstance->id;
            } else {
                $teamassigns[] = $assigninstance->id;
            }
            $attributes = ['data-teamsubmission' => $assigninstance->teamsubmission];
            $select->addOption($assigninstance->name, $assigninstance->id, $attributes);
        }
        $this->teamassigns = $teamassigns;
        $mform->addElement($select);
        $mform->addHelpButton('importfrom', 'assignment', 'publication');
        $mform->hideIf('importfrom', 'mode', 'neq', PUBLICATION_MODE_IMPORT);
        $mform->addElement('html', '<span id="teamassignids" data-assignids="' . implode(',', $teamassigns) . '"></span>');

        // Publication mode upload specific elements.
        $maxfiles = [];
        for ($i = 1; $i <= 100 || $i <= get_config('publication', 'maxfiles'); $i++) {
            $maxfiles[$i] = $i;
        }

        $mform->addElement('select', 'maxfiles', get_string('maxfiles', 'publication'), $maxfiles);
        $mform->addHelpButton('maxfiles', 'maxfiles', 'publication');
        $mform->hideIf('maxfiles', 'mode', 'neq', PUBLICATION_MODE_UPLOAD);

        $choices = get_max_upload_sizes($CFG->maxbytes, $COURSE->maxbytes);
        $choices[0] = get_string('courseuploadlimit', 'publication') . ' (' . display_size($COURSE->maxbytes) . ')';
        $mform->addElement('select', 'maxbytes', get_string('maxbytes', 'publication'), $choices);
        $mform->addHelpButton('maxbytes', 'maxbytes', 'publication');
        $mform->hideIf('maxbytes', 'mode', 'neq', PUBLICATION_MODE_UPLOAD);

        $mform->addElement('filetypes', 'allowedfiletypes', get_string('allowedfiletypes', 'publication'));
        $mform->addHelpButton('allowedfiletypes', 'allowedfiletypes', 'publication');
        $mform->hideIf('allowedfiletypes', 'mode', 'neq', PUBLICATION_MODE_UPLOAD);

        $name = get_string('allowsubmissionsfromdate', 'publication');
        $mform->addElement('date_time_selector', 'allowsubmissionsfromdate', $name, ['optional' => true]);
        $mform->addHelpButton('allowsubmissionsfromdate', 'allowsubmissionsfromdate', 'publication');
        $mform->hideIf('allowsubmissionsfromdate', 'mode', 'neq', PUBLICATION_MODE_UPLOAD);

        $name = get_string('duedate', 'publication');
        $mform->addElement('date_time_selector', 'duedate', $name, ['optional' => true]);
        $mform->addHelpButton('duedate', 'duedate', 'publication');
        $mform->hideIf('duedate', 'mode', 'neq', PUBLICATION_MODE_UPLOAD);

        $mform->addElement('hidden', 'cutoffdate', false);
        $mform->setType('cutoffdate', PARAM_BOOL);

        // Approval settings start.
        $mform->addElement('header', 'approvalsettings', get_string('approvalsettings', 'publication'));
        $mform->setExpanded('approvalsettings', true);

        // Teacher approval.
        $attributes = [];
        $options = [
            '0' => get_string('obtainapproval_automatic', 'publication'),
            '1' => get_string('obtainapproval_required', 'publication'),
        ];

        $mform->addElement(
            'select',
            'obtainteacherapproval',
            get_string('obtainteacherapproval', 'publication'),
            $options,
            $attributes
        );
        $mform->addHelpButton('obtainteacherapproval', 'obtainteacherapproval', 'publication');

        // Student approval.
        $attributes = [];
        $options = [
            '0' => get_string('obtainapproval_automatic', 'publication'),
            '1' => get_string('obtainapproval_required', 'publication'),
        ];

        $mform->addElement(
            'select',
            'obtainstudentapproval',
            get_string('obtainstudentapproval', 'publication'),
            $options,
            $attributes
        );
        $mform->addHelpButton('obtainstudentapproval', 'obtainstudentapproval', 'publication');

        // Group approval.
        $attributes = [];
        $options = [
            PUBLICATION_APPROVAL_GROUPAUTOMATIC => get_string('obtainapproval_automatic', 'publication'),
            PUBLICATION_APPROVAL_SINGLE => get_string('obtaingroupapproval_single', 'publication'),
            PUBLICATION_APPROVAL_ALL => get_string('obtaingroupapproval_all', 'publication'),
        ];

        $mform->addElement(
            'select',
            'obtaingroupapproval',
            get_string('obtaingroupapproval', 'publication'),
            $options,
            $attributes
        );
        $mform->addHelpButton('obtaingroupapproval', 'obtaingroupapproval', 'publication');

        $mform->addElement(
            'date_time_selector',
            'approvalfromdate',
            get_string('approvalfromdate', 'publication'),
            ['optional' => true]
        );
        $mform->addHelpButton('approvalfromdate', 'approvalfromdate', 'publication');

        $mform->addElement(
            'date_time_selector',
            'approvaltodate',
            get_string('approvaltodate', 'publication'),
            ['optional' => true]
        );
        $mform->addHelpButton('approvaltodate', 'approvaltodate', 'publication');
        // Approval code end.

        $mform->addElement('hidden', 'alwaysshowdescription', true);
        $mform->setType('alwaysshowdescription', PARAM_BOOL);

        // Apply availability restrictions.
        $mform->addElement(
            'select',
            'availabilityrestriction',
            get_string('availabilityrestriction', 'publication'),
            [get_string('no'), get_string('yes')]
        );
        $mform->addHelpButton('availabilityrestriction', 'availabilityrestriction', 'publication');

        $mform->addElement('header', 'notifications', get_string('notifications', 'publication'));

        $options = [
            PUBLICATION_NOTIFY_NONE => get_string('notify:setting:0', 'publication'),
            PUBLICATION_NOTIFY_TEACHER => get_string('notify:setting:1', 'publication'),
            PUBLICATION_NOTIFY_STUDENT => get_string('notify:setting:2', 'publication'),
            PUBLICATION_NOTIFY_ALL => get_string('notify:setting:3', 'publication'),
        ];

        $mform->addElement('select', 'notifyfilechange', get_string('notify:filechange', 'publication'), $options);
        $mform->addHelpButton('notifyfilechange', 'notify:filechange', 'publication');

        $mform->addElement('select', 'notifystatuschange', get_string('notify:statuschange', 'publication'), $options);
        $mform->addHelpButton('notifystatuschange', 'notify:statuschange', 'publication');

        // Standard coursemodule elements.
        $this->standard_coursemodule_elements();

        $this->apply_admin_defaults();

        // Buttons.
        $this->add_action_buttons();
        $PAGE->requires->js_call_amd('mod_publication/modform');
    }


    /**
     * Add any custom completion rules to the form.
     *
     * @return array Contains the names of the added form elements
     */
    public function add_completion_rules() {
        $mform =& $this->_form;

        $suffix = $this->get_suffix();
        $completionuploadlabel = 'completionupload' . $suffix;

        $mform->addElement('advcheckbox', $completionuploadlabel, '', get_string('completionupload', 'publication'));
        // Enable this completion rule by default.
        $mform->setDefault($completionuploadlabel, 1);
        $mform->hideIf($completionuploadlabel, 'mode', 'neq', PUBLICATION_MODE_UPLOAD);

        $completionassignsubmission = 'completionassignsubmission' . $suffix;
        $mform->addElement('advcheckbox', $completionassignsubmission, '', get_string('completionassignsubmission', 'publication'));
        // Enable this completion rule by default.
        $mform->setDefault($completionassignsubmission, 1);
        $mform->hideIf($completionassignsubmission, 'mode', 'neq', PUBLICATION_MODE_IMPORT);
        return [$completionuploadlabel, $completionassignsubmission];
    }

    /**
     * Determines whether the custom completion rule is enabled based on the provided data.
     *
     * @param array $data The form data to check.
     * @return bool True if the completion rule is enabled, false otherwise.
     */
    public function completion_rule_enabled($data) {
        $suffix = $this->get_suffix();
        $completionuploadlabel = 'completionupload' . $suffix;
        $completionassignsubmissionlabel = 'completionassignsubmission' . $suffix;
        if (
            ($data['mode'] == PUBLICATION_MODE_UPLOAD && !empty($data[$completionuploadlabel])) ||
            ($data['mode'] == PUBLICATION_MODE_IMPORT && !empty($data[$completionassignsubmissionlabel]))
        ) {
            return true;
        }
        return false;
    }

    /**
     * Processes form data after submission to adjust values as needed before saving.
     *
     * @param object $data The submitted form data object.
     * @return void
     */
    public function data_postprocessing($data) {
        global $DB;
        parent::data_postprocessing($data);
        $suffix = $this->get_suffix();
        $completionuploadlabel = 'completionupload' . $suffix;
        if (!isset($data->mode) || $data->mode != PUBLICATION_MODE_UPLOAD) {
            $data->{$completionuploadlabel} = 0;
        }
        $completionassignsubmissionlabel = 'completionassignsubmission' . $suffix;
        if (!isset($data->mode) || $data->mode != PUBLICATION_MODE_IMPORT) {
            $data->{$completionassignsubmissionlabel} = 0;
        }

        $data->groupapproval = 0;
        if ($data->mode == PUBLICATION_MODE_IMPORT && $data->importfrom != -1) {
            $assigninstance = $DB->get_record('assign', ['id' => $data->importfrom], '*', MUST_EXIST);
            if ($assigninstance->teamsubmission) {
                if ($data->obtaingroupapproval == PUBLICATION_APPROVAL_GROUPAUTOMATIC) {
                    $data->groupapproval = 0;
                    $data->obtainstudentapproval = 0;
                } else {
                    $data->obtainstudentapproval = 1;
                    $data->groupapproval = $data->obtaingroupapproval;
                }
            }
        }
    }

    /**
     * Prepares default values for the form before it is displayed to the user.
     *
     * @param array $defaultvalues The array of default values to preprocess (passed by reference).
     * @return void
     */
    public function data_preprocessing(&$defaultvalues) {
        global $DB;
        // phpcs:disable moodle.Commenting.TodoComment
        parent::data_preprocessing($defaultvalues); // TODO: Change the autogenerated stub.

        if (isset($defaultvalues['mode']) && $defaultvalues['mode'] == PUBLICATION_MODE_IMPORT) {
            $assign = $DB->get_record('assign', ['id' => $defaultvalues['importfrom']]);
            if ($assign && $assign->teamsubmission) {
                if ($defaultvalues['obtainstudentapproval'] == 0) {
                    $defaultvalues['obtaingroupapproval'] = PUBLICATION_APPROVAL_GROUPAUTOMATIC;
                } else {
                    $defaultvalues['obtaingroupapproval'] = $defaultvalues['groupapproval'];
                }
            }
        }
    }

    /**
     * Perform minimal validation on the settings form
     *
     * @param array $data
     * @param array $files
     * @return string[] errors
     */
    public function validation($data, $files) {
        $errors = parent::validation($data, $files);

        if ($data['allowsubmissionsfromdate'] && $data['duedate']) {
            if ($data['allowsubmissionsfromdate'] > $data['duedate']) {
                $errors['duedate'] = get_string('duedatevalidation', 'publication');
            }
        }
        if ($data['duedate'] && $data['cutoffdate']) {
            if ($data['duedate'] > $data['cutoffdate']) {
                $errors['cutoffdate'] = get_string('cutoffdatevalidation', 'publication');
            }
        }
        if ($data['allowsubmissionsfromdate'] && $data['cutoffdate']) {
            if ($data['allowsubmissionsfromdate'] > $data['cutoffdate']) {
                $errors['cutoffdate'] = get_string('cutoffdatefromdatevalidation', 'publication');
            }
        }

        if ($data['approvalfromdate'] && $data['approvaltodate']) {
            $studentapprovalrequired = $data['obtainstudentapproval'] == 1;
            if ($data['mode'] == PUBLICATION_MODE_IMPORT && in_array($data['importfrom'], $this->teamassigns)) {
                $studentapprovalrequired = $data['obtaingroupapproval'] != -1;
            }
            if ($studentapprovalrequired && $data['approvalfromdate'] > $data['approvaltodate']) {
                $errors['approvaltodate'] = get_string('approvaltodatevalidation', 'publication');
            }
        }

        if ($data['mode'] == PUBLICATION_MODE_IMPORT) {
            if ($data['importfrom'] == '0' || $data['importfrom'] == '-1') {
                $errors['importfrom'] = get_string('importfrom_err', 'publication');
            }
        }

        return $errors;
    }
}
