<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Renderer for outputting the weeksrev course format.
 *
 * @package format_weeksrev
 * @copyright 2018 Arnaud Trouvé <moodle@arnaudtrouve.fr>
 *            based on code by Mat Cannings
 *            based on code from 2012 Dan Poltawski
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace format_weeksrev\output;

use core_courseformat\output\section_renderer;
use core_courseformat\output\local\content\section;
use core_courseformat\output\local\content\section\cmlist;
use core_courseformat\output\local\content\section\availability;
use core_courseformat\output\local\content\section\summary;
use core_courseformat\output\local\content\addsection;
use html_writer;

defined('MOODLE_INTERNAL') || die();
require_once($CFG->dirroot.'/course/format/weeksrev/classes/output/renderer.php');
require_once($CFG->dirroot.'/course/format/weeksrev/lib.php');

/**
 * Basic renderer for weeksrev format.
 *
 * @copyright 2018 Arnaud Trouvé <moodle@arnaudtrouve.fr>
 *            based on code by Mat Cannings
 *            based on code from 2012 Dan Poltawski
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class renderer extends section_renderer {

    /**
     * Generate the starting container html for a list of sections
     * @return string HTML to output.
     */
    protected function start_section_list() {
        return html_writer::start_tag('ul', ['class' => 'weeksrev']);
    }

    /**
     * Generate the closing container html for a list of sections
     * @return string HTML to output.
     */
    protected function end_section_list() {
        return html_writer::end_tag('ul');
    }
    /**
     * Generate the title for this section page
     * @return string the page title
     */
    protected function page_title() {
        return get_string('weeklyoutline', 'format_weeksrev');
    }

    /**
     * Generate the section title, wraps it in a link to the section page if page is to be displayed on a separate page
     *
     * @param stdClass $section The course_section entry from DB
     * @param stdClass $course The course entry from DB
     * @return string HTML to output.
     */
    public function section_title($section, $course) {
        return $this->render(course_get_format($course)->inplace_editable_render_section_name($section));
    }

    /**
     * Generate the section title to be displayed on the section page, without a link
     *
     * @param stdClass $section The course_section entry from DB
     * @param stdClass $course The course entry from DB
     * @return string HTML to output.
     */
    public function section_title_without_link($section, $course) {
        return $this->render(course_get_format($course)->inplace_editable_render_section_name($section, false));
    }
    /**
     * Generate the display of the header part of a section before
     * course modules are included
     *
     * @param stdClass $section The course_section entry from DB
     * @param stdClass $course The course entry from DB
     * @param bool $onsectionpage true if being printed on a single-section page
     * @param int $sectionreturn The section to return to after an action
     * @return string HTML to output.
     */
    protected function section_header($section, $course, $onsectionpage, $sectionreturn=null) {
        $o = '';
        $sectionstyle = '';
        $courseformat = course_get_format($course);

        if ($section->section != 0) {
            // Only in the non-general sections.
            if (!$section->visible) {
                $sectionstyle .= ' hidden';
            }
            if ($courseformat->is_section_current($section)) {
                $sectionstyle .= ' current';
            } else if ($course->hidefuture && $courseformat->is_section_future($section)) {
                $sectionstyle .= ' future';
            }
        }

        $o .= html_writer::start_tag('li', ['id' => 'section-' . $section->section,
            $sectionstyle, 'role' => 'region',
            'aria-label' => get_section_name($course, $section)]);

        // Create a span that contains the section title to be used to create the keyboard section move menu.
        $o .= html_writer::tag('span', get_section_name($course, $section), ['class' => 'hidden sectionname']);

        $sectionoutput = new section($courseformat, $section);
        $o .= $this->render($sectionoutput);

        // When not on a section page, we display the section titles except the general section if null.
        $hasnamenotsecpg = (!$onsectionpage && ($section->section != 0 || !is_null($section->name)));

        // When on a section page, we only display the general section title, if title is not the default one.
        $hasnamesecpg = ($onsectionpage && ($section->section == 0 && !is_null($section->name)));

        $classes = ' accesshide';
        if ($hasnamenotsecpg || $hasnamesecpg) {
            $classes = '';
        }

        $availability = new availability($courseformat, $section);
        $o .= $this->render($availability);

        $o .= html_writer::start_tag('div', ['class' => 'summary']);
        $summary = new summary($courseformat, $section);
        $o .= $summary->format_summary_text();
        $o .= html_writer::end_tag('div');

        return $o;
    }
    /**
     * Generate a summary of a section for display on the 'coruse index page'
     *
     * @param stdClass $section The course_section entry from DB
     * @param stdClass $course The course entry from DB
     * @param array    $mods (argument not used)
     * @return string HTML to output.
     */
    protected function section_summary($section, $course, $mods) {
        $courseformat = course_get_format($course);
        $classattr = 'section main section-summary clearfix';
        $linkclasses = '';

        // If section is hidden then display grey section link
        if (!$section->visible) {
            $classattr .= ' hidden';
            $linkclasses .= ' dimmed_text';
        }
        if ($courseformat->is_section_current($section)) {
            $classattr .= ' current';
        } else if ($course->hidefuture && $courseformat->is_section_future($section)) {
            $classattr .= ' future';
        }

        $title = get_section_name($course, $section);
        $o = '';
        $o .= html_writer::start_tag('li', ['id' => 'section-'.$section->section,
            'class' => $classattr, 'role' => 'region', 'aria-label' => $title]);

        $o .= html_writer::tag('div', '', ['class' => 'left side']);
        $o .= html_writer::tag('div', '', ['class' => 'right side']);
        $o .= html_writer::start_tag('div', ['class' => 'content']);

        if ($section->uservisible) {
            $title = html_writer::tag('a', $title,
                ['href' => course_get_url($course, $section->section), 'class' => $linkclasses]);
        }
        $o .= $this->output->heading($title, 3, 'section-title');

        $o .= html_writer::start_tag('div', ['class' => 'summarytext']);
        $summary = new summary($courseformat, $section);
        $o .= $summary->format_summary_text();
        $o .= html_writer::end_tag('div');
        $o .= $this->section_activity_summary($section, $course, null);

        $availability = new availability($courseformat, $section);
        $o .= $this->render($availability);

        $o .= html_writer::end_tag('div');
        $o .= html_writer::end_tag('li');

        return $o;
    }
    /**
     * Output the html for a single section page .
     *
     * @param stdClass $course The course entry from DB
     * @param array $sections (argument not used)
     * @param array $mods (argument not used)
     * @param array $modnames (argument not used)
     * @param array $modnamesused (argument not used)
     * @param int $displaysection The section number in the course which is being displayed
     */
    public function print_single_section_page($course, $sections, $mods, $modnames, $modnamesused, $displaysection) {
        $context = \context_course::instance($course->id);
        $canviewhidden = has_capability('moodle/course:viewhiddensections', $context);
        if ($course->hidefuture && $canviewhidden) {
            echo html_writer::tag('ul',
                html_writer::tag('li', get_string('futureweeks', 'format_weeksrev'), ['class' => 'future-legend']),
                ['class' => 'weeksrev']
            );
        }
        parent::print_single_section_page($course, $sections, $mods, $modnames, $modnamesused, $displaysection);
    }
    /**
     * Output the html for a multiple section page
     *
     * @param stdClass $course The course entry from DB
     * @param array $sections (argument not used)
     * @param array $mods (argument not used)
     * @param array $modnames (argument not used)
     * @param array $modnamesused (argument not used)
     */
    public function print_multiple_section_page($course, $sections, $mods, $modnames, $modnamesused) {
        global $PAGE;
        $modinfo = get_fast_modinfo($course);
        $courseformat = course_get_format($course);
        $course = $courseformat->get_course();

        $context = \context_course::instance($course->id);
        // Title with completion help icon.
        $completioninfo = new \completion_info($course);
        echo $this->output->help_icon('completion', 'completion');
        echo $this->output->heading($this->page_title(), 2, 'accesshide');

        $canviewhidden = has_capability('moodle/course:viewhiddensections', $context);
        if ($course->hidefuture && $canviewhidden) {
            echo \html_writer::tag('ul',
                \html_writer::tag('li', get_string('futureweeks', 'format_weeksrev'), ['class' => 'future-legend']),
                ['class' => 'weeksrev']
            );
        }
        // Now the list of sections..
        echo $this->start_section_list();
        $numsections = $courseformat->get_last_section_number();

        $sectionsinfo = $modinfo->get_section_info_all();
        if (!$PAGE->user_is_editing()) {
            $sectionsinfo = array_reverse($modinfo->get_section_info_all(), true);
            $sectionsinfo = [array_pop($sectionsinfo)] + $sectionsinfo; // Move section 0 back to top.
        }

        foreach ($sectionsinfo as $section => $thissection) {
            if ($section == 0) {
                // 0-section is displayed a little different then the others.
                if ($thissection->summary or !empty($modinfo->sections[0]) or $PAGE->user_is_editing()) {
                    echo $this->section_header($thissection, $course, false, 0);
                }
                continue;
            }
            if ($section > $numsections) {
                // Activities inside this section are 'orphaned', this section will be printed as 'stealth' below.
                continue;
            }

            // Show the section if the user is permitted to access it, OR if it's not available
            // but there is some available info text which explains the reason & should display.
            $showsection = $thissection->uservisible ||
                ($thissection->visible && !$thissection->available &&
                    !empty($thissection->availableinfo));
            if (!$showsection) {
                // If the hiddensections option is set to 'show hidden sections in collapsed
                // form', then display the hidden section message - UNLESS the section is
                // hidden by the availability system, which is set to hide the reason.
                if (!$course->hiddensections && $thissection->available) {
                    echo $this->section_hidden($section, $course->id);
                }

                continue;
            }

            if (!$PAGE->user_is_editing() && $course->coursedisplay == COURSE_DISPLAY_MULTIPAGE) {
                // Display section summary only.
                echo $this->section_summary($thissection, $course, null);
            } else {
                echo $this->section_header($thissection, $course, false, 0);
            }
        }

        if ($PAGE->user_is_editing() and has_capability('moodle/course:update', $context)) {
            // Print stealth sections if present.
            foreach ($modinfo->get_section_info_all() as $section => $thissection) {
                if ($section <= $numsections or empty($modinfo->sections[$section])) {
                    // this is not stealth section or it is empty
                    continue;
                }
                echo $this->stealth_section_header($section);
            }

            echo $this->end_section_list();

            $addsection = new addsection($courseformat);
            echo $this->render($addsection);
        } else {
            echo $this->end_section_list();
        }
    }
}
