define(['core/ajax', 'core/notification'], function(Ajax, Notification) {
    'use strict';

    var Charts = {
        _container: null,
        _resultsAll: [],
        _showAllResults: false,
        _resultsLimit: 4,
        
        init: function(courseid) {
            this._container = document.querySelector('.progress-analytics-block');
            if (this._container) {
                const limitAttr = parseInt(this._container.getAttribute('data-resultslimit'), 10);
                if (!isNaN(limitAttr) && limitAttr > 0) {
                    this._resultsLimit = limitAttr;
                }
                // Cache localized labels from data attributes.
                this._labels = {
                    notstarted: this._container.getAttribute('data-label-notstarted') || 'Not started',
                    completed: this._container.getAttribute('data-label-completed') || 'Completed',
                    noresults: this._container.getAttribute('data-label-noresults') || 'No results available',
                    noresultsdesc: this._container.getAttribute('data-label-noresultsdesc') || 'You have not completed any quizzes yet',
                    personalavg: this._container.getAttribute('data-label-personalavg') || 'Personal average',
                    quizzes: this._container.getAttribute('data-label-quizzes') || 'quizzes',
                    myaverage: this._container.getAttribute('data-label-myaverage') || 'My average',
                    courseaverage: this._container.getAttribute('data-label-courseaverage') || 'Course average',
                    comparisonpending: this._container.getAttribute('data-label-comparisonpending') || 'Comparison data will appear when more students complete quizzes',
                    courseavgprefix: this._container.getAttribute('data-label-courseavgprefix') || 'Course average:',
                    yourpercentileprefix: this._container.getAttribute('data-label-yourpercentileprefix') || 'Your percentile:'
                };
            }
            this._bindToggle();
            this.loadData(courseid);
        },

        loadData: function(courseid) {
            const container = this._container || document.querySelector('.progress-analytics-block');
            if (!container) {
                console.error('Progress analytics container not found');
                return;
            }

            Ajax.call([{
                methodname: 'block_progressanalytics_get_course_quiz_metrics',
                args: { courseid: courseid }
            }])[0].then(data => {
                console.log('Received data:', data);
                console.log('Progress data:', data.progress);
                console.log('Results data:', data.results, 'Length:', data.results ? data.results.length : 0);
                console.log('Comparison data:', data.comparison);
                
                // TEMPORAL: Add test data if no results to test chart rendering
                if (data.results && data.results.length === 0 && data.progress.completed > 0) {
                    console.log('TEMPORAL: Adding test data to verify chart rendering');
                    data.results = [
                        {name: "Cuestionario 1", grade: 85.5, date: Date.now()/1000},
                        {name: "Cuestionario 2", grade: 72.0, date: Date.now()/1000}
                    ];
                    data.comparison.studentMean = 78.8;
                    data.comparison.courseMean = 68.5;
                    data.comparison.percentile = 75;
                    data.comparison.hasComparison = true;
                }
                
                this.hideLoading();
                
                if (data.progress.total === 0) {
                    console.log('No quizzes found, showing no quizzes message');
                    this.showNoQuizzes();
                    return;
                }

                console.log('Showing content and rendering charts');
                this.showContent();
                // Keep full results list for toggling between limited/all views.
                this._resultsAll = Array.isArray(data.results) ? data.results : [];
                this._updateToggleVisibility();
                // Render charts with limited or full results as needed.
                const dataForRender = Object.assign({}, data, { results: this._getVisibleResults() });
                this.renderCharts(dataForRender);
            }).catch(error => {
                console.error('Error loading quiz metrics:', error);
                this.showError();
            });
        },

        renderCharts: function(data) {
            console.log('Starting to render charts with data:', data);
            this.renderProgressChart(data.progress);
            this.renderResultsChart(data.results);
            this.renderComparisonChart(data.comparison);
        },

        renderProgressChart: function(progress) {
            const container = document.getElementById('progress-chart');
            if (!container) {
                return;
            }

            const percentage = progress.percentage || 0;
            const completed = progress.completed || 0;
            const total = progress.total || 0;

            // Create a simple but visible circular progress
            const circumference = 2 * Math.PI * 70;
            const offset = circumference - (percentage / 100) * circumference;
            
            // Choose colors based on progress
            let progressColor = '#dc3545'; // Red for 0%
            if (percentage >= 25) progressColor = '#fd7e14'; // Orange for 25%+
            if (percentage >= 50) progressColor = '#ffc107'; // Yellow for 50%+
            if (percentage >= 75) progressColor = '#28a745'; // Green for 75%+

            container.innerHTML = `
                <div style="display: flex; justify-content: center; align-items: center; height: 200px; flex-direction: column;">
                    <div style="position: relative; width: 160px; height: 160px;">
                        <svg width="160" height="160" style="transform: rotate(-90deg);">
                            <circle cx="80" cy="80" r="70" stroke="#e9ecef" stroke-width="12" fill="none"/>
                            <circle cx="80" cy="80" r="70" stroke="${progressColor}" stroke-width="12" fill="none"
                                stroke-dasharray="${circumference}" 
                                stroke-dashoffset="${offset}"
                                style="transition: stroke-dashoffset 1s ease-in-out;"/>
                        </svg>
                        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); text-align: center;">
                            <div style="font-size: 24px; font-weight: bold; color: #495057;">${percentage.toFixed(1)}%</div>
                            <div style="font-size: 14px; color: #6c757d;">${percentage === 0 ? this._labels.notstarted : this._labels.completed}</div>
                        </div>
                    </div>
                </div>
            `;

            const description = document.getElementById('progress-description');
            if (description) {
                description.textContent = `${completed} / ${total}`;
            }
        },

        _getVisibleResults: function() {
            const results = this._resultsAll || [];
            if (this._showAllResults) {
                return results;
            }
            if (results.length > this._resultsLimit) {
                // Results are sorted by date ascending; take the last N (most recent).
                return results.slice(-this._resultsLimit);
            }
            return results;
        },

        _bindToggle: function() {
            const btn = document.getElementById('results-toggle-btn');
            if (!btn) return;
            btn.addEventListener('click', () => {
                this._showAllResults = !this._showAllResults;
                // Update button text and aria state
                btn.setAttribute('aria-expanded', this._showAllResults ? 'true' : 'false');
                btn.textContent = this._showAllResults ? (this._container?.getAttribute('data-showless') || 'Ver menos') : (this._container?.getAttribute('data-showall') || 'Ver todos');
                // Re-render results area with new subset
                this.renderResultsChart(this._getVisibleResults());
            });
        },

        _updateToggleVisibility: function() {
            const btn = document.getElementById('results-toggle-btn');
            if (!btn) return;
            const hasMore = (this._resultsAll || []).length > this._resultsLimit;
            // Seed button text with the localized label already rendered by PHP ("Show all").
            // Hide if not needed.
            btn.style.display = hasMore ? '' : 'none';
            // Ensure default state shows limited set.
            this._showAllResults = false;
            btn.setAttribute('aria-expanded', 'false');
        },

        renderResultsChart: function(results) {
            const container = document.getElementById('results-chart');
            if (!container) {
                return;
            }

            if (!results || results.length === 0) {
                container.innerHTML = '<div style="text-align: center; color: #6c757d; padding: 40px; border: 2px dashed #dee2e6; border-radius: 8px; background: #f8f9fa;">📊<br><br>' + (this._labels.noresults || '') + '</div>';
                const description = document.getElementById('results-description');
                if (description) {
                    description.textContent = this._labels.noresultsdesc || '';
                }
                return;
            }

            // Create visible bar chart
            const maxGrade = Math.max(...results.map(r => r.grade));
            let chartHTML = '<div style="display: flex; justify-content: space-around; align-items: end; height: 200px; padding: 20px 0;">';
            
            results.forEach((quiz, index) => {
                const height = Math.max((quiz.grade / 100) * 160, 20); // Minimum 20px height
                const shortName = quiz.name.length > 10 ? quiz.name.substring(0, 10) + '...' : quiz.name;
                chartHTML += `
                    <div style="display: flex; flex-direction: column; align-items: center; margin: 0 5px;">
                        <div style="background: linear-gradient(to top, #007bff, #66b3ff); 
                                    width: 40px; height: ${height}px; border-radius: 4px 4px 0 0;
                                    display: flex; align-items: end; justify-content: center; padding: 5px 0;
                                    transition: transform 0.3s ease;">
                            <span style="color: white; font-size: 10px; font-weight: bold;">${quiz.grade.toFixed(1)}%</span>
                        </div>
                        <div style="font-size: 10px; color: #6c757d; margin-top: 5px; text-align: center; width: 50px; word-wrap: break-word;">
                            ${shortName}
                        </div>
                    </div>
                `;
            });
            
            chartHTML += '</div>';
            container.innerHTML = chartHTML;

            const description = document.getElementById('results-description');
            if (description) {
                const average = results.reduce((sum, r) => sum + r.grade, 0) / results.length;
                description.textContent = `${this._labels.personalavg}: ${average.toFixed(1)}% (${results.length} ${this._labels.quizzes})`;
            }
            // Update toggle button text based on current state.
            const btn = document.getElementById('results-toggle-btn');
            if (btn) {
                const showAllLabel = this._container?.getAttribute('data-showall') || btn.textContent || 'Ver todos';
                const showLessLabel = this._container?.getAttribute('data-showless') || 'Ver menos';
                btn.textContent = this._showAllResults ? showLessLabel : showAllLabel;
            }
        },

        renderComparisonChart: function(comparison) {
            const container = document.getElementById('comparison-chart');
            if (!container) {
                return;
            }

            if (!comparison.hasComparison || comparison.courseMean === 0) {
                container.innerHTML = '<div style="text-align: center; color: #6c757d; padding: 40px; border: 2px dashed #dee2e6; border-radius: 8px; background: #f8f9fa;">📈<br><br>' + (this._labels.comparisonpending || '') + '</div>';
                const description = document.getElementById('comparison-description');
                if (description) {
                    description.textContent = '';
                }
                return;
            }

            const studentMean = comparison.studentMean || 0;
            const courseMean = comparison.courseMean || 0;
            const maxValue = Math.max(studentMean, courseMean, 50); // Minimum scale of 50
            
            const studentHeight = Math.max((studentMean / maxValue) * 140, 20);
            const courseHeight = Math.max((courseMean / maxValue) * 140, 20);

            container.innerHTML = `
                <div style="display: flex; justify-content: space-around; align-items: end; height: 180px; padding: 20px;">
                    <div style="display: flex; flex-direction: column; align-items: center;">
                        <div style="background: linear-gradient(to top, #007bff, #66b3ff); 
                                    width: 60px; height: ${studentHeight}px; border-radius: 8px 8px 0 0;
                                    display: flex; align-items: end; justify-content: center; padding: 8px 0;">
                            <span style="color: white; font-size: 12px; font-weight: bold;">${studentMean.toFixed(1)}%</span>
                        </div>
                        <div style="font-size: 12px; color: #6c757d; margin-top: 8px; text-align: center;">${this._labels.myaverage}</div>
                    </div>
                    <div style="display: flex; flex-direction: column; align-items: center;">
                        <div style="background: linear-gradient(to top, #6c757d, #adb5bd); 
                                    width: 60px; height: ${courseHeight}px; border-radius: 8px 8px 0 0;
                                    display: flex; align-items: end; justify-content: center; padding: 8px 0;">
                            <span style="color: white; font-size: 12px; font-weight: bold;">${courseMean.toFixed(1)}%</span>
                        </div>
                        <div style="font-size: 12px; color: #6c757d; margin-top: 8px; text-align: center;">${this._labels.courseaverage}</div>
                    </div>
                </div>
            `;

            const description = document.getElementById('comparison-description');
            if (description) {
                let text = `${this._labels.courseavgprefix} ${courseMean.toFixed(1)}%`;
                if (comparison.hasComparison && studentMean > 0) {
                    text += ` • ${this._labels.yourpercentileprefix} ${comparison.percentile}%`;
                }
                description.textContent = text;
            }
        },

        showLoading: function() {
            const loading = document.getElementById('analytics-loading');
            const error = document.getElementById('analytics-error');
            const noquizzes = document.getElementById('analytics-noquizzes');
            const content = document.getElementById('analytics-content');
            
            if (loading) loading.classList.remove('d-none');
            if (error) error.classList.add('d-none');
            if (noquizzes) noquizzes.classList.add('d-none');
            if (content) content.classList.add('d-none');
        },

        hideLoading: function() {
            const loading = document.getElementById('analytics-loading');
            if (loading) loading.classList.add('d-none');
        },

        showError: function() {
            this.hideLoading();
            const error = document.getElementById('analytics-error');
            if (error) error.classList.remove('d-none');
        },

        showNoQuizzes: function() {
            this.hideLoading();
            const noquizzes = document.getElementById('analytics-noquizzes');
            if (noquizzes) noquizzes.classList.remove('d-none');
        },

        showContent: function() {
            this.hideLoading();
            const content = document.getElementById('analytics-content');
            if (content) {
                content.classList.remove('d-none');
            } else {
                // Element not found.
            }
        }
    };

    return Charts;
});
