// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * JavaScript for Progress Analytics Block charts
 *
 * @package
 * @copyright  2024 Progress Analytics Block
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

define(['core/ajax', 'core/chartjs'], function(Ajax, Chart) {
    'use strict';

    var Charts = {
        progressChart: null,
        resultsChart: null,
        comparisonChart: null,
        /**
         * Initialize the charts
         * @param {int} courseid Course ID
         */
        init: function(courseid) {
            // Chart.js provided by Moodle core (core/chartjs).
            this.loadData(courseid);
        },

        // No external loader. Chart is provided via AMD dependency.

        /**
         * Load data from the server
         * @param {int} courseid Course ID
         */
        loadData: function(courseid) {
            Ajax.call([{
                methodname: 'block_progressanalytics_get_course_quiz_metrics',
                args: { courseid: courseid }
            }])[0].then(data => {
                this.hideLoading();

                if (data.progress.total === 0) {
                    this.showNoQuizzes();
                    return;
                }

                this.showContent();
                this.renderCharts(data);
            }).catch(() => {
                // Error loading quiz metrics
                this.showError();
            });
        },

        /**
         * Render all charts
         * @param {Object} data Quiz metrics data
         */
        renderCharts: function(data) {
            this.renderProgressChart(data.progress);
            this.renderResultsChart(data.results);
            this.renderComparisonChart(data.comparison);
        },

        /**
         * Render progress doughnut chart
         * @param {Object} progress Progress data
         */
        renderProgressChart: function(progress) {
            const ctx = document.getElementById('progress-chart');
            if (!ctx) {
                return;
            }

            const completed = progress.completed;
            const remaining = progress.total - completed;

            this.progressChart = new Chart(ctx, {
                type: 'doughnut',
                data: {
                    datasets: [{
                        data: [completed, remaining],
                        backgroundColor: [
                            '#28a745',
                            '#e9ecef'
                        ],
                        borderColor: [
                            '#1e7e34',
                            '#dee2e6'
                        ],
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    if (context.dataIndex === 0) {
                                        return `Completados: ${completed}`;
                                    } else {
                                        return `Pendientes: ${remaining}`;
                                    }
                                }
                            }
                        }
                    },
                    elements: {
                        center: {
                            text: progress.percentage.toFixed(1) + '%',
                            color: '#495057',
                            fontStyle: 'bold',
                            fontSize: 24
                        }
                    }
                },
                plugins: [{
                    beforeDraw: function(chart) {
                        if (chart.config.options.elements.center) {
                            const ctx = chart.ctx;
                            const centerConfig = chart.config.options.elements.center;
                            const fontStyle = centerConfig.fontStyle || 'Arial';
                            const fontSize = centerConfig.fontSize || 24;

                            ctx.restore();
                            ctx.font = fontSize + 'px ' + fontStyle;
                            ctx.textBaseline = 'middle';
                            ctx.fillStyle = centerConfig.color || '#000';

                            const centerX = ((chart.chartArea.left + chart.chartArea.right) / 2);
                            const centerY = ((chart.chartArea.top + chart.chartArea.bottom) / 2);
                            ctx.textAlign = 'center';
                            ctx.fillText(centerConfig.text, centerX, centerY);
                            ctx.save();
                        }
                    }
                }]
            });

            // Update description
            const description = document.getElementById('progress-description');
            if (description) {
                description.textContent = `${completed} de ${progress.total} cuestionarios completados`;
            }
        },

        /**
         * Render results line chart
         * @param {Array} results Results data
         */
        renderResultsChart: function(results) {
            const ctx = document.getElementById('results-chart');
            if (!ctx) {
                return;
            }

            if (results.length === 0) {
                ctx.parentElement.innerHTML = '<div class="text-center text-muted p-3">Sin resultados disponibles</div>';
                return;
            }

            const labels = results.map(r => r.name.length > 15 ? r.name.substring(0, 15) + '...' : r.name);
            const grades = results.map(r => r.grade);

            this.resultsChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: labels,
                    datasets: [{
                        label: 'Calificación',
                        data: grades,
                        borderColor: '#007bff',
                        backgroundColor: 'rgba(0, 123, 255, 0.1)',
                        borderWidth: 2,
                        fill: true,
                        tension: 0.1,
                        pointBackgroundColor: '#007bff',
                        pointBorderColor: '#fff',
                        pointBorderWidth: 2,
                        pointRadius: 5
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            callbacks: {
                                title: function(context) {
                                    return results[context[0].dataIndex].name;
                                },
                                label: function(context) {
                                    return `Calificación: ${context.parsed.y.toFixed(1)}%`;
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            max: 100,
                            ticks: {
                                callback: function(value) {
                                    return value + '%';
                                }
                            }
                        },
                        x: {
                            ticks: {
                                maxRotation: 45
                            }
                        }
                    }
                }
            });

            // Update description
            const description = document.getElementById('results-description');
            if (description && results.length > 0) {
                const average = (grades.reduce((a, b) => a + b, 0) / grades.length).toFixed(1);
                description.textContent = `Promedio personal: ${average}% (${results.length} cuestionarios)`;
            }
        },

        /**
         * Render comparison bar chart
         * @param {Object} comparison Comparison data
         */
        renderComparisonChart: function(comparison) {
            const ctx = document.getElementById('comparison-chart');
            if (!ctx) {
                return;
            }

            if (!comparison.hasComparison || comparison.courseMean === 0) {
                ctx.parentElement.innerHTML = '<div class="text-center text-muted p-3">' +
                    'Los datos de comparación aparecerán cuando más estudiantes completen los cuestionarios</div>';
                return;
            }

            const data = {
                labels: ['Mi promedio', 'Promedio del curso'],
                datasets: [{
                    data: [comparison.studentMean, comparison.courseMean],
                    backgroundColor: [
                        '#007bff',
                        '#6c757d'
                    ],
                    borderColor: [
                        '#0056b3',
                        '#5a6268'
                    ],
                    borderWidth: 1
                }]
            };

            this.comparisonChart = new Chart(ctx, {
                type: 'bar',
                data: data,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return `${context.label}: ${context.parsed.y.toFixed(1)}%`;
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            max: 100,
                            ticks: {
                                callback: function(value) {
                                    return value + '%';
                                }
                            }
                        }
                    }
                }
            });

            // Update description
            const description = document.getElementById('comparison-description');
            if (description) {
                let text = `Promedio del curso: ${comparison.courseMean.toFixed(1)}%`;
                if (comparison.hasComparison && comparison.studentMean > 0) {
                    text += ` • Tu percentil: ${comparison.percentile}%`;
                }
                description.textContent = text;
            }
        },

        /**
         * Show loading state
         */
        showLoading: function() {
            document.getElementById('analytics-loading').classList.remove('d-none');
            document.getElementById('analytics-error').classList.add('d-none');
            document.getElementById('analytics-noquizzes').classList.add('d-none');
            document.getElementById('analytics-content').classList.add('d-none');
        },

        /**
         * Hide loading state
         */
        hideLoading: function() {
            document.getElementById('analytics-loading').classList.add('d-none');
        },

        /**
         * Show error state
         */
        showError: function() {
            this.hideLoading();
            document.getElementById('analytics-error').classList.remove('d-none');
        },

        /**
         * Show no quizzes state
         */
        showNoQuizzes: function() {
            this.hideLoading();
            document.getElementById('analytics-noquizzes').classList.remove('d-none');
        },

        /**
         * Show main content
         */
        showContent: function() {
            this.hideLoading();
            document.getElementById('analytics-content').classList.remove('d-none');
        },

        /**
         * Destroy existing charts
         */
        destroy: function() {
            if (this.progressChart) {
                this.progressChart.destroy();
                this.progressChart = null;
            }
            if (this.resultsChart) {
                this.resultsChart.destroy();
                this.resultsChart = null;
            }
            if (this.comparisonChart) {
                this.comparisonChart.destroy();
                this.comparisonChart = null;
            }
        }
    };

    return Charts;
});
