<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Implementation of the Amanote filter plugin.
 *
 * @package     filter_amanote
 * @copyright   2020 Amaplex Software
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace filter_amanote;

defined('MOODLE_INTERNAL') || die;

require_once($CFG->libdir . '/externallib.php');
require_once(__DIR__ . '/../helpers/filehelper.php');
require_once(__DIR__ . '/../models/annotatable.php');

// Define an alias for compatibility with all versions.
if (class_exists('\core_filters\text_filter')) {
    class_alias('\core_filters\text_filter', 'generic_base_text_filter');
} else {
    class_alias('\moodle_text_filter', 'generic_base_text_filter');
}

/**
 * Filter for processing file links for Amanote enhancements.
 *
 * @copyright  2020 Amaplex Software
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class text_filter extends \generic_base_text_filter {
    /**
     * Set up the filter and insert JS scripts to page using amd.
     *
     * @param moodle_page $page The current page.
     * @param context $context The current context.
     */
    public function setup($page, $context) {
        try {
            global $PAGE, $COURSE, $USER;
            static $scriptinserted = false;
            static $folderwarninginserted = false;

            // Check if the user is logged in.
            if (empty($USER->id) || isguestuser()) {
                return;
            }

            // Check if the filter is enabled for the current course.
            $enabledfilters = filter_get_active_in_context(\context_course::instance($COURSE->id));
            if (!isset($enabledfilters['amanote'])) {
                return;
            }

            // If folder edit page, add folder edit warning.
            if ($PAGE->pagetype === 'mod-folder-edit' && !$folderwarninginserted) {
                \core\notification::add(get_string('foldereditwarning', 'filter_amanote'), \core\notification::WARNING);
                $folderwarninginserted = true;
                return;
            }

            // Insert script if needed.

            if ($scriptinserted ||
                (stripos($PAGE->pagetype, 'course-view') !== 0 && $PAGE->pagetype !== 'mod-folder-view') ||
                (strpos($PAGE->url, 'user/') !== false)) {
                return;
            }

            $this->insert_script($page);
            $scriptinserted = true;
        } catch (Exception $e) {
            debugging('An error occurred: ' . $e->getMessage(), DEBUG_DEVELOPER);
        }
    }

    /**
     * Insert the js script.
     *
     * @param moodle_page $page The page in which the script should be inserted.
     */
    private function insert_script($page) {
        global $CFG;

        $params = $this->generate_json_params();
        $userparams = $this->generate_json_user_params();

        // Insert user params using AMD.
        $page->requires->js_call_amd('filter_amanote/main', 'init', [$userparams]);

        // Insert params in footer (AMD can't handle long params).
        $script = <<<EOF
            <script>
                var amanote_params = $params;
            </script>
EOF;

        if (!isset($CFG->additionalhtmlfooter)) {
            $CFG->additionalhtmlfooter = '';
        }

        $CFG->additionalhtmlfooter .= $script;
    }

    /**
     * Return the Moodle params needed by the js script in JSON format.
     *
     * @return string The params in JSON string format.
     */
    private function generate_json_params() {
        global $USER, $PAGE, $COURSE, $CFG, $OUTPUT;

        try {
            $config = get_config('filter_amanote');
            $siteurl = $this->get_site_url();
            $language = substr($USER->lang, 0, 2);
            $usercontext = \context_user::instance($USER->id);
            $privatefilepath = '/' . $usercontext->id . '/user/private/Amanote/';
            $contentkind = $config->anonymous ? 'document' : null;
            $annotatables = get_annotatables($COURSE->id, $contentkind);
            $moodleversion = preg_replace('/(\d+\.\d+(\.\d+)?) .*$/', '$1', $CFG->release);
            $notelogo = $this->get_note_taking_logo_name($config->openingmode);
            $logourl = $OUTPUT->image_url($notelogo , 'filter_amanote')->__toString();
            $annotatedlogourl = $OUTPUT->image_url($notelogo . '-annotated', 'filter_amanote')->__toString();
            $savednotes = null;

            if ($config->saveinprivate) {
                $savednotes = get_user_notes_for_course($USER->id, $COURSE->id);
            }

            return json_encode([
                'courseId' => $COURSE->id,
                'pageType' => $PAGE->pagetype,
                'siteURL' => $siteurl,
                'privateFilePath' => $privatefilepath,
                'savedNotes' => $savednotes,
                'annotatables' => $annotatables,
                'moodle' => [
                    'version' => $moodleversion,
                ],
                'plugin' => [
                    'version' => $config->version,
                    'saveInProvider' => ($config->saveinprivate == '1'),
                    'preventDownload' => ($config->preventdownload == '1'),
                    'openingMode' => $config->openingmode,
                    'target' => $config->target,
                    'key' => $config->key,
                    'anonymous' => ($config->anonymous == '1'),
                    'worksheet' => ($config->worksheet == '1'),
                    'logo' => $logourl,
                    'annotatedLogo' => $annotatedlogourl,
                ],
                'language' => $language,
                'strings' => [
                    'aiAssistant' => get_string('aiassistant', 'filter_amanote'),
                    'modalDescription' => get_string('modaldescription', 'filter_amanote'),
                    'annotateResource' => get_string('annotateresource', 'filter_amanote'),
                    'viewResource' => get_string('viewresource', 'filter_amanote'),
                    'downloadNotes' => get_string('downloadnotes', 'filter_amanote'),
                    'openAnalytics' => get_string('openanalytics', 'filter_amanote'),
                    'openPodcastCreator' => get_string('openpodcast', 'filter_amanote'),
                    'openStudentsWorks' => get_string('openstudentsworks', 'filter_amanote'),
                    'teacher' => get_string('teacher', 'filter_amanote'),
                    'deletefilewarning' => get_string('deletefilewarning', 'filter_amanote'),
                    'seeguide' => get_string('seeguide', 'filter_amanote'),
                    'stopmodal' => get_string('stopmodal', 'filter_amanote'),
                ],
            ]);
        } catch (Exception $e) {
            debugging('An error occurred: ' . $e->getMessage() . $e->getTraceAsString(), DEBUG_DEVELOPER);
        }
    }

    /**
     * Return the note-taking logo name for the given opening mode.
     *
     * @param integer $openingmode The opening mode.
     *
     * @return string The note-taking logo name.
     */
    private function get_note_taking_logo_name($openingmode) {
        $openingmodeicons = [
            '0' => 'note-icon',
            '1' => 'amanote-logo',
            '2' => 'note-icon',
            '3' => get_string('takenotesicon', 'filter_amanote'),
        ];

        return $openingmodeicons[$openingmode];
    }

    /**
     * Return the user params needed by the js script in JSON format.
     *
     * @return string The user params in JSON string format.
     */
    private function generate_json_user_params() {
        global $USER, $COURSE;

        $coursecontext = \context_course::instance($COURSE->id, MUST_EXIST);
        $isteacher = has_capability('moodle/course:update', $coursecontext, $USER->id, false);
        $isaicreator = $this->is_user_ai_creator();

        $token = $this->get_current_user_token();

        return json_encode([
            'id' => $USER->id,
            'token' => [
                'value' => $token->token,
                'expiration' => $token->validuntil,
            ],
            'isTeacher' => $isteacher,
            'isAICreator' => $isaicreator,
        ]);
    }

    /**
     * Return the token of the current user.
     *
     * @return token The user token.
     */
    private function get_current_user_token() {
        global $DB;

        $params = ['shortname' => MOODLE_OFFICIAL_MOBILE_SERVICE, 'enabled' => 1];
        $service = $DB->get_record('external_services', $params);

        if (empty($service)) {
            return null;
        }

        return external_generate_token_for_current_user($service);
    }

    /**
     * Check if the user is an AI creator.
     *
     * @return bool True if the user is an AI creator, false otherwise.
     */
    private function is_user_ai_creator() {
        global $USER, $COURSE;

        if (!get_config('filter_amanote', 'aiassistant')) {
            return false;
        }

        try {
            $coursecontext = \context_course::instance($COURSE->id, MUST_EXIST);

            if (has_capability('moodle/course:update', $coursecontext, $USER->id, false) &&
                has_capability('filter/amanote:createaiassistant', $coursecontext)) {
                return true;
            }

            // Check if the user is an admin.
            $systemcontext = \context_system::instance();

            if (has_capability('moodle/site:configview', $systemcontext, $USER->id)) {
                return true;
            }
        } catch (Exception $e) {
            debugging('An error occurred: ' . $e->getMessage() . $e->getTraceAsString(), DEBUG_DEVELOPER);
        }

        return false;
    }

    /**
     * Return the site url.
     *
     * @return string The site url (example: https://demo.moodle.com).
     */
    private function get_site_url() {
        global $CFG;

        // Check if protocol is HTTP.
        if (preg_match('/^http:\/\//', $CFG->wwwroot)) {

            $securewwwroot = preg_replace('/^http:\/\//', 'https://', $CFG->wwwroot, 1);

            // Check if Moodle site exists in HTTPS.
            $curl = curl_init($securewwwroot);
            if (curl_exec($curl)) {
                return $securewwwroot;
            }
            curl_close($curl);
        }

        return $CFG->wwwroot;
    }

    /**
     * Filters the given HTML text.
     *
     * @param string $text HTML to be processed.
     * @param array $options The options.
     *
     * @return string String containing processed HTML.
     */
    public function filter($text, array $options = []) {
        return $text;
    }
}
