<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Amanote external api.
 *
 * @package    filter_amanote
 * @copyright  2020 Amaplex Software
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die;

require_once($CFG->libdir . "/externallib.php");
require_once(__DIR__ . "/helpers/notificationhelper.php");
require_once(__DIR__ . "/helpers/filehelper.php");

/**
 * Amanote external functions.
 *
 * @copyright  2020 Amaplex Software
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class filter_amanote_external extends external_api {

    /**
     * Returns description of method parameters.
     *
     * @return external_function_parameters
     */
    public static function can_update_course_parameters() {
        return new external_function_parameters(
                [
                    'courseid' => new external_value(PARAM_INT, 'id of the course'),
                ]
        );
    }

    /**
     * Determine if the user can update a specific Amanote resource.
     *
     * @param integer $courseid File id.
     *
     * @return boolean True if current user can update the amanote resource
     */
    public static function can_update_course($courseid) {
        global $DB;
        // Parameters validation.
        $params = self::validate_parameters(self::can_update_course_parameters(),
                ['courseid' => $courseid]);

        if (is_nan($courseid) || $courseid < 0) {
            throw new invalid_parameter_exception('Invalid course id');
        }

        $context = context_course::instance($courseid, MUST_EXIST);

        if ($context && has_capability('moodle/course:update', $context)) {
            return true;
        }

        return false;
    }

    /**
     * Returns description of method result value.
     *
     * @return external_description
     */
    public static function can_update_course_returns() {
        return new external_value(PARAM_BOOL, 'true if the current user can update the course');
    }

    /**
     * Returns description of method parameters.
     *
     * @return external_function_parameters
     */
    public static function notify_parameters() {
        return new external_function_parameters(
                [
                    'kind' => new external_value(PARAM_TEXT, 'the notification kind (e.g. "submission", "feedback", "question")'),
                    'annotatableid' => new external_value(PARAM_TEXT, 'id of the annotatable'),
                ]
        );
    }

    /**
     * Send a notification related to a given annotatable.
     *
     * @param string $kind The kind of notification.
     * @param integer $annotatableid The annotatable id.
     *
     * @return array  An array containing the ids of sent notifications.
     */
    public static function notify($kind, $annotatableid) {
        global $USER;

        return send_annotatable_notifications($kind, $annotatableid);
    }

    /**
     * Returns description of method result value.
     *
     * @return external_description
     */
    public static function notify_returns() {
        return new external_multiple_structure(
            new external_value(PARAM_INT, 'the ids of the notifications sent.')
        );
    }

    /**
     * Returns description of method parameters.
     *
     * @return external_function_parameters
     */
    public static function get_amanote_url_parameters() {
        return new external_function_parameters([
            'annotatableid' => new external_value(PARAM_TEXT, 'id of the annotatable'),
        ]);
    }

    /**
     * Get the Amanote app URL for a given annotatable.
     *
     * @param string $annotatableid The id of the annotatable resource.
     * @return string The Amanote app URL.
     */
    public static function get_amanote_url($annotatableid) {
        return generate_amanote_url($annotatableid, null, null, 'note-taking', false);
    }

    /**
     * Returns description of method result value.
     *
     * @return external_description
     */
    public static function get_amanote_url_returns() {
        return new external_value(PARAM_RAW, 'The Amanote app URL for the given annotatable');
    }

    /**
     * Returns description of method parameters.
     *
     * @return external_function_parameters
     */
    public static function get_amanote_annotatables_parameters() {
        return new external_function_parameters([
            'courseid' => new external_value(PARAM_INT, 'id of the course'),
            'contentkind' => new external_value(PARAM_TEXT, 'specific content type', VALUE_DEFAULT, null),
        ]);
    }

    /**
     * Get the list of annotatables for a given course.
     *
     * @param int $courseid The id of the course.
     * @param string|null $contentkind The specific content type.
     * @return array The list of annotatables.
     */
    public static function get_amanote_annotatables($courseid, $contentkind = null) {
        global $DB;

        // Validate parameters.
        self::validate_parameters(self::get_amanote_annotatables_parameters(),
                ['courseid' => $courseid, 'contentkind' => $contentkind]);

        // Check the user is enrolled in the course.
        $context = context_course::instance($courseid);
        self::validate_context($context);
        require_capability('moodle/course:view', $context);

        // Get the annotatables.
        return get_annotatables($courseid, $contentkind, true);
    }

    /**
     * Returns description of method result value.
     *
     * @return external_description
     */
    public static function get_amanote_annotatables_returns() {
        return new external_multiple_structure(
            new external_single_structure([
                'id' => new external_value(PARAM_RAW, 'The annotatable\'s ID'),
                'legacyid' => new external_value(PARAM_RAW, 'The annotatable\'s legacy ID'),
                'cmid' => new external_value(PARAM_RAW, 'Course Module ID'),
                'mimetype' => new external_value(PARAM_RAW, 'MIME type of the annotatable'),
                'internal' => new external_value(PARAM_BOOL, 'Whether the content is internal to Moodle'),
                'url' => new external_value(PARAM_RAW, 'URL to access the content'),
                'kind' => new external_value(PARAM_RAW, 'The annotatable\'s kind'),
                'component' => new external_value(PARAM_RAW, 'The annotatable\'s component'),
                'name' => new external_value(PARAM_RAW, 'The annotatable\'s name'),
                'visible' => new external_value(PARAM_BOOL, 'Whether the annotatable is visible to students'),
            ])
        );
    }
}
