<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace repository_mediasite;

/**
 * Tests for util class
 *
 * @package    repository_mediasite
 * @category   test
 * @copyright  2026 Andrew Rowatt <A.J.Rowatt@massey.ac.nz>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @covers \repository_mediasite\util
 */
final class util_test extends \advanced_testcase {
    /**
     * Test duration formatting through reflection
     *
     * @param int $milliseconds Input duration in milliseconds
     * @param string $expected Expected formatted duration string
     *
     * @covers \repository_mediasite\util::format_duration
     * @dataProvider duration_formatting_provider
     */
    public function test_duration_formatting($milliseconds, $expected): void {
        // Use reflection to test the private method.
        $reflection = new \ReflectionClass(util::class);
        $method = $reflection->getMethod('format_duration');
        $method->setAccessible(true);

        $result = $method->invoke(null, $milliseconds);
        $this->assertEquals($expected, $result);
    }

    /**
     * Data provider for duration formatting tests
     *
     * @return array Test cases with milliseconds and expected formatted output
     */
    public static function duration_formatting_provider(): array {
        return [
            'Zero duration' => [0, ''],
            'Negative duration' => [-1000, ''],
            '30 seconds' => [30933, '30 Seconds'],
            '4 minutes 35 seconds' => [275000, '4 Minutes 35 Seconds'],
            '1 hour 8 minutes 55 seconds' => [4135000, '1 Hour 8 Minutes 55 Seconds'],
            '1 second' => [1000, '1 Second'],
            '1 minute' => [60000, '1 Minute'],
            '1 hour' => [3600000, '1 Hour'],
            '2 hours 30 minutes 45 seconds' => [9045000, '2 Hours 30 Minutes 45 Seconds'],
            'Exact hour' => [7200000, '2 Hours'],
            'Exact minutes' => [300000, '5 Minutes'],
        ];
    }

    /**
     * Test that manage URL is included when set
     *
     * @covers \repository_mediasite\util::get_mediasite_presentations
     */
    public function test_manage_url_included_when_set(): void {
        $this->resetAfterTest(true);

        // Set the manageurl config.
        set_config('manageurl', 'example.com/mediasite/mymediasite', 'mediasite');

        // Mock the get_presentations call to avoid making actual API requests.
        // Since get_presentations is private, we test the public method that calls it.
        // This test verifies the structure, not the actual API integration.
        $reflection = new \ReflectionClass(util::class);
        $method = $reflection->getMethod('get_mediasite_presentations');

        // We can't easily test this without mocking the API call, so this test is a placeholder.
        // In a real scenario, you'd want to mock the API response.
        $this->assertTrue(true, 'Placeholder test for manage URL inclusion');
    }

    /**
     * Test that manage URL is not included when not set
     *
     * @covers \repository_mediasite\util::get_mediasite_presentations
     */
    public function test_manage_url_not_included_when_not_set(): void {
        $this->resetAfterTest(true);

        // Clear the manageurl config.
        set_config('manageurl', '', 'mediasite');

        // This test is a placeholder for the actual integration test.
        $this->assertTrue(true, 'Placeholder test for manage URL exclusion');
    }
}
