/**
 * eFlow Ai Tutor chat module.
 *
 * @module     local_eflow_aitutor/chat
 * @copyright  2025 Nawaz sharif<shasharif02@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
define(['jquery', 'core/ajax', 'core/notification', 'core/str', './poller'],
function($, Ajax, Notification, Str, ApiPoller) {

    /**
     * Initialize the chat interface
     *
     * @param {Object} params - Parameters including courseId and userId
     */
    var init = function(params) {
        console.log('eFlow Ai Tutor chat module initialized with params:', params);

        // Make sure jQuery is available
        if (!$ || typeof $ !== 'function') {
            console.error('jQuery is not available');
            return;
        }

        var poller = null;
        var selectedImage = null; // Variable to store the selected image file
        console.log('Initial selectedImage:', selectedImage);

        // Wait for DOM to be fully loaded
        $(document).ready(function() {

            if (window.location.href.includes('course/view.php?id=')) {
                var id = 0;
            } else {
                var id = new URLSearchParams(window.location.search).get('id') ?? 0;
            }
            var element = $('.floating-chat');
            console.log('Found floating-chat elements:', element.length);

            if (element.length === 0) {
                console.error('No floating-chat element found in the DOM');
                return;
            }

            // Set user profile image if available
            if (params.userProfileImageUrl) {
                $('<style>')
                    .prop('type', 'text/css')
                    .html('.floating-chat .messages li.self:before { background-image: url("' + params.userProfileImageUrl + '") !important; background-size: cover !important; }')
                    .appendTo('head');
            } else if (params.defaultImageUrl) {
                $('<style>')
                    .prop('type', 'text/css')
                    .html('.floating-chat .messages li.self:before { background-image: url("' + params.defaultImageUrl + '") !important; background-size: cover !important; }')
                    .appendTo('head');
            }

            // Check if iframe mode is enabled - simplified detection
            var isIframeMode = element.find('.iframe-container').length > 0;
            console.log('Iframe mode is ' + (isIframeMode ? 'enabled' : 'disabled'));
            console.log('Iframe container elements found:', element.find('.iframe-container').length);
            console.log('Iframe elements found:', element.find('iframe').length);

            // Determine if float option is enabled - we'll set it to 'with_option' by default,
            // and the poller will fall back to 'without_option' if needed
            var floatOption = 'with_option';
            if (params.floatOption !== undefined) {
                floatOption = params.floatOption;
            }
            console.log('Float option is set to:', floatOption);

            // Create the API poller instance
            poller = new ApiPoller({
                courseId: params.courseId,
                userId: params.userId,
                interval: 30000, // 30 seconds
                onNewResponse: handleNewResponses,
                floatOption: floatOption
            });

            // Set up the close button handler immediately
            element.find('.header button').on('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                closeChat();
            });

            var myStorage = localStorage;

            if (!myStorage.getItem('chatID')) {
                myStorage.setItem('chatID', createUUID());
            }

            // Add the enter class after a delay
            setTimeout(function() {
                console.log('Adding enter class to floating chat element');
                element.addClass('enter');

                // Make the chat button work properly after initialization
                element.on('click', function(e) {
                    // If already expanded, don't respond to clicks on the chat body
                    if (element.hasClass('expand') && !$(e.target).is('.floating-chat, .custom-logo, i.fa-comments')) {
                        return;
                    }

                    // If not expanded, open the chat
                    if (!element.hasClass('expand')) {
                        openChat();
                    }
                });
            }, 1000);

            // Set up image upload functionality
            element.find('.file-upload').on('change', handleImageUpload);
            element.find('.remove-image').on('click', removeSelectedImage);

            // Add click handler for upload button to trigger file input
            element.find('.upload-btn').on('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                console.log('Upload button clicked');
                element.find('.file-upload').click();
            });

            /**
             * Check if the send button should be enabled
             * Enable if: input has text OR input has focus OR an image is selected
             */
            function updateSendButtonState() {
                var textBox = element.find('.text-box');
                var sendButton = element.find('#sendMessage');

                // Check if the text box has text content, is focused, or if an image is selected
                // First clean the HTML content to handle empty divs, br tags, etc.
                var cleanContent = textBox.html().replace(/<div><br><\/div>|<div><\/div>|<br>|&nbsp;/gi, '').trim();
                var hasText = cleanContent.length > 0;
                var hasFocus = textBox.is(':focus');
                var hasImage = selectedImage !== null;

                console.log('Updating send button state - hasText:', hasText, 'hasFocus:', hasFocus, 'hasImage:', hasImage, 'cleanContent:', cleanContent);

                if (hasText || hasFocus || hasImage) {
                    sendButton.prop('disabled', false).css('background-color', '#308bba');
                } else {
                    sendButton.prop('disabled', true).css('background-color', '#9dbacd');
                }
            }

            /**
             * Handle image file selection
             * @param {Event} e - The change event
             */
            function handleImageUpload(e) {
                console.log('File upload triggered', e.target.files);
                var file = e.target.files[0];
                if (!file) {
                    console.log('No file selected');
                    return;
                }

                // Check if file is an image
                if (!file.type.match('image.*')) {
                    console.error('Invalid file type:', file.type);
                    Notification.alert(strings.error, strings.selectImageError);
                    return;
                }

                // Check file size (limit to 5MB)
                if (file.size > 5 * 1024 * 1024) {
                    console.error('File too large:', file.size);
                    Notification.alert(strings.error, strings.imageSizeError);
                    return;
                }

                console.log('Valid image selected:', file.name, file.type, file.size);

                // Store the selected file - create a safe copy
                var safeCopy = createSafeFileCopy(file);
                if (safeCopy) {
                    selectedImage = safeCopy;
                    console.log('Image stored in selectedImage variable:', selectedImage);
                } else {
                    console.error('Failed to create safe copy, using original file');
                    selectedImage = file;
                    console.log('Using original file object:', selectedImage);
                }

                // Show image preview
                var reader = new FileReader();
                reader.onload = function(e) {
                    try {
                        console.log('Image preview loaded');
                        var preview = element.find('.image-preview');
                        preview.find('img').attr('src', e.target.result);
                        preview.css('display', 'block');

                        // Update send button state after image is loaded
                        updateSendButtonState();
                    } catch (err) {
                        console.error('Error showing image preview:', err);
                    }
                };

                reader.onerror = function(err) {
                    console.error('Error reading image file for preview:', err);
                };

                try {
                    reader.readAsDataURL(file);
                } catch (err) {
                    console.error('Error calling readAsDataURL for preview:', err);
                }
            }

            /**
             * Remove the selected image
             */
            function removeSelectedImage() {
                console.log('Removing selected image');
                selectedImage = null;
                console.log('selectedImage after removal:', selectedImage);
                element.find('.image-preview').css('display', 'none');
                element.find('.file-upload').val('');

                // Update send button state after removing image
                updateSendButtonState();
            }

            /**
             * Open the chat panel - works for both modes
             */
            function openChat() {
                console.log('Opening chat panel');

                // Hide the icon
                element.find('i.fa-comments, img.custom-logo').hide();

                // Expand the chat
                element.addClass('expand');

                // Show the chat interface
                element.find('.chat').addClass('enter');

                // Regular chat mode specific setup
                if (!isIframeMode) {
                    var messages = element.find('.messages');
                    var textInput = element.find('.text-box');

                    // Make sure loader is hidden initially (in case it was left active)
                    hideMainLoader();

                    // Then explicitly show the loader - with a slight delay to ensure CSS transitions work
                    setTimeout(function() {
                        showMainLoader();
                    }, 50);

                    // Enable and focus text input
                    textInput.prop("disabled", false).focus();

                    // Add message handlers
                    element.find('#sendMessage').click(sendNewMessage);
                    textInput.keydown(onMetaAndEnter);

                    // Set up event listeners for send button state
                    textInput.on('focus', updateSendButtonState);
                    textInput.on('blur', updateSendButtonState);
                    textInput.on('input', updateSendButtonState);

                    // Initially disable send button
                    updateSendButtonState();

                    // Scroll to bottom
                    messages.scrollTop(messages.prop("scrollHeight"));

                    // Start polling for new responses
                    poller.start();
                }
            }

            /**
             * Close the chat panel
             */
            function closeChat() {
                console.log('Closing chat panel');

                // Hide chat interface
                element.find('.chat').removeClass('enter').hide();

                // Show icon
                element.find('i.fa-comments, img.custom-logo').show();

                // Collapse chat container
                element.removeClass('expand');

                // Remove message handlers if in regular mode
                if (!isIframeMode) {
                    element.find('#sendMessage').off('click', sendNewMessage);
                    element.find('.text-box').off('keydown', onMetaAndEnter).prop("disabled", true).blur();

                    // Stop polling when chat is closed
                    poller.stop();
                }

                // Restore chat interface
                setTimeout(function() {
                    element.find('.chat').show();
                }, 500);
            }

            /**
             * Generate appropriate HTML for media based on file type
             * @param {string} mediaUrl - The media URL
             * @returns {string} HTML for displaying the media
             */
            function getMediaHtml(mediaUrl) {
                var videoExtensions = ['.mp4', '.webm', '.mov', '.ogg'];
                var isVideo = videoExtensions.some(function(ext) {
                    return mediaUrl.toLowerCase().endsWith(ext);
                });

                if (isVideo) {
                    return '<div class="media-container"><video src="' + mediaUrl + '" controls class="chat-media"></video></div>';
                } else {
                    return '<div class="media-container"><img src="' + mediaUrl + '" alt="Media" class="chat-media" onerror="this.style.display=\'none\'"></div>';
                }
            }

            /**
             * Handle new responses from the API poller
             *
             * @param {Array} responses Array of new responses
             */
            function handleNewResponses(responses) {
                // Hide the loader since we have a response
                hideMainLoader();
                hideMessageLoader();

                // Re-enable send button when new responses arrive
                element.find('#sendMessage').prop('disabled', false);

                if (!responses || !responses.length) {
                    return;
                }

                var messagesContainer = element.find('.messages');

                // Clear existing welcome message if this is the first load
                if (messagesContainer.find('li').length === 1 &&
                    messagesContainer.find('li').first().text().indexOf("Hello! I'm your AI tutor assistant") === 0) {
                    messagesContainer.empty();
                }

                // Sort responses by timestamp to ensure correct chronological order
                responses.sort(function(a, b) {
                    return a.id - b.id;
                });

                console.log('Processing responses:', responses);

                // Add each response to the chat
                responses.forEach(function(response) {
                    // Skip if message already exists in DOM (external API ignores last_id filter)
                    if (messagesContainer.find('[id="' + response.id + '"]').length > 0) {
                        return;
                    }

                    // For user messages, update existing message with ID instead of appending duplicate
                    if (response.is_user) {
                        var existingUserMsg = messagesContainer.find('li.self:not([id])').last();
                        if (existingUserMsg.length > 0) {
                            // Check if message content matches
                            var existingContent = existingUserMsg.html().replace(/<[^>]*>/g, '').trim();
                            var responseContent = (response.message || '').replace(/<[^>]*>/g, '').trim();
                            if (existingContent === responseContent) {
                                existingUserMsg.attr('id', response.id);
                                return;
                            }
                        }
                    }

                    // Check if it's a user message or bot message
                    var messageClass = response.is_user ? 'self' : 'other';
                    var messageContent = response.message;

                    console.log('Processing message:', response);

                    // Check if there's media to display - allow media for both user and bot messages
                    if (response.media && response.media !== null && response.media !== "") {
                        // For media URLs, ensure they are complete URLs
                        var mediaUrl = response.media;
                        console.log('Original media URL:', mediaUrl);

                        if (!mediaUrl.startsWith('http')) {
                            mediaUrl = 'https://eflow-static01.s3-us-west-2.amazonaws.com/media/' + mediaUrl;
                            console.log('Updated media URL:', mediaUrl);
                        }

                        // Add media element to message content (video or image based on file type)
                        messageContent += getMediaHtml(mediaUrl);
                    }

                    messagesContainer.append([
                        '<li class="' + messageClass + '" id="'+response.id+'">',
                        messageContent,
                        '</li>'
                    ].join(''));
                });

                // Scroll to bottom
                messagesContainer.finish().animate({
                    scrollTop: messagesContainer.prop("scrollHeight")
                }, 250);
            }

            /**
             * Show the main chat loader
             */
            function showMainLoader() {
                console.log('Showing main loader');
                element.find('.loader-container').addClass('active');

                // Disable send button during initial loading
                element.find('#sendMessage').prop('disabled', true).css('background-color', '#9dbacd');
            }

            /**
             * Hide the main chat loader
             */
            function hideMainLoader() {
                console.log('Hiding main loader');
                element.find('.loader-container').removeClass('active');

                // Let updateSendButtonState handle enabling the button based on input state
                updateSendButtonState();
            }

            /**
             * Show the message loader
             */
            function showMessageLoader() {
                console.log('Showing message loader');
                element.find('.message-loader').addClass('active');

                // Disable send button during loading
                element.find('#sendMessage').prop('disabled', true).css('background-color', '#9dbacd');
            }

            /**
             * Hide the message loader
             */
            function hideMessageLoader() {
                console.log('Hiding message loader');
                element.find('.message-loader').removeClass('active');

                // Don't re-enable the button automatically - user needs to type or focus
                // updateSendButtonState will be called when user interacts with input
            }

            function createUUID() {
                // http://www.ietf.org/rfc/rfc4122.txt
                var s = [];
                var hexDigits = "0123456789abcdef";
                for (var i = 0; i < 36; i++) {
                    s[i] = hexDigits.substr(Math.floor(Math.random() * 0x10), 1);
                }
                s[14] = "4";
                s[19] = hexDigits.substr((s[19] & 0x3) | 0x8, 1);
                s[8] = s[13] = s[18] = s[23] = "-";

                var uuid = s.join("");
                return uuid;
            }

            function sendNewMessage() {
                var userInput = element.find('.text-box');
                var newMessage = userInput.html().replace(/\<div\>|\<br.*?\>/ig, '\n').replace(/\<\/div\>/g, '').trim().replace(/\n/g, '<br>');
                var sendButton = element.find('#sendMessage');

                console.log('sendNewMessage called, selectedImage:', selectedImage);

                // Check if button is disabled
                if (sendButton.prop('disabled')) {
                    console.log('Send button is disabled, not sending message');
                    return;
                }

                if (!newMessage && !selectedImage) {
                    console.log('No message or image to send');
                    return;
                }

                var messagesContainer = element.find('.messages');

                // Create message content with image preview if an image is selected
                var messageContent = newMessage;

                // Make a local copy of the selectedImage to prevent issues if it gets cleared
                var currentImage = selectedImage ? createSafeFileCopy(selectedImage) : null;

                if (currentImage && currentImage instanceof Blob) {
                    console.log('Sending message with image:', currentImage.name || strings.unnamedImage);
                    var reader = new FileReader();
                    reader.onload = function(e) {
                        try {
                            // Add image preview to the message
                            messageContent += '<div class="media-container"><img src="' + e.target.result + '" alt="' + strings.uploadedImageAlt + '" class="chat-media"></div>';

                            // Append the message with image to the chat
                            messagesContainer.append([
                                '<li class="self">',
                                messageContent,
                                '</li>'
                            ].join(''));

                            // Scroll to bottom
                            messagesContainer.finish().animate({
                                scrollTop: messagesContainer.prop("scrollHeight")
                            }, 250);

                            // Send the message with image
                            sendMessageWithImage(newMessage, currentImage);
                        } catch (err) {
                            console.error('Error processing image in sendNewMessage:', err);
                            // Fall back to sending text-only message
                            messagesContainer.append([
                                '<li class="self">',
                                newMessage,
                                '</li>'
                            ].join(''));
                            sendTextMessage(newMessage);
                        }
                    };
                    reader.onerror = function(err) {
                        console.error('Error reading image in sendNewMessage:', err);
                        // Fall back to sending text-only message
                        messagesContainer.append([
                            '<li class="self">',
                            newMessage,
                            '</li>'
                        ].join(''));
                        sendTextMessage(newMessage);
                    };

                    try {
                        reader.readAsDataURL(currentImage);
                    } catch (err) {
                        console.error('Error calling readAsDataURL in sendNewMessage:', err);
                        // Fall back to sending text-only message
                        messagesContainer.append([
                            '<li class="self">',
                            newMessage,
                            '</li>'
                        ].join(''));
                        sendTextMessage(newMessage);
                    }

                    // Clear the image preview and selection
                    removeSelectedImage();
                } else {
                    console.log('No valid image, sending text-only message');
                    // No image, just send text message
                    messagesContainer.append([
                        '<li class="self">',
                        messageContent,
                        '</li>'
                    ].join(''));

                    // Send text-only message
                    sendTextMessage(newMessage);
                }

                // Clean out old message
                userInput.html('');
                // Focus on input
                userInput.focus();

                // Disable send button after sending until user types again
                sendButton.prop('disabled', true).css('background-color', '#9dbacd');

                // Scroll to bottom
                messagesContainer.finish().animate({
                    scrollTop: messagesContainer.prop("scrollHeight")
                }, 250);
            }

            /**
             * Send a text-only message to the API
             * @param {string} message - The message text
             */
            function sendTextMessage(message) {
                // Hide previous loader if it's still visible
                hideMessageLoader();

                // Show the message loader with a small delay to ensure transition works
                setTimeout(function() {
                    showMessageLoader();
                }, 50);

                // Send message to API
                Ajax.call([{
                    methodname: 'local_eflow_aitutor_send_message',
                    args: {
                        courseid: params.courseId,
                        message: message,
                        userid: params.userId || 0,
                        activity_id: (id == 0) ? 0 : id
                    }
                }])[0].then(function(response) {
                    try {
                        // Check if we have a successful response
                        if (response.status) {
                            // Display AI response
                            var messagesContainer = element.find('.messages');
                            var messageContent = response.message;

                            // Check if the response contains media
                            if (response.data) {
                                try {
                                    var jsonData = JSON.parse(response.data);
                                    console.log('Parsed response data:', jsonData);

                                    if (jsonData && jsonData.media && jsonData.media !== null && jsonData.media !== "") {
                                        var mediaUrl = jsonData.media;
                                        console.log('Media URL from response:', mediaUrl);

                                        if (!mediaUrl.startsWith('http')) {
                                            mediaUrl = 'https://eflow-static01.s3-us-west-2.amazonaws.com/media/' + mediaUrl;
                                            console.log('Updated media URL:', mediaUrl);
                                        }

                                        // Add media element (video or image based on file type)
                                        messageContent += getMediaHtml(mediaUrl);
                                    } else {
                                        console.log('No media in response or media is empty');
                                    }
                                } catch (jsonError) {
                                    console.error('Error parsing response data JSON:', jsonError);
                                }
                            } else {
                                console.log('No response.data available');
                            }

                            // Only append and hide loader if messageContent is not empty (has text or media)
                            if (messageContent && messageContent.trim() !== '') {
                                hideMessageLoader();

                                messagesContainer.append([
                                    '<li class="other">',
                                    messageContent,
                                    '</li>'
                                ].join(''));

                                // Scroll to bottom again
                                messagesContainer.finish().animate({
                                    scrollTop: messagesContainer.prop("scrollHeight")
                                }, 250);
                            }
                        } else {
                            // Keep the loader active - don't hide it
                            // Just log the error to console instead of showing an error message
                            console.error('API response received but invalid format:', response);
                        }
                    } catch (e) {
                        // Keep the loader active - don't hide it
                        // Just log the error to console instead of showing an error message
                        console.error('Error processing API response:', e);
                    }
                }).catch(function(error) {
                    // Keep the loader active - don't hide it
                    // Just log the error to console instead of showing an error message
                    console.error('API call failed:', error);
                    // Hide the loader but don't re-enable the send button
                    element.find('.message-loader').removeClass('active');
                });

                // Set a timer to check for new messages after 5 seconds
                // This happens regardless of whether the API call succeeds or fails
                setTimeout(function() {
                    console.log('Checking for new messages 5 seconds after sending a message');
                    poller.poll();
                }, 5000);
            }

            /**
             * Send a message with an image attachment
             * @param {string} message - The message text
             * @param {File|Blob} image - The image file to upload
             */
            function sendMessageWithImage(message, image) {
                console.log('sendMessageWithImage called with:', message, image);

                // Validate that image is a valid File/Blob object
                if (!image) {
                    console.error('No image provided to sendMessageWithImage');
                    hideMessageLoader();
                    return;
                }

                if (!(image instanceof Blob)) {
                    console.error('Invalid image object provided to sendMessageWithImage, type:', typeof image);
                    hideMessageLoader();
                    return;
                }

                console.log('Image validation passed:', image.name, image.type, image.size);

                // Hide previous loader if it's still visible
                hideMessageLoader();

                // Show the message loader with a small delay to ensure transition works
                setTimeout(function() {
                    showMessageLoader();
                }, 50);

                // Convert the image to base64 for sending via AJAX
                var reader = new FileReader();
                reader.onload = function(e) {
                    try {
                        console.log('Image loaded successfully for API call');
                        // Get base64 data (remove the data:image/jpeg;base64, prefix)
                        var base64Result = e.target.result;
                        var base64data = base64Result.split(',')[1];

                        console.log('Base64 data prepared, length:', base64data.length);

                        // Send message with image using the external API
                        Ajax.call([{
                            methodname: 'local_eflow_aitutor_send_message',
                            args: {
                                courseid: params.courseId,
                                message: message,
                                filecontents: base64data,
                                filename: image.name || strings.unnamedImage,
                                filetype: image.type,
                                userid: params.userId || 0,
                                activity_id: (id == 0) ? 0 : id
                            }
                        }])[0].then(function(response) {
                            console.log('API response received:', response);
                            try {
                                // Check if we have a successful response
                                if (response.status) {
                                    // Display AI response
                                    var messagesContainer = element.find('.messages');
                                    var messageContent = response.message;

                                    // Check if the response contains media
                                    if (response.data) {
                                        try {
                                            var jsonData = JSON.parse(response.data);
                                            console.log('Parsed response data:', jsonData);

                                            if (jsonData && jsonData.media && jsonData.media !== null && jsonData.media !== "") {
                                                var mediaUrl = jsonData.media;
                                                console.log('Media URL from response:', mediaUrl);

                                                if (!mediaUrl.startsWith('http')) {
                                                    mediaUrl = 'https://eflow-static01.s3-us-west-2.amazonaws.com/media/' + mediaUrl;
                                                    console.log('Updated media URL:', mediaUrl);
                                                }

                                                // Add media element (video or image based on file type)
                                                messageContent += getMediaHtml(mediaUrl);
                                            } else {
                                                console.log('No media in response or media is empty');
                                            }
                                        } catch (jsonError) {
                                            console.error('Error parsing response data JSON:', jsonError);
                                        }
                                    } else {
                                        console.log('No response.data available');
                                    }

                                    // Only append and hide loader if messageContent is not empty (has text or media)
                                    if (messageContent && messageContent.trim() !== '') {
                                        hideMessageLoader();

                                        messagesContainer.append([
                                            '<li class="other">',
                                            messageContent,
                                            '</li>'
                                        ].join(''));

                                        // Scroll to bottom again
                                        messagesContainer.finish().animate({
                                            scrollTop: messagesContainer.prop("scrollHeight")
                                        }, 250);
                                    }
                                } else {
                                    // Keep the loader active - don't hide it
                                    // Just log the error to console instead of showing an error message
                                    console.error('API response received but invalid format:', response);
                                    // Hide the loader but don't re-enable the send button
                                    element.find('.message-loader').removeClass('active');
                                }
                            } catch (e) {
                                // Keep the loader active - don't hide it
                                // Just log the error to console instead of showing an error message
                                console.error('Error processing API response:', e);
                                // Hide the loader but don't re-enable the send button
                                element.find('.message-loader').removeClass('active');
                            }
                        }).catch(function(error) {
                            // Keep the loader active - don't hide it
                            // Just log the error to console instead of showing an error message
                            console.error('Image upload failed:', error);
                            // Hide the loader but don't re-enable the send button
                            element.find('.message-loader').removeClass('active');
                        });

                        // Set a timer to check for new messages after 5 seconds
                        setTimeout(function() {
                            console.log('Checking for new messages 5 seconds after sending a message with image');
                            poller.poll();
                        }, 5000);
                    } catch (e) {
                        console.error('Error in FileReader onload handler:', e);
                        // Hide the loader but don't re-enable the send button
                        element.find('.message-loader').removeClass('active');
                    }
                };

                reader.onerror = function(e) {
                    console.error('Error reading image file:', e);
                    // Hide the loader but don't re-enable the send button
                    element.find('.message-loader').removeClass('active');
                };

                try {
                    console.log('Starting to read image as data URL');
                    reader.readAsDataURL(image);
                    console.log('readAsDataURL called successfully');
                } catch (e) {
                    console.error('Error calling readAsDataURL:', e);
                    // Hide the loader but don't re-enable the send button
                    element.find('.message-loader').removeClass('active');
                }
            }

            function onMetaAndEnter(event) {
                if ((event.metaKey || event.ctrlKey) && event.keyCode == 13) {
                    // Check if send button is enabled before sending
                    if (!element.find('#sendMessage').prop('disabled')) {
                        sendNewMessage();
                    } else {
                        console.log('Send button is disabled, not sending message on Ctrl/Meta+Enter');
                    }
                }
            }

            /**
             * Creates a safe copy of a File/Blob object
             * @param {File|Blob} originalFile - The original file object
             * @return {File|Blob|null} A safe copy of the file or null if failed
             */
            function createSafeFileCopy(originalFile) {
                if (!originalFile || !(originalFile instanceof Blob)) {
                    console.error('Invalid file provided to createSafeFileCopy');
                    return null;
                }

                try {
                    // Create a new File object (which is a specialized Blob)
                    if (window.File && originalFile instanceof File) {
                        return new File([originalFile], originalFile.name, {
                            type: originalFile.type,
                            lastModified: originalFile.lastModified
                        });
                    } else {
                        // Fallback to creating a Blob
                        return new Blob([originalFile], {type: originalFile.type});
                    }
                } catch (err) {
                    console.error('Error creating safe file copy:', err);
                    return originalFile; // Return original as fallback
                }
            }
        });
    };

    return {
        init: init
    };
});
