// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * API polling module for the eFlow Ai Tutor.
 *
 * @module     local_eflow_aitutor/poller
 * @copyright  2025 Nawaz sharif<shasharif02@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

import {call as fetchMany} from 'core/ajax';
import Notification from 'core/notification';

/**
 * API poller for checking responses periodically.
 */
export default class ApiPoller {
    /**
     * Constructor
     *
     * @param {Object} config Configuration options
     * @param {Number} config.courseId The course ID
     * @param {Number} config.userId The user ID
     * @param {Function} config.onNewResponse Callback for when a new response is received
     * @param {Number} config.interval Polling interval in milliseconds (default: 30000)
     * @param {String} config.floatOption Whether to use with_option or without_option (default: with_option)
     */
    constructor(config) {
        this.courseId = config.courseId;
        this.userId = config.userId;
        this.onNewResponse = config.onNewResponse;
        this.interval = config.interval || 30000; // Default 30 seconds
        this.floatOption = config.floatOption || 'with_option';
        this.isPolling = false;
        this.timerId = null;
        this.lastResponseId = null;
        this.initialHistoryLoaded = false;
    }

    /**
     * Start polling for new responses
     */
    start() {
        if (this.isPolling) {
            return;
        }

        this.isPolling = true;

        // Load initial history first
        this.loadInitialHistory();

        // Set up recurring polling
        this.timerId = setInterval(() => {
            this.poll();
        }, this.interval);
    }

    /**
     * Stop polling
     */
    stop() {
        if (this.timerId) {
            clearInterval(this.timerId);
            this.timerId = null;
        }
        this.isPolling = false;
    }

    /**
     * Load initial chat history
     */
    loadInitialHistory() {
        console.log('Loading initial chat history with flag:', this.floatOption);

        const promises = fetchMany([{
            methodname: 'local_eflow_aitutor_check_responses',
            args: {
                courseid: this.courseId,
                userid: this.userId,
                lastresponseid: 0, // Start from beginning to get history
                group_id: 2,
                course_id: 298,
                flag: this.floatOption
            }
        }]);

        promises[0].then((response) => {
            if (response.status && response.data && response.data.length > 0) {
                console.log('Loaded initial chat history with ' + response.data.length + ' messages');
                // Store the ID of the last response
                this.lastResponseId = response.data[response.data.length - 1].id;

                // Notify callback with all history messages
                this.onNewResponse(response.data);
                this.initialHistoryLoaded = true;
            } else {
                console.log('No initial chat history found or error loading history');

                // If first attempt with 'with_option' fails, try 'without_option'
                if (this.floatOption === 'with_option') {
                    console.log('Retrying with without_option flag');
                    this.floatOption = 'without_option';
                    this.loadInitialHistory();
                } else {
                    // We've tried both options and neither worked, notify callback with empty data
                    // This will hide the loader
                    this.onNewResponse([]);
                }
            }
        }).catch((error) => {
            console.error('Error loading initial chat history:', error);

            // If first attempt with 'with_option' fails, try 'without_option'
            if (this.floatOption === 'with_option') {
                console.log('Retrying with without_option flag after error');
                this.floatOption = 'without_option';
                this.loadInitialHistory();
            } else {
                // We've tried both options and neither worked, notify callback with empty data
                // This will hide the loader
                this.onNewResponse([]);
            }
        });
    }

    /**
     * Execute a single poll operation
     */
    poll() {
        // Don't poll until initial history is loaded or attempted
        if (!this.initialHistoryLoaded && this.floatOption === 'without_option') {
            console.log('Waiting for initial history to load before polling');
            return;
        }

        console.log('Polling for new messages with flag:', this.floatOption);

        const promises = fetchMany([{
            methodname: 'local_eflow_aitutor_check_responses',
            args: {
                courseid: this.courseId,
                userid: this.userId,
                lastresponseid: this.lastResponseId,
                group_id: 2,
                course_id: 298,
                flag: this.floatOption
            }
        }]);

        promises[0].then((response) => {
            if (response.status && response.data && response.data.length > 0) {
                console.log('Received ' + response.data.length + ' new messages');

                // Store the ID of the last response
                this.lastResponseId = response.data[response.data.length - 1].id;

                // Notify callback with new responses
                this.onNewResponse(response.data);
            }
        }).catch((error) => {
            console.error('Error polling for new responses:', error);
        });
    }
}
