<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * External API for checking for new responses from the AI Tutor.
 *
 * @package    local_eflow_aitutor
 * @copyright  2025 Nawaz sharif<shasharif02@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_eflow_aitutor\external;

// We need to explicitly require and use these classes.
use core_external\external_api;
use core_external\external_function_parameters;
use core_external\external_value;
use core_external\external_single_structure;
use core_external\external_multiple_structure;

defined('MOODLE_INTERNAL') || die();

require_once($CFG->libdir . '/externallib.php');
require_once($CFG->dirroot . '/local/eflow_aitutor/lib.php');

use context_course;
use moodle_exception;

/**
 * External function to check for new responses from the AI Tutor.
 *
 * @package    local_eflow_aitutor
 * @copyright  2025 Nawaz sharif<shasharif02@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class check_responses extends external_api
{

    /**
     * Returns description of method parameters.
     *
     * @return external_function_parameters
     */
    public static function execute_parameters()
    {
        return new external_function_parameters([
            'courseid' => new external_value(PARAM_INT, 'The course ID'),
            'userid' => new external_value(PARAM_INT, 'The user ID', VALUE_DEFAULT, 0),
            'lastresponseid' => new external_value(PARAM_INT, 'The ID of the last response', VALUE_DEFAULT, 0),
            'group_id' => new external_value(PARAM_INT, 'The group ID', VALUE_DEFAULT, 2),
            'course_id' => new external_value(PARAM_INT, 'The course ID for API', VALUE_DEFAULT, 298),
            'flag' => new external_value(PARAM_TEXT, 'Additional flag parameter', VALUE_DEFAULT, 'with_option'),
        ]);
    }

    /**
     * Checks for new responses from the external API.
     *
     * @param int $courseid The course ID
     * @param int $userid The user ID
     * @param int $lastresponseid The ID of the last response
     * @param int $group_id The group ID
     * @param int $course_id The course ID for API
     * @param string $flag Additional flag parameter
     * @return array Response data
     */
    public static function execute($courseid, $userid = 0, $lastresponseid = 0, $group_id = 2, $course_id = 298, $flag = 'with_option')
    {
        global $USER, $DB, $CFG;

        // Parameter validation.
        $params = self::validate_parameters(self::execute_parameters(), [
            'courseid' => $courseid,
            'userid' => $userid,
            'lastresponseid' => $lastresponseid,
            'group_id' => $group_id,
            'course_id' => $course_id,
            'flag' => $flag,
        ]);

        // If no userid is provided, use the current user.
        if (empty($params['userid'])) {
            $params['userid'] = $USER->id;
        }

        // Context validation.
        $context = \context_course::instance($params['courseid']);
        self::validate_context($context);

        // Check capability.
        if (!has_capability('local/eflow_aitutor:view', $context)) {
            throw new \moodle_exception('nopermissions', 'error', '', 'local/eflow_aitutor:view');
        }

        // Get API settings.
        $apibaseurl = rtrim(get_config('local_eflow_aitutor', 'apiurl'), '/');
        if (empty($apibaseurl)) {
            throw new \moodle_exception('apiurlnotset', 'local_eflow_aitutor');
        }

        // Authentication credentials
        $username = get_config('local_eflow_aitutor', 'api_username');
        $password = get_config('local_eflow_aitutor', 'api_password');

        // Check if credentials are configured
        if (empty($username) || empty($password)) {
            throw new \moodle_exception('apicredentialsnotset', 'local_eflow_aitutor');
        }

        // Get authentication token
        $token = \local_eflow_aitutor_get_api_token($username, $password, $apibaseurl);
        if (empty($token)) {
            throw new \moodle_exception('apiauthfailed', 'local_eflow_aitutor');
        }

        // Fetch responses from API
        $responses = self::fetch_responses(
            $apibaseurl,
            $token,
            $params['courseid'],
            $params['userid'],
            $params['lastresponseid'],
            intval(get_config('local_eflow_aitutor', 'group_id') ?: 2),
            intval(get_config('local_eflow_aitutor', 'course_id') ?: 298),
            $params['flag']
        );

        return [
            'status' => true,
            'message' => get_string('chat_responses_success', 'local_eflow_aitutor'),
            'data' => $responses['data'] ?? [],
        ];
    }

    /**
     * Returns description of method result value.
     *
     * @return external_single_structure
     */
    public static function execute_returns()
    {
        return new external_single_structure([
            'status' => new external_value(PARAM_BOOL, 'The status of the request'),
            'message' => new external_value(PARAM_TEXT, 'Status message'),
            'data' => new external_multiple_structure(
                new external_single_structure([
                    'id' => new external_value(PARAM_INT, 'Response ID'),
                    'message' => new external_value(PARAM_RAW, 'Response message'),
                    'timestamp' => new external_value(PARAM_INT, 'Response timestamp'),
                    'is_user' => new external_value(PARAM_BOOL, 'Whether the message is from the user', VALUE_DEFAULT, false),
                    'media' => new external_value(PARAM_TEXT, 'Response media', VALUE_DEFAULT, null),
                ]),
                'Array of responses',
                VALUE_DEFAULT,
                []
            ),
        ]);
    }

    /**
     * Fetches responses from the external API.
     *
     * @param string $apibaseurl The API base URL
     * @param string $token The authentication token
     * @param int $courseid The course ID
     * @param int $userid The user ID
     * @param int $lastresponseid The ID of the last response
     * @param int $groupid The group ID
     * @param int $apicourseid The course ID for API
     * @param string $flag Additional flag parameter
     * @return array The API response
     */
    private static function fetch_responses($apibaseurl, $token, $courseid, $userid, $lastresponseid, $groupid = 2, $apicourseid = 298, $flag = 'with_option')
    {
        $curl = new \curl();
        $curl->setHeader(['Content-Type: application/json']);
        $curl->setHeader(['Authorization: Bearer ' . $token]);
        $curl->setHeader(['User-Agent: moodle/eflow_aitutor']);

        // Build the API request URL with query parameters
        $url = $apibaseurl . '/ma/app/messages';

        // Add query parameters
        $params = [
            'group_id' => $groupid,
            'course_id' => $apicourseid,
            'flag' => $flag
        ];

        // Add last response ID if available for pagination
        if ($lastresponseid > 0) {
            $params['last_id'] = $lastresponseid;
        }

        // Log the request for debugging
        debugging('eFlow AI Tutor: Sending GET request to: ' . $url . '?' . http_build_query($params), DEBUG_DEVELOPER);

        // Perform GET request to get messages history
        $response = $curl->get($url, $params);
        $result = json_decode($response, true);

        // Log the response for debugging
        if ($result === null) {
            debugging('eFlow AI Tutor: Invalid API response format', DEBUG_DEVELOPER);
        } else {
            debugging('eFlow AI Tutor: API response received: ' . substr(json_encode($result), 0, 200) . '...', DEBUG_DEVELOPER);
        }

        if ($result === null) {
            return [
                'data' => [],
            ];
        }

        // Format the response data according to the new API structure
        $responses = [];

        // Check if we have the body.data structure from the API
        if (isset($result['body']) && isset($result['body']['data']) && is_array($result['body']['data'])) {
            foreach ($result['body']['data'] as $item) {
                // Only include messages that have direction "outbound-api" (bot messages) or
                // direction "Incoming" (user messages)
                if (isset($item['direction'])) {
                    $isUser = ($item['direction'] === 'Incoming');
                    $responses[] = [
                        'id' => $item['id'] ?? 0,
                        'message' => $item['text'] ?? '',
                        'timestamp' => strtotime($item['date_sent'] ?? 'now'),
                        'is_user' => $isUser,
                        'media' => $item['media'] ?? null // Include media field if available
                    ];
                }
            }
        } elseif (isset($result['messages']) && is_array($result['messages'])) {
            // Process using the new API format
            $responses = self::process_response_data($result);
        }

        // Process question data if available (for initial messages)
        if (isset($result['question']) && !empty($result['question']['title'])) {
            // Add the current question as a message
            $responses[] = [
                'id' => $result['question']['id'] ?? 0,
                'message' => $result['question']['title'] ?? '',
                'timestamp' => strtotime('now'),
                'is_user' => false,
                'media' => $result['question']['media'] ?? null // Include media field if available
            ];
        }

        return [
            'data' => $responses,
        ];
    }

    /**
     * Process the API response data.
     *
     * @param array $responseData The API response data
     * @return array Processed messages
     */
    private static function process_response_data($responseData)
    {
        $messages = [];

        if (isset($responseData['messages']) && is_array($responseData['messages'])) {
            foreach ($responseData['messages'] as $message) {
                // Ensure we have the required fields
                if (isset($message['id']) && isset($message['text'])) {
                    $isUser = isset($message['moodle_user_id']) && $message['moodle_user_id'] > 0;

                    $processedMessage = [
                        'id' => $message['id'],
                        'message' => $message['text'],
                        'is_user' => $isUser,
                        'timestamp' => isset($message['date_sent']) ? strtotime($message['date_sent']) : time(),
                    ];

                    // Add media URL if available and ensure it's a full URL
                    if (isset($message['media']) && !empty($message['media'])) {
                        $mediaUrl = $message['media'];
                        // If the media URL is not already a full URL, prepend the S3 bucket URL
                        if (!preg_match('/^https?:\/\//', $mediaUrl)) {
                            $mediaUrl = 'https://eflow-static01.s3-us-west-2.amazonaws.com/media/' . $mediaUrl;
                        }
                        $processedMessage['media'] = $mediaUrl;

                        // Log the processed media URL for debugging
                        debugging('Processed media URL: ' . $mediaUrl, DEBUG_DEVELOPER);
                    }

                    $messages[] = $processedMessage;
                }
            }
        }

        return $messages;
    }
}
