<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * External API for sending messages to the AI Tutor.
 *
 * @package    local_eflow_aitutor
 * @copyright  2025 Nawaz sharif<shasharif02@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_eflow_aitutor\external;

defined('MOODLE_INTERNAL') || die();

require_once($CFG->libdir . '/externallib.php');
require_once($CFG->dirroot . '/local/eflow_aitutor/lib.php');
require_once($CFG->libdir . '/filelib.php');

use external_api;
use external_function_parameters;
use external_multiple_structure;
use external_single_structure;
use external_value;
use context_course;
use moodle_exception;

/**
 * External function to send a message to the AI Tutor.
 *
 * @package    local_eflow_aitutor
 * @copyright  2025 Nawaz sharif<shasharif02@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class send_message extends external_api
{

    /**
     * Returns description of method parameters.
     *
     * @return external_function_parameters
     */
    public static function execute_parameters()
    {
        return new external_function_parameters([
            'courseid' => new external_value(PARAM_INT, 'The course ID'),
            'message' => new external_value(PARAM_TEXT, 'The message to send'),
            'userid' => new external_value(PARAM_INT, 'The user ID', VALUE_DEFAULT, 0),
            'filecontents' => new external_value(PARAM_TEXT, 'The base64 encoded file contents', VALUE_DEFAULT, ''),
            'filename' => new external_value(PARAM_FILE, 'The name of the file', VALUE_DEFAULT, ''),
            'filetype' => new external_value(PARAM_TEXT, 'The MIME type of the file', VALUE_DEFAULT, ''),
            'activity_id' => new external_value(PARAM_INT, 'The INT type of the activity', VALUE_DEFAULT, '0'),
        ]);
    }

    /**
     * Sends a message to the external API.
     *
     * @param int $courseid The course ID
     * @param string $message The message to send
     * @param int $userid The user ID
     * @param string $filecontents The base64 encoded file contents (optional)
     * @param string $filename The name of the file (optional)
     * @param string $filetype The MIME type of the file (optional)
     * @return array Response data
     */
    public static function execute($courseid, $message, $userid = 0, $filecontents = '', $filename = '', $filetype = '',$activity_id = 0)
    {
        global $USER, $DB, $CFG, $PAGE;

        // Parameter validation.
        $params = self::validate_parameters(self::execute_parameters(), [
            'courseid' => $courseid,
            'message' => $message,
            'userid' => $userid,
            'filecontents' => $filecontents,
            'filename' => $filename,
            'filetype' => $filetype,
        ]);

        // If no userid is provided, use the current user.
        if (empty($params['userid'])) {
            $params['userid'] = $USER->id;
        }


        // Context validation.
        $context = \context_course::instance($params['courseid']);
        self::validate_context($context);

        // Check capability.
        if (!has_capability('local/eflow_aitutor:view', $context)) {
            throw new \moodle_exception('nopermissions', 'error', '', 'local/eflow_aitutor:view');
        }

        // Get API settings.
        $apibaseurl = rtrim(get_config('local_eflow_aitutor', 'apiurl'), '/');
        if (empty($apibaseurl)) {
            throw new \moodle_exception('apiurlnotset', 'local_eflow_aitutor');
        }

        // Authentication credentials
        $username = get_config('local_eflow_aitutor', 'api_username');
        $password = get_config('local_eflow_aitutor', 'api_password');

        // Check if credentials are configured
        if (empty($username) || empty($password)) {
            throw new \moodle_exception('apicredentialsnotset', 'local_eflow_aitutor');
        }

        // Get authentication token
        $token = \local_eflow_aitutor_get_api_token($username, $password, $apibaseurl);
        if (empty($token)) {
            throw new \moodle_exception('apiauthfailed', 'local_eflow_aitutor');
        }

        // Format current date for API
        $currentdate = gmdate('Y-m-d\TH:i:s.v\Z');

        // Check if we have a file to upload
        $tempfile = null;
        $cfile = null;

        if (!empty($params['filecontents']) && !empty($params['filename']) && !empty($params['filetype'])) {
            // Validate file type if provided
            $allowedtypes = ['image/jpeg', 'image/png', 'image/gif'];
            if (!in_array($params['filetype'], $allowedtypes)) {
                throw new moodle_exception('invalidfiletype', 'local_eflow_aitutor');
            }

            // Decode base64 file contents
            $decodedcontents = base64_decode($params['filecontents']);
            if ($decodedcontents === false) {
                throw new moodle_exception('invalidfilecontents', 'local_eflow_aitutor');
            }

            // Create a temporary file
            $tempdir = make_temp_directory('local_eflow_aitutor');
            $tempfile = $tempdir . '/' . uniqid('upload_', true) . '_' . clean_filename($params['filename']);

            if (file_put_contents($tempfile, $decodedcontents) === false) {
                throw new moodle_exception('cannotwritefile', 'local_eflow_aitutor');
            }

            // Create cURL file object for the media file
            $cfile = new \CURLFile($tempfile, $params['filetype'], $params['filename']);
        }

        // Prepare the API endpoint for sending messages
        $messageendpoint = $apibaseurl . "/web/app/receiveMessage";

        // Use the exact URL from the curl example if the API base URL is not set or if it's different
        if (empty($apibaseurl) || strpos($apibaseurl, 'api7.eflow.app') === false) {
            $messageendpoint = "https://api7.eflow.app/api/v2/web/app/receiveMessage";
            debugging('Using hardcoded API endpoint: ' . $messageendpoint, DEBUG_DEVELOPER);
        }

        // Prepare data with the required API format - use exact field names from curl example
        $requestdata = [
            'group_id' => intval(get_config('local_eflow_aitutor', 'group_id') ?: 2),
            'text' => (isset($params['message']) && $params['message'] !== '') ? strval($params['message']) : "",
            'date_sent' => $currentdate,
            'moodle_course_id' => (int)$params['courseid'],
            'moodle_activity_id' => $activity_id,
            'moodle_user_id' => (int)$params['userid']
        ];

        // Initialize Moodle's curl class
        $curl = new \curl();
        $curl->setHeader(['Authorization: Bearer ' . $token]);
        $curl->setHeader(['User-Agent: Moodle/FloatingAITutor']);

        // If we have a file, use multipart/form-data
        if ($cfile) {
            // For file uploads, we need to use a different approach
            // as Moodle's curl class doesn't directly support CURLFile

            // Create a temporary file with JSON data
            $tempjsonfile = make_temp_directory('local_eflow_aitutor') . '/request_data_' . uniqid() . '.json';
            file_put_contents($tempjsonfile, json_encode($requestdata));

            // Build multipart form data with both the JSON data and the file
            $postfields = [
                'json' => new \curl_file($tempjsonfile, 'application/json', 'data.json'),
                'media' => new \curl_file($tempfile, $params['filetype'], $params['filename'])
            ];

            // Use Moodle's curl to post the multipart form data
            $apiresponse = $curl->post($messageendpoint, $postfields);

            // Clean up temporary JSON file
            if (file_exists($tempjsonfile)) {
                unlink($tempjsonfile);
            }
        } else {
            // For regular JSON requests, just post the JSON data
            $curl->setHeader(['Content-Type: application/json']);
            $apiresponse = $curl->post($messageendpoint, json_encode($requestdata));
        }

        // Get response information
        $info = $curl->get_info();
        $httpcode = $info['http_code'];
        $err = $curl->get_errno();

        // Debug logging
        if (debugging()) {
            $debuginfo = "eFlow AI Tutor API request:\n";
            $debuginfo .= "URL: $messageendpoint\n";
            $debuginfo .= "REQUEST: " . json_encode($requestdata) . "\n";
            $debuginfo .= "RESPONSE CODE: $httpcode\n";
            $debuginfo .= "RESPONSE: $apiresponse\n";
            if ($err) {
                $debuginfo .= "ERROR: " . $curl->error . "\n";
            }
            debugging($debuginfo, DEBUG_DEVELOPER);
        }

        // Clean up temporary file if it exists
        if ($tempfile && file_exists($tempfile)) {
            unlink($tempfile);
        }

        // Handle API response
        if ($err) {
            return [
                'status' => false,
                'message' => get_string('chat_failed_to_connect_with_error', 'local_eflow_aitutor', $curl->error),
                'data' => null
            ];
        }

        if ($httpcode < 200 || $httpcode >= 300) {
            return [
                'status' => false,
                'message' => get_string('chat_error_code_with_number', 'local_eflow_aitutor', $httpcode),
                'data' => null
            ];
        }

        $result = json_decode($apiresponse, true);

        if ($result && isset($result['message'])) {
            // Process the response to ensure media URLs are properly formatted
            if (isset($result['media']) && $result['media'] !== null && $result['media'] !== "") {
                // If the media URL is not already a full URL, prepend the S3 bucket URL
                if (!preg_match('/^https?:\/\//', $result['media'])) {
                    $result['media'] = 'https://eflow-static01.s3-us-west-2.amazonaws.com/media/' . $result['media'];
                }
            }

            // Log the processed result for debugging
            if (debugging()) {
                debugging('Processed API response: ' . json_encode($result), DEBUG_DEVELOPER);
            }

            return [
                'status' => true,
                'message' => $result['message'],
                'data' => json_encode($result)
            ];
        }

        return [
            'status' => true,
            'message' => isset($result['message']) ? $result['message'] : '',
            'data' => $apiresponse,
        ];
    }

    /**
     * Returns description of method result value.
     *
     * @return external_single_structure
     */
    public static function execute_returns()
    {
        return new external_single_structure([
            'status' => new external_value(PARAM_BOOL, 'The status of the message sending'),
            'message' => new external_value(PARAM_TEXT, 'Status message'),
            'data' => new external_value(PARAM_RAW, 'Response data from API', VALUE_OPTIONAL),
        ]);
    }
}
