<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Hook callbacks for eFlow Ai Tutor.
 *
 * @package    local_eflow_aitutor
 * @copyright  2025 Nawaz sharif<shasharif02@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_eflow_aitutor;

use core_course\hook\after_form_definition;
use core_course\hook\after_form_submission;
use core\hook\output\before_footer_html_generation;
use core\hook\output\before_standard_head_html_generation;
use context_course;
use context_system;
use moodle_url;
use user_picture;

/**
 * Hook callbacks class for the eflow_aitutor plugin.
 */
class hook_callbacks
{
    /**
     * Hook to modify the form after its definition.
     *
     * @param after_form_definition $hook The hook instance
     */
    public static function after_form_definition(after_form_definition $hook)
    {
        global $COURSE;

        $mform = $hook->mform;
        $mform->addElement('header', 'eflow_aitutor', \get_string('pluginname', 'local_eflow_aitutor'), [], [
            'collapsed' => false,
        ]);

        $mform->addElement('select', 'eflow_aitutor_status', \get_string('status', 'local_eflow_aitutor'), [
            '0' => \get_string('disabled', 'local_eflow_aitutor'),
            '1' => \get_string('enabled', 'local_eflow_aitutor'),
        ]);
        $default = \get_config('local_eflow_aitutor', "eflow_aitutor-$COURSE->id");
        if ($default === false) {
            $mform->setDefault('eflow_aitutor_status', '1');
        } else {
            $mform->setDefault('eflow_aitutor_status', $default);
        }
    }

    /**
     * Hook to handle form submission after it is processed.
     *
     * @param after_form_submission $hook The hook instance containing the form submission data
     */
    public static function after_form_submission(after_form_submission $hook)
    {
        $courseid = $hook->get_data()->id;
        $status = $hook->get_data()->eflow_aitutor_status ?? false;
        $name = "eflow_aitutor-$courseid";
        \set_config($name, $status, 'local_eflow_aitutor');
    }

    /**
     * Add CSS to the page head.
     *
     * @param before_standard_head_html_generation $hook The hook instance
     */
    public static function before_standard_head_html_generation(before_standard_head_html_generation $hook)
    {
        global $PAGE, $COURSE, $CFG, $OUTPUT;

        // Only include the CSS on course pages.
        if ($PAGE->pagelayout !== 'course' && $PAGE->pagelayout !== 'incourse') {
            return;
        }

        // Check if the plugin is enabled globally.
        $enabled = \get_config('local_eflow_aitutor', 'enabled');
        if (empty($enabled)) {
            return;
        }

        // Check for course-specific settings
        if ($COURSE->id && $COURSE->id > 1) {
            $courseenabled = \get_config('local_eflow_aitutor', "eflow_aitutor-{$COURSE->id}");

            // If course setting exists and it's disabled, don't include the CSS
            if (isset($courseenabled) && $courseenabled == '0') {
                return;
            }
        }

        // Check if custom AI avatar is set
        $aiAvatarUrl = '';
        $fs = \get_file_storage();
        $syscontext = context_system::instance();
        $aiAvatarFiles = $fs->get_area_files($syscontext->id, 'local_eflow_aitutor', 'ai_avatar', 0, 'sortorder', false);

        if ($aiAvatarFiles) {
            foreach ($aiAvatarFiles as $file) {
                $aiAvatarUrl = moodle_url::make_pluginfile_url(
                    $file->get_contextid(),
                    $file->get_component(),
                    $file->get_filearea(),
                    0,  // Using 0 as itemid to match the file storage
                    $file->get_filepath(),
                    $file->get_filename()
                )->out();

                // Set default if URL is empty (shouldn't happen, but just in case)
                if (empty($aiAvatarUrl)) {
                    $aiAvatarUrl = $CFG->wwwroot . '/local/eflow_aitutor/pix/eflow-logo.png';
                }

                $customCSS = ".floating-chat .messages li.other:before { background-image: url({$aiAvatarUrl}) !important; }";
                $hook->add_html("<style>{$customCSS}</style>");

                break;
            }
        }

        // Add Font Awesome CSS directly as HTML
        $hook->add_html('<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.min.css">');

        // Load plugin CSS using proper Moodle API - must be relative to Moodle root
        $PAGE->requires->css('/local/eflow_aitutor/styles/eflow_aitutor.css');
    }

    /**
     * Inject the eFlow Ai Tutor UI into course pages.
     *
     * @param before_footer_html_generation $hook The hook instance
     */
    public static function before_footer_html_generation(before_footer_html_generation $hook)
    {
        global $CFG, $COURSE, $USER, $PAGE, $DB, $OUTPUT;

        // Only include the chat on course pages.
        if ($PAGE->pagelayout !== 'course' && $PAGE->pagelayout !== 'incourse') {
            return;
        }

        // Check if the plugin is enabled globally.
        $enabled = \get_config('local_eflow_aitutor', 'enabled');
        if (empty($enabled)) {
            return;
        }


        // Check for course-specific settings
        if ($COURSE->id && $COURSE->id > 1) {
            $courseenabled = \get_config('local_eflow_aitutor', "eflow_aitutor-{$COURSE->id}");

            // If course setting exists and it's disabled, don't show the chat
            if (isset($courseenabled) && $courseenabled == '0') {
                return;
            }
        }

        // Check if the user has the capability to view the chat.
        // Using Moodle's context system instead of direct class instantiation
        $coursecontext = context_course::instance($COURSE->id);
        if (!\has_capability('local/eflow_aitutor:view', $coursecontext)) {
            return;
        }

        // Get the chat settings
        $buttoncolor = \get_config('local_eflow_aitutor', 'buttoncolor');
        $buttonposition = \get_config('local_eflow_aitutor', 'buttonposition');
        $buttonsize = \get_config('local_eflow_aitutor', 'buttonsize');
        $chatwidth = \get_config('local_eflow_aitutor', 'chatwidth');
        $chatheight = \get_config('local_eflow_aitutor', 'chatheight');
        $chattitle = \get_config('local_eflow_aitutor', 'chattitle');

        // Get the new color settings
        $headerbgcolor = \get_config('local_eflow_aitutor', 'headerbgcolor');
        $bodybgcolor = \get_config('local_eflow_aitutor', 'bodybgcolor');
        $headerfontcolor = \get_config('local_eflow_aitutor', 'headerfontcolor');
        $bodyfontcolor = \get_config('local_eflow_aitutor', 'bodyfontcolor');

        // Get iframe settings
        $iframe_enabled = \get_config('local_eflow_aitutor', 'iframe_enabled');
        $iframe_url = \get_config('local_eflow_aitutor', 'iframe_url');

        // If iframe URL contains the pattern format, replace with Moodle variables
        if (!empty($iframe_url)) {
            // Get the current activity ID if available
            $activity_id = 0;
            if ($PAGE->cm) {
                $activity_id = $PAGE->cm->id;
            }

            // Replace placeholder strings in the URL with actual Moodle values
            $iframe_url = str_replace('moodle_user_id', $USER->id, $iframe_url);
            $iframe_url = str_replace('moodle_course_id', $COURSE->id, $iframe_url);
            $iframe_url = str_replace('moodle_activity_id', $activity_id, $iframe_url);
        }

        // Set defaults if settings are not configured
        if (empty($buttoncolor)) {
            $buttoncolor = '#199393';
        }
        if (empty($buttonsize)) {
            $buttonsize = 40;
        }
        if (empty($chatwidth)) {
            $chatwidth = 250;
        }
        if (empty($chatheight)) {
            $chatheight = 400;
        }
        if (empty($chattitle)) {
            $chattitle = \get_string('defaultchattitle', 'local_eflow_aitutor');
        }
        if (empty($headerbgcolor)) {
            $headerbgcolor = '#3c719a';
        }
        if (empty($bodybgcolor)) {
            $bodybgcolor = '#cde6fa';
        }
        if (empty($headerfontcolor)) {
            $headerfontcolor = '#FFFFFF';
        }
        if (empty($bodyfontcolor)) {
            $bodyfontcolor = '#000000';
        }

        // Convert position to CSS values
        $positioncss = [];
        switch ($buttonposition) {
            case 'bottom-right':
                $positioncss = [
                    'bottom' => '10px',
                    'right' => '10px',
                    'top' => 'auto',
                    'left' => 'auto'
                ];
                break;
            case 'bottom-left':
                $positioncss = [
                    'bottom' => '10px',
                    'left' => '10px',
                    'top' => 'auto',
                    'right' => 'auto'
                ];
                break;
            case 'top-right':
                $positioncss = [
                    'top' => '10px',
                    'right' => '10px',
                    'bottom' => 'auto',
                    'left' => 'auto'
                ];
                break;
            case 'top-left':
                $positioncss = [
                    'top' => '10px',
                    'left' => '10px',
                    'bottom' => 'auto',
                    'right' => 'auto'
                ];
                break;
            default:
                $positioncss = [
                    'bottom' => '10px',
                    'right' => '10px',
                    'top' => 'auto',
                    'left' => 'auto'
                ];
        }

        // Check if custom logo is set
        $customlogourl = '';
        $fs = \get_file_storage();
        $syscontext = context_system::instance();
        $files = $fs->get_area_files($syscontext->id, 'local_eflow_aitutor', 'logo', 0, 'sortorder', false);
        if ($files) {
            foreach ($files as $file) {
                $customlogourl = moodle_url::make_pluginfile_url(
                    $file->get_contextid(),
                    $file->get_component(),
                    $file->get_filearea(),
                    0,  // Using 0 as itemid to match the file storage
                    $file->get_filepath(),
                    $file->get_filename()
                )->out();
                break;
            }
        }

        // Set default logo URL if no custom logo is uploaded
        if (empty($customlogourl)) {
            $customlogourl = $CFG->wwwroot . '/local/eflow_aitutor/pix/eflow-logo.png';
        }

        // Get AI avatar URL
        $aiAvatarUrl = '';
        $aiAvatarFiles = $fs->get_area_files($syscontext->id, 'local_eflow_aitutor', 'ai_avatar', 0, 'sortorder', false);
        if ($aiAvatarFiles) {
            foreach ($aiAvatarFiles as $file) {
                $aiAvatarUrl = moodle_url::make_pluginfile_url(
                    $file->get_contextid(),
                    $file->get_component(),
                    $file->get_filearea(),
                    0,
                    $file->get_filepath(),
                    $file->get_filename()
                )->out();
                break;
            }
        }

        // Set default AI avatar URL if no custom avatar is uploaded
        if (empty($aiAvatarUrl)) {
            $aiAvatarUrl = $CFG->wwwroot . '/local/eflow_aitutor/pix/eflow-logo.png';
        }

        // Add custom CSS for AI avatar
        $customCSS = ".floating-chat .messages li.other:before { background-image: url({$aiAvatarUrl}) !important; background-size: cover !important; }";
        $hook->add_html("<style>{$customCSS}</style>");

        // Get user profile image URL
        $userProfileImageUrl = '';
        $userpicture = new user_picture($USER);
        $userpicture->size = 50; // Size in pixels
        $userProfileImageUrl = $userpicture->get_url($PAGE)->out(false);

        // Prepare the HTML template with all settings
        $templatecontext = [
            'chatid' => uniqid('eflow_aitutor_'),
            'courseid' => $COURSE->id,
            'chattitle' => $chattitle,
            'buttoncolor' => $buttoncolor,
            'buttonsizepx' => $buttonsize . 'px',
            'chatwidthpx' => $chatwidth . 'px',
            'chatheightpx' => $chatheight . 'px',
            'positionbottom' => $positioncss['bottom'],
            'positionright' => $positioncss['right'],
            'positiontop' => $positioncss['top'],
            'positionleft' => $positioncss['left'],
            'customlogourl' => $customlogourl,
            'aiAvatarUrl' => $aiAvatarUrl,
            'userProfileImageUrl' => $userProfileImageUrl,
            'headerbgcolor' => $headerbgcolor,
            'bodybgcolor' => $bodybgcolor,
            'headerfontcolor' => $headerfontcolor,
            'bodyfontcolor' => $bodyfontcolor,
            'iframe_enabled' => !empty($iframe_enabled),
            'iframe_url' => $iframe_url
        ];

        // Get float option from URL first, then default to with_option
        $floatOption = \optional_param('float_option', 'with_option', \PARAM_ALPHA);

        // Check if there's a global default set
        $globalDefault = \get_config('local_eflow_aitutor', 'default_float_option');
        if (!empty($globalDefault)) {
            // Override if the global default is set
            $floatOption = $globalDefault;
        }

        // Validate that it's one of the allowed values
        if ($floatOption !== 'with_option' && $floatOption !== 'without_option') {
            $floatOption = 'with_option';
        }

        // Get template HTML
        $html = $OUTPUT->render_from_template('local_eflow_aitutor/chat', $templatecontext);

        // Initialize JavaScript using proper Moodle API
        $PAGE->requires->js_call_amd('local_eflow_aitutor/chat', 'init', [
            [
                'courseId' => $COURSE->id,
                'userId' => $USER->id,
                'userProfileImageUrl' => $userProfileImageUrl,
                'defaultImageUrl' => $customlogourl,
                'floatOption' => $floatOption
            ]
        ]);

        // Add the HTML to the hook output
        $hook->add_html($html);
    }
}
