<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Event observers for the eFlow Ai Tutor plugin.
 *
 * @package    local_eflow_aitutor
 * @copyright  2025 Nawaz sharif<shasharif02@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_eflow_aitutor;

defined('MOODLE_INTERNAL') || die();

use \core_user;
use \moodle_url;
use \context_module;
use \stored_file;

// Include lib.php for the local_eflow_aitutor_get_api_token function
global $CFG;
require_once($CFG->dirroot . '/local/eflow_aitutor/lib.php');
require_once($CFG->libdir . '/moodlelib.php'); // For fullname function
require_once($CFG->libdir . '/filelib.php'); // For get_file_storage function
require_once($CFG->dirroot . '/mod/forum/lib.php'); // For forum functions
require_once($CFG->dirroot . '/mod/quiz/lib.php'); // For quiz functions
require_once($CFG->dirroot . '/question/editlib.php'); // For question functions

// Define MUST_EXIST constant if not defined
if (!defined('MUST_EXIST')) {
    define('MUST_EXIST', true);
}

/**
 * Event observer class for the eflow_aitutor plugin.
 *
 * @package    local_eflow_aitutor
 * @copyright  2025 Nawaz sharif<shasharif02@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class observer
{
    /**
     * Send data to webhook.
     *
     * @param array $data The data to send
     * @return bool Success status
     */
    protected static function send_to_webhook($data)
    {
        // Get webhook URL from settings
        $webhookurl = \get_config('local_eflow_aitutor', 'webhook_url');
        if (empty($webhookurl)) {
            return false;
        }

        // Prepare the data for sending
        $jsondata = json_encode($data);

        // Use Moodle's curl class
        $curl = new \curl();
        $curl->setHeader(['Content-Type: application/json']);
        $curl->setHeader(['Content-Length: ' . strlen($jsondata)]);

        // Execute the request
        $result = $curl->post($webhookurl, $jsondata);
        $info = $curl->get_info();
        $httpcode = $info['http_code'];
        $error = $curl->error;

        // Use Moodle's debugging functions instead of file-based logging
        if (debugging()) {
            $debuginfo = "eFlow AI Tutor webhook request:\n";
            $debuginfo .= "URL: $webhookurl\n";
            $debuginfo .= "REQUEST: $jsondata\n";
            $debuginfo .= "RESPONSE: $result\n";
            $debuginfo .= "HTTP CODE: $httpcode\n";
            if (!empty($error)) {
                $debuginfo .= "ERROR: $error\n";
            }
            debugging($debuginfo, DEBUG_DEVELOPER);
        }

        return ($httpcode >= 200 && $httpcode < 300);
    }

    /**
     * Get file information for a module.
     *
     * @param object $moduleinfo The module info record
     * @param object $moduleinstance The module instance record
     * @param object $cm The course module record
     * @return array File information or null if no file
     */
    protected static function get_module_file_info($moduleinfo, $moduleinstance, $cm) {
        if (empty($moduleinstance->id)) {
            return null;
        }

        $fs = \get_file_storage();
        $context = \context_module::instance($cm->id);
        $component = '';
        $filearea = '';

        // Determine component and filearea based on module type
        if ($moduleinfo->name === 'resource' || $moduleinfo->name === 'file') {
            $component = 'mod_' . $moduleinfo->name;
            $filearea = $moduleinfo->name === 'resource' ? 'content' : 'content';
        } elseif ($moduleinfo->name === 'scorm') {
            $component = 'mod_scorm';
            $filearea = 'package';
        } else {
            return null;
        }

        $files = $fs->get_area_files($context->id, $component, $filearea, 0, 'sortorder DESC, id ASC', false);

        if (empty($files)) {
            return null;
        }

        $file = reset($files);  // Get the first/main file

        return [
            'filename' => $file->get_filename(),
            'fileurl' => \moodle_url::make_pluginfile_url(
                $file->get_contextid(),
                $file->get_component(),
                $file->get_filearea(),
                $file->get_itemid(),
                $file->get_filepath(),
                $file->get_filename(),
                false,  // Force download
                true    // This makes the URL publicly accessible without login
            )->out(),
            'mimetype' => $file->get_mimetype(),
            'size' => $file->get_filesize(),
            'visible' => (bool)$cm->visible
        ];
    }

    /**
     * Observer for the course_updated event.
     *
     * @param \core\event\course_updated $event The event
     * @return bool Success status
     */
    public static function course_updated(\core\event\course_updated $event)
    {
        global $DB;

        // Get the course data.
        $course = $DB->get_record('course', ['id' => $event->objectid], '*', \MUST_EXIST);

        // Get the user who updated the course.
        $user = $DB->get_record('user', ['id' => $event->userid]);

        // Prepare the data to send to the webhook.
        $data = [
            'event' => 'course_updated',
            'course' => [
                'id' => $course->id,
                'name' => $course->fullname,
                'summary' => strip_tags($course->summary),
                'visible' => (bool)$course->visible
            ],
            'updated_by' => [
                'userid' => $event->userid,
                'fullname' => $user ? \fullname($user) : ''
            ],
            'timestamp' => $event->timecreated
        ];

        // Send data to webhook.
        return self::send_to_webhook($data);
    }

    /**
     * Observer for the course_module_created event.
     *
     * @param \core\event\course_module_created $event The event
     * @return bool Success status
     */
    public static function course_module_created(\core\event\course_module_created $event)
    {
        global $DB, $USER;

        // Get the course data.
        $course = $DB->get_record('course', ['id' => $event->courseid], '*', \MUST_EXIST);

        // Get the course module data.
        $cm = $DB->get_record('course_modules', ['id' => $event->objectid], '*', \MUST_EXIST);
        $moduleinfo = $DB->get_record('modules', ['id' => $cm->module], 'name', \MUST_EXIST);

        // Get the module instance.
        $moduleinstance = $DB->get_record($moduleinfo->name, ['id' => $cm->instance]);

        // Get the user who created the module.
        $user = $DB->get_record('user', ['id' => $event->userid]);

        // Prepare the data to send to the webhook.
        $data = [
            'event' => 'course_module_created',
            'course' => [
                'id' => $course->id,
                'name' => $course->fullname,
                'summary' => strip_tags($course->summary),
                'visible' => (bool)$course->visible
            ],
            'module' => [
                'id' => $cm->id,
                'name' => $moduleinstance ? $moduleinstance->name : '',
                'modulename' => $moduleinfo->name,
                'visible' => (bool)$cm->visible
            ],
            'updated_by' => [
                'userid' => $event->userid,
                'fullname' => $user ? \fullname($user) : ''
            ],
            'timestamp' => $event->timecreated
        ];

        // Add module-specific data
        if ($moduleinstance) {
            // Get file information if this is a file-based module
            $fileInfo = self::get_module_file_info($moduleinfo, $moduleinstance, $cm);
            if ($fileInfo) {
                $data['module']['file'] = $fileInfo;
            }

            // Add page specific information
            if ($moduleinfo->name === 'page' && !empty($moduleinstance->id)) {
                // Include page content
                $data['module']['page_content'] = [
                    'title' => $moduleinstance->name,
                    'content' => strip_tags($moduleinstance->content),
                    'format' => $moduleinstance->contentformat,
                    'display_options' => [
                        'display_description' => (bool)$moduleinstance->displayoptions,
                        'print_heading' => (bool)$moduleinstance->printheading,
                        'print_last_modified' => (bool)$moduleinstance->printlastmodified
                    ]
                ];
            }
            // Add forum specific information
            elseif ($moduleinfo->name === 'forum' && !empty($moduleinstance->id)) {
                // Get forum type
                $data['module']['forum_type'] = $moduleinstance->type;

                // Get discussions
                $discussions = $DB->get_records('forum_discussions', ['forum' => $moduleinstance->id], 'timemodified DESC', '*', 0, 5);

                if (!empty($discussions)) {
                    $data['module']['discussions'] = [];

                    foreach ($discussions as $discussion) {
                        // Get the most recent post for this discussion
                        $latestpost = $DB->get_record('forum_posts', ['id' => $discussion->firstpost], '*');

                        if ($latestpost) {
                            $postauthor = $DB->get_record('user', ['id' => $latestpost->userid]);

                            $discussionData = [
                                'id' => $discussion->id,
                                'name' => $discussion->name,
                                'timemodified' => $discussion->timemodified,
                                'post' => [
                                    'id' => $latestpost->id,
                                    'subject' => $latestpost->subject,
                                    'message' => strip_tags($latestpost->message),
                                    'created' => $latestpost->created,
                                    'author' => $postauthor ? \fullname($postauthor) : 'Unknown'
                                ]
                            ];

                            $data['module']['discussions'][] = $discussionData;
                        }
                    }
                }
            }
            // Add quiz specific information
            elseif ($moduleinfo->name === 'quiz' && !empty($moduleinstance->id)) {
                // Get basic quiz information
                $data['module']['quiz_info'] = [
                    'timeopen' => $moduleinstance->timeopen,
                    'timeclose' => $moduleinstance->timeclose,
                    'timelimit' => $moduleinstance->timelimit,
                    'attempts_allowed' => $moduleinstance->attempts,
                    'grade' => $moduleinstance->grade
                ];

                // Get questions using DB queries
                $quiz_question_instances = $DB->get_records('quiz_slots', ['quizid' => $moduleinstance->id], 'slot');

                if (!empty($quiz_question_instances)) {
                    $questionids = [];
                    foreach ($quiz_question_instances as $instance) {
                        $questionids[] = $instance->questionid;
                    }

                    if (!empty($questionids)) {
                        $questions = $DB->get_records_list('question', 'id', $questionids, '', 'id, name, questiontext, qtype');

                        if (!empty($questions)) {
                            $data['module']['questions'] = [];

                            foreach ($questions as $question) {
                                $data['module']['questions'][] = [
                                    'id' => $question->id,
                                    'name' => $question->name,
                                    'type' => $question->qtype,
                                    'text' => strip_tags($question->questiontext)
                                ];
                            }
                        }
                    }
                }
            }
        }

        // Send data to webhook.
        return self::send_to_webhook($data);
    }

    /**
     * Observer for the course_module_updated event.
     *
     * @param \core\event\course_module_updated $event The event
     * @return bool Success status
     */
    public static function course_module_updated(\core\event\course_module_updated $event)
    {
        global $DB, $USER;

        // Get the course data.
        $course = $DB->get_record('course', ['id' => $event->courseid], '*', \MUST_EXIST);

        // Get the course module data.
        $cm = $DB->get_record('course_modules', ['id' => $event->objectid], '*', \MUST_EXIST);
        $moduleinfo = $DB->get_record('modules', ['id' => $cm->module], 'name', \MUST_EXIST);

        // Get the module instance.
        $moduleinstance = $DB->get_record($moduleinfo->name, ['id' => $cm->instance]);

        // Get the user who updated the module.
        $user = $DB->get_record('user', ['id' => $event->userid]);

        // Prepare the data to send to the webhook.
        $data = [
            'event' => 'course_module_updated',
            'course' => [
                'id' => $course->id,
                'name' => $course->fullname,
                'summary' => strip_tags($course->summary),
                'visible' => (bool)$course->visible
            ],
            'module' => [
                'id' => $cm->id,
                'name' => $moduleinstance ? $moduleinstance->name : '',
                'modulename' => $moduleinfo->name,
                'visible' => (bool)$cm->visible
            ],
            'updated_by' => [
                'userid' => $event->userid,
                'fullname' => $user ? \fullname($user) : ''
            ],
            'timestamp' => $event->timecreated
        ];

        // Add module-specific data
        if ($moduleinstance) {
            // Get file information if this is a file-based module
            $fileInfo = self::get_module_file_info($moduleinfo, $moduleinstance, $cm);
            if ($fileInfo) {
                $data['module']['file'] = $fileInfo;
            }

            // Add page specific information
            if ($moduleinfo->name === 'page' && !empty($moduleinstance->id)) {
                // Include page content
                $data['module']['page_content'] = [
                    'title' => $moduleinstance->name,
                    'content' => strip_tags($moduleinstance->content),
                    'format' => $moduleinstance->contentformat,
                    'display_options' => [
                        'display_description' => (bool)$moduleinstance->displayoptions,
                        'print_heading' => (bool)$moduleinstance->printheading,
                        'print_last_modified' => (bool)$moduleinstance->printlastmodified
                    ]
                ];
            }
            // Add forum specific information
            elseif ($moduleinfo->name === 'forum' && !empty($moduleinstance->id)) {
                // Get forum type
                $data['module']['forum_type'] = $moduleinstance->type;

                // Get discussions
                $discussions = $DB->get_records('forum_discussions', ['forum' => $moduleinstance->id], 'timemodified DESC', '*', 0, 5);

                if (!empty($discussions)) {
                    $data['module']['discussions'] = [];

                    foreach ($discussions as $discussion) {
                        // Get the most recent post for this discussion
                        $latestpost = $DB->get_record('forum_posts', ['id' => $discussion->firstpost], '*');

                        if ($latestpost) {
                            $postauthor = $DB->get_record('user', ['id' => $latestpost->userid]);

                            $discussionData = [
                                'id' => $discussion->id,
                                'name' => $discussion->name,
                                'timemodified' => $discussion->timemodified,
                                'post' => [
                                    'id' => $latestpost->id,
                                    'subject' => $latestpost->subject,
                                    'message' => strip_tags($latestpost->message),
                                    'created' => $latestpost->created,
                                    'author' => $postauthor ? \fullname($postauthor) : 'Unknown'
                                ]
                            ];

                            $data['module']['discussions'][] = $discussionData;
                        }
                    }
                }
            }
            // Add quiz specific information
            elseif ($moduleinfo->name === 'quiz' && !empty($moduleinstance->id)) {
                // Get basic quiz information
                $data['module']['quiz_info'] = [
                    'timeopen' => $moduleinstance->timeopen,
                    'timeclose' => $moduleinstance->timeclose,
                    'timelimit' => $moduleinstance->timelimit,
                    'attempts_allowed' => $moduleinstance->attempts,
                    'grade' => $moduleinstance->grade
                ];

                // Get questions using DB queries
                $quiz_question_instances = $DB->get_records('quiz_slots', ['quizid' => $moduleinstance->id], 'slot');

                if (!empty($quiz_question_instances)) {
                    $questionids = [];
                    foreach ($quiz_question_instances as $instance) {
                        $questionids[] = $instance->questionid;
                    }

                    if (!empty($questionids)) {
                        $questions = $DB->get_records_list('question', 'id', $questionids, '', 'id, name, questiontext, qtype');

                        if (!empty($questions)) {
                            $data['module']['questions'] = [];

                            foreach ($questions as $question) {
                                $data['module']['questions'][] = [
                                    'id' => $question->id,
                                    'name' => $question->name,
                                    'type' => $question->qtype,
                                    'text' => strip_tags($question->questiontext)
                                ];
                            }
                        }
                    }
                }
            }
        }

        // Send data to webhook.
        return self::send_to_webhook($data);
    }

    /**
     * Observer for the course_module_deleted event.
     *
     * @param \core\event\course_module_deleted $event The event
     * @return bool Success status
     */
    public static function course_module_deleted(\core\event\course_module_deleted $event)
    {
        global $DB;

        // Get the course data.
        $course = $DB->get_record('course', ['id' => $event->courseid], '*', \MUST_EXIST);

        // Get the user who deleted the module.
        $user = $DB->get_record('user', ['id' => $event->userid]);

        // Prepare the data to send to the webhook.
        $data = [
            'event' => 'course_module_deleted',
            'course' => [
                'id' => $course->id,
                'name' => $course->fullname,
                'visible' => (bool)$course->visible
            ],
            'module' => [
                'id' => $event->objectid,
                'modulename' => $event->other['modulename'],
                'name' => $event->other['name']
            ],
            'updated_by' => [
                'userid' => $event->userid,
                'fullname' => $user ? \fullname($user) : ''
            ],
            'timestamp' => $event->timecreated
        ];

        // Send data to webhook.
        return self::send_to_webhook($data);
    }
}
