<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Installation script for local_eflow_aitutor
 *
 * @package    local_eflow_aitutor
 * @copyright  2025 Nawaz sharif<shasharif02@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

/**
 * Main installation function for the plugin
 */
function xmldb_local_eflow_aitutor_install() {
    global $DB, $CFG;

    // Enable web services
    enable_webservice();

    // Enable REST protocol
    enable_webservice_protocol('rest');

    // Create external service for eFlow Ai Tutor
    create_eflow_aitutor_service();
}

/**
 * Enable web services in Moodle
 */
function enable_webservice() {
    set_config('enablewebservices', 1);
}

/**
 * Enable a specific web service protocol
 *
 * @param string $protocol The protocol to enable ('rest', 'soap', etc.)
 */
function enable_webservice_protocol($protocol) {
    global $DB;

    // Check if the protocol is already enabled
    $enabled = get_config('core', 'webserviceprotocols');
    $protocols = empty($enabled) ? array() : explode(',', $enabled);

    if (!in_array($protocol, $protocols)) {
        $protocols[] = $protocol;
        set_config('webserviceprotocols', implode(',', $protocols));
    }

    // Enable the specific protocol
    $name = 'webservice'.$protocol.'enabled';
    set_config($name, 1);
}

/**
 * Create the external service for eFlow Ai Tutor
 */
function create_eflow_aitutor_service() {
    global $DB, $CFG;

    // Create a custom service for eFlow Ai Tutor if it doesn't exist
    $servicename = 'eFlow Ai Tutor';
    $service = $DB->get_record('external_services', array('name' => $servicename));

    if (empty($service)) {
        $service = new stdClass();
        $service->name = $servicename;
        $service->enabled = 1;
        $service->requiredcapability = 'local/eflow_aitutor:view';
        $service->restrictedusers = 0;
        $service->downloadfiles = 0;
        $service->uploadfiles = 0;
        $service->shortname = 'eflow_aitutor';
        $service->timecreated = time();

        $service->id = $DB->insert_record('external_services', $service);

        // Add the send_message function to the service
        $function = $DB->get_record('external_functions', array('name' => 'local_eflow_aitutor_send_message'));
        if (!empty($function)) {
            $servicefunction = new stdClass();
            $servicefunction->externalserviceid = $service->id;
            $servicefunction->functionname = 'local_eflow_aitutor_send_message';

            $DB->insert_record('external_services_functions', $servicefunction);
        }

        // Add additional functions as needed
        $additionalfunctions = [
            'core_course_get_contents',
            'core_user_get_course_user_profiles'
        ];

        foreach ($additionalfunctions as $functionname) {
            $function = $DB->get_record('external_functions', array('name' => $functionname));
            if (!empty($function)) {
                $servicefunction = new stdClass();
                $servicefunction->externalserviceid = $service->id;
                $servicefunction->functionname = $functionname;

                $DB->insert_record('external_services_functions', $servicefunction);
            }
        }
    }
}
