<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * eFlow Ai Tutor library functions.
 *
 * @package    local_eflow_aitutor
 * @copyright  2025 Nawaz sharif<shasharif02@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

// These are core Moodle classes, no need to use statements as they're in global scope
// Uncomment if you get linting errors in your IDE
use core\navigation\node;

/**
 * Extends the course settings navigation
 *
 * @param navigation_node $parentnode
 * @param stdClass $course
 * @param \context_course $context
 */
function local_eflow_aitutor_extend_navigation_course($parentnode, $course, $context)
{
    if (has_capability('local/eflow_aitutor:manage', $context)) {
        $url = new \moodle_url('/local/eflow_aitutor/course_settings.php', array('id' => $course->id));
        $parentnode->add(
            get_string('eflow_aitutorsettings', 'local_eflow_aitutor'),
            $url,
            navigation_node::TYPE_SETTING,
            null,
            'eflow_aitutorsettings',
            new \pix_icon('i/settings', '')
        );
    }
}

/**
 * Serves the logo file.
 *
 * @param stdClass $course The course object
 * @param stdClass $cm The course module object
 * @param stdClass $context The context
 * @param string $filearea The name of the file area
 * @param array $args Extra arguments
 * @param bool $forcedownload Whether or not force download
 * @param array $options Additional options affecting the file serving
 * @return bool
 */
function local_eflow_aitutor_pluginfile($course, $cm, $context, $filearea, $args, $forcedownload, array $options = [])
{
    if ($context->contextlevel != CONTEXT_SYSTEM) {
        return false;
    }

    if ($filearea !== 'logo' && $filearea !== 'ai_avatar') {
        return false;
    }

    // Logo should be accessible without login
    // require_login();

    $itemid = array_shift($args);
    $filename = array_shift($args);
    $filepath = '/';

    $fs = get_file_storage();
    $file = $fs->get_file($context->id, 'local_eflow_aitutor', $filearea, $itemid, $filepath, $filename);
    if (!$file) {
        return false;
    }

    send_stored_file($file, 86400, 0, $forcedownload, $options);

    return true;
}

/**
 * Get API authentication token
 *
 * @param string $username API username
 * @param string $password API password
 * @param string $apibaseurl Base URL for the API
 * @return string|bool The authentication token or false on failure
 */
function local_eflow_aitutor_get_api_token($username, $password, $apibaseurl)
{
    // URL for authentication
    $authendpoint = $apibaseurl . "/login";

    // Authentication data
    $authdata = [
        'username' => $username,
        'password' => $password
    ];

    // Use Moodle's curl class
    $curl = new \curl();
    $curl->setHeader(['Content-Type: application/json']);
    $curl->setHeader(['User-Agent: moodle/eflow_aitutor']);

    // Execute the request
    $response = $curl->post($authendpoint, json_encode($authdata));
    $info = $curl->get_info();
    $httpcode = $info['http_code'];
    $err = $curl->get_errno();

    // Check for errors
    if ($err) {
        debugging('eFlow AI Tutor: Authentication cURL Error: ' . $curl->error, DEBUG_DEVELOPER);
        return false;
    }

    // Check HTTP status code
    if ($httpcode != 200) {
        debugging('eFlow AI Tutor: Authentication Error: ' . $response, DEBUG_DEVELOPER);
        return false;
    }

    // Parse the response
    $result = json_decode($response, true);

    // Check if we received a token
    if (empty($result) || !isset($result['token'])) {
        debugging('eFlow AI Tutor: No token in authentication response', DEBUG_DEVELOPER);
        return false;
    }

    return $result['token'];
}

/**
 * Inject the floating chat into Moodle pages
 *
 * @param moodle_page $page The page object
 */
function local_eflow_aitutor_before_footer() {
    global $PAGE, $USER, $COURSE, $CFG, $OUTPUT;

    // Check if the plugin is enabled globally
    $enabled = get_config('local_eflow_aitutor', 'enabled');
    if (empty($enabled)) {
        return;
    }

    // Don't show the chat if not enrolled in the course
    $context = context_course::instance($COURSE->id);
    if (!is_enrolled($context, $USER)) {
        return;
    }

    // Check if the plugin is enabled for this course
    $courseenabled = get_config('local_eflow_aitutor', "eflow_aitutor-{$COURSE->id}");
    if (isset($courseenabled) && $courseenabled === '0') {
        return;
    }

    // Check if the user has permission to view the tutor
    if (!has_capability('local/eflow_aitutor:view', $context)) {
        return;
    }

    // Get float option from URL first, then default to with_option
    $floatOption = optional_param('float_option', 'with_option', PARAM_ALPHA);

    // Check if there's a global default set
    $globalDefault = get_config('local_eflow_aitutor', 'default_float_option');
    if (!empty($globalDefault)) {
        // Override if the global default is set
        $floatOption = $globalDefault;
    }

    // Validate that it's one of the allowed values
    if ($floatOption !== 'with_option' && $floatOption !== 'without_option') {
        $floatOption = 'with_option';
    }

    // Initialize chat with the float option parameter
    $PAGE->requires->js_call_amd('local_eflow_aitutor/chat', 'init', [
        [
            'courseId' => $COURSE->id,
            'userId' => $USER->id,
            'floatOption' => $floatOption
        ]
    ]);

    // ... rest of the function
}

/**
 * Get course-specific settings for the eFlow AI Tutor.
 *
 * @param stdClass $course The course object
 * @return stdClass Settings object
 */
function local_eflow_aitutor_get_course_settings($course) {
    $settings = new stdClass();
    $settings->courseid = $course->id;

    // Get the enabled status for this course
    $enabled = get_config('local_eflow_aitutor', "eflow_aitutor-{$course->id}");
    $settings->enabled = isset($enabled) ? $enabled : 1; // Default to enabled if not set

    return $settings;
}

/**
 * Save course-specific settings for the eFlow AI Tutor.
 *
 * @param stdClass $data Form data
 * @return bool Success status
 */
function local_eflow_aitutor_save_course_settings($data) {
    // Save the enabled status for this course
    set_config("eflow_aitutor-{$data->courseid}", $data->enabled, 'local_eflow_aitutor');

    return true;
}
