<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace filter_bsmigrate;

/**
 * Bootstrap 4 to Bootstrap 5 class translator
 *
 * @package    filter_bsmigrate
 * @copyright  2025 Bas Brands <bas@sonsbeekmedia.nl>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class bs_translator {

    /**
     * @var array Bootstrap 4 to Bootstrap 5 class mappings
     */
    private static $classmappings = [
        // Margin and padding direction changes.
        'ml-0' => 'ms-0', 'ml-1' => 'ms-1', 'ml-2' => 'ms-2', 'ml-3' => 'ms-3', 'ml-4' => 'ms-4', 'ml-5' => 'ms-5',
        'mr-0' => 'me-0', 'mr-1' => 'me-1', 'mr-2' => 'me-2', 'mr-3' => 'me-3', 'mr-4' => 'me-4', 'mr-5' => 'me-5',
        'pl-0' => 'ps-0', 'pl-1' => 'ps-1', 'pl-2' => 'ps-2', 'pl-3' => 'ps-3', 'pl-4' => 'ps-4', 'pl-5' => 'ps-5',
        'pr-0' => 'pe-0', 'pr-1' => 'pe-1', 'pr-2' => 'pe-2', 'pr-3' => 'pe-3', 'pr-4' => 'pe-4', 'pr-5' => 'pe-5',

        // Auto margins.
        'ml-auto' => 'ms-auto',
        'mr-auto' => 'me-auto',

        // Responsive margin and padding.
        'ml-sm-0' => 'ms-sm-0', 'ml-sm-1' => 'ms-sm-1', 'ml-sm-2' => 'ms-sm-2', 'ml-sm-3' => 'ms-sm-3',
        'ml-sm-4' => 'ms-sm-4', 'ml-sm-5' => 'ms-sm-5',
        'mr-sm-0' => 'me-sm-0', 'mr-sm-1' => 'me-sm-1', 'mr-sm-2' => 'me-sm-2', 'mr-sm-3' => 'me-sm-3',
        'mr-sm-4' => 'me-sm-4', 'mr-sm-5' => 'me-sm-5',
        'pl-sm-0' => 'ps-sm-0', 'pl-sm-1' => 'ps-sm-1', 'pl-sm-2' => 'ps-sm-2', 'pl-sm-3' => 'ps-sm-3',
        'pl-sm-4' => 'ps-sm-4', 'pl-sm-5' => 'ps-sm-5',
        'pr-sm-0' => 'pe-sm-0', 'pr-sm-1' => 'pe-sm-1', 'pr-sm-2' => 'pe-sm-2', 'pr-sm-3' => 'pe-sm-3',
        'pr-sm-4' => 'pe-sm-4', 'pr-sm-5' => 'pe-sm-5',

        'ml-md-0' => 'ms-md-0', 'ml-md-1' => 'ms-md-1', 'ml-md-2' => 'ms-md-2', 'ml-md-3' => 'ms-md-3',
        'ml-md-4' => 'ms-md-4', 'ml-md-5' => 'ms-md-5',
        'mr-md-0' => 'me-md-0', 'mr-md-1' => 'me-md-1', 'mr-md-2' => 'me-md-2', 'mr-md-3' => 'me-md-3',
        'mr-md-4' => 'me-md-4', 'mr-md-5' => 'me-md-5',
        'pl-md-0' => 'ps-md-0', 'pl-md-1' => 'ps-md-1', 'pl-md-2' => 'ps-md-2', 'pl-md-3' => 'ps-md-3',
        'pl-md-4' => 'ps-md-4', 'pl-md-5' => 'ps-md-5',
        'pr-md-0' => 'pe-md-0', 'pr-md-1' => 'pe-md-1', 'pr-md-2' => 'pe-md-2', 'pr-md-3' => 'pe-md-3',
        'pr-md-4' => 'pe-md-4', 'pr-md-5' => 'pe-md-5',

        'ml-lg-0' => 'ms-lg-0', 'ml-lg-1' => 'ms-lg-1', 'ml-lg-2' => 'ms-lg-2', 'ml-lg-3' => 'ms-lg-3',
        'ml-lg-4' => 'ms-lg-4', 'ml-lg-5' => 'ms-lg-5',
        'mr-lg-0' => 'me-lg-0', 'mr-lg-1' => 'me-lg-1', 'mr-lg-2' => 'me-lg-2', 'mr-lg-3' => 'me-lg-3',
        'mr-lg-4' => 'me-lg-4', 'mr-lg-5' => 'me-lg-5',
        'pl-lg-0' => 'ps-lg-0', 'pl-lg-1' => 'ps-lg-1', 'pl-lg-2' => 'ps-lg-2', 'pl-lg-3' => 'ps-lg-3',
        'pl-lg-4' => 'ps-lg-4', 'pl-lg-5' => 'ps-lg-5',
        'pr-lg-0' => 'pe-lg-0', 'pr-lg-1' => 'pe-lg-1', 'pr-lg-2' => 'pe-lg-2', 'pr-lg-3' => 'pe-lg-3',
        'pr-lg-4' => 'pe-lg-4', 'pr-lg-5' => 'pe-lg-5',

        'ml-xl-0' => 'ms-xl-0', 'ml-xl-1' => 'ms-xl-1', 'ml-xl-2' => 'ms-xl-2', 'ml-xl-3' => 'ms-xl-3',
        'ml-xl-4' => 'ms-xl-4', 'ml-xl-5' => 'ms-xl-5',
        'mr-xl-0' => 'me-xl-0', 'mr-xl-1' => 'me-xl-1', 'mr-xl-2' => 'me-xl-2', 'mr-xl-3' => 'me-xl-3',
        'mr-xl-4' => 'me-xl-4', 'mr-xl-5' => 'me-xl-5',
        'pl-xl-0' => 'ps-xl-0', 'pl-xl-1' => 'ps-xl-1', 'pl-xl-2' => 'ps-xl-2', 'pl-xl-3' => 'ps-xl-3',
        'pl-xl-4' => 'ps-xl-4', 'pl-xl-5' => 'ps-xl-5',
        'pr-xl-0' => 'pe-xl-0', 'pr-xl-1' => 'pe-xl-1', 'pr-xl-2' => 'pe-xl-2', 'pr-xl-3' => 'pe-xl-3',
        'pr-xl-4' => 'pe-xl-4', 'pr-xl-5' => 'pe-xl-5',

        // Auto margins responsive.
        'ml-sm-auto' => 'ms-sm-auto', 'mr-sm-auto' => 'me-sm-auto',
        'ml-md-auto' => 'ms-md-auto', 'mr-md-auto' => 'me-md-auto',
        'ml-lg-auto' => 'ms-lg-auto', 'mr-lg-auto' => 'me-lg-auto',
        'ml-xl-auto' => 'ms-xl-auto', 'mr-xl-auto' => 'me-xl-auto',

        // Float utilities.
        'float-left' => 'float-start',
        'float-right' => 'float-end',
        'float-sm-left' => 'float-sm-start',
        'float-sm-right' => 'float-sm-end',
        'float-md-left' => 'float-md-start',
        'float-md-right' => 'float-md-end',
        'float-lg-left' => 'float-lg-start',
        'float-lg-right' => 'float-lg-end',
        'float-xl-left' => 'float-xl-start',
        'float-xl-right' => 'float-xl-end',

        // Text alignment.
        'text-left' => 'text-start',
        'text-right' => 'text-end',
        'text-sm-left' => 'text-sm-start',
        'text-sm-right' => 'text-sm-end',
        'text-md-left' => 'text-md-start',
        'text-md-right' => 'text-md-end',
        'text-lg-left' => 'text-lg-start',
        'text-lg-right' => 'text-lg-end',
        'text-xl-left' => 'text-xl-start',
        'text-xl-right' => 'text-xl-end',

        // Border utilities.
        'border-left' => 'border-start',
        'border-right' => 'border-end',
        'border-left-0' => 'border-start-0',
        'border-right-0' => 'border-end-0',
        'rounded-left' => 'rounded-start',
        'rounded-right' => 'rounded-end',

        // Grid system changes.
        'no-gutters' => 'g-0',

        // Close button.
        'close' => 'btn-close',

        // Form controls.
        'form-control-file' => 'form-control',
        'form-control-range' => 'form-range',

        // Input groups.
        'input-group-prepend' => 'input-group-text',
        'input-group-append' => 'input-group-text',

        // Custom forms (now standard).
        'custom-control' => 'form-check',
        'custom-control-input' => 'form-check-input',
        'custom-control-label' => 'form-check-label',
        'custom-checkbox' => 'form-check',
        'custom-radio' => 'form-check',
        'custom-switch' => 'form-check form-switch',
        'custom-select' => 'form-select',
        'custom-file' => 'form-control',
        'custom-file-input' => 'form-control',
        'custom-file-label' => 'form-label',
        'custom-range' => 'form-range',

        // Media object (removed in BS5).
        'media' => 'd-flex',
        'media-object' => 'flex-shrink-0',
        'media-body' => 'flex-grow-1 ms-3',

        // Card deck (removed in BS5).
        'card-deck' => 'row row-cols-1 row-cols-md-3 g-4',
        'card-columns' => 'row row-cols-1 row-cols-md-2 row-cols-xl-3',

        // Jumbotron (removed in BS5).
        'jumbotron' => 'bg-light p-5 rounded-3',
        'jumbotron-fluid' => 'bg-light p-5',

        // Font weight utilities (renamed in BS5).
        'font-weight-bold' => 'fw-bold',
        'font-weight-bolder' => 'fw-bolder',
        'font-weight-normal' => 'fw-normal',
        'font-weight-light' => 'fw-light',
        'font-weight-lighter' => 'fw-lighter',

        // Button block (removed in BS5, replaced with grid/flex utilities).
        'btn-block' => 'd-grid',

        // XXL breakpoint mappings (new in BS5).
        'ml-xxl-0' => 'ms-xxl-0', 'ml-xxl-1' => 'ms-xxl-1', 'ml-xxl-2' => 'ms-xxl-2', 'ml-xxl-3' => 'ms-xxl-3',
        'ml-xxl-4' => 'ms-xxl-4', 'ml-xxl-5' => 'ms-xxl-5',
        'mr-xxl-0' => 'me-xxl-0', 'mr-xxl-1' => 'me-xxl-1', 'mr-xxl-2' => 'me-xxl-2', 'mr-xxl-3' => 'me-xxl-3',
        'mr-xxl-4' => 'me-xxl-4', 'mr-xxl-5' => 'me-xxl-5',
        'pl-xxl-0' => 'ps-xxl-0', 'pl-xxl-1' => 'ps-xxl-1', 'pl-xxl-2' => 'ps-xxl-2', 'pl-xxl-3' => 'ps-xxl-3',
        'pl-xxl-4' => 'ps-xxl-4', 'pl-xxl-5' => 'ps-xxl-5',
        'pr-xxl-0' => 'pe-xxl-0', 'pr-xxl-1' => 'pe-xxl-1', 'pr-xxl-2' => 'pe-xxl-2', 'pr-xxl-3' => 'pe-xxl-3',
        'pr-xxl-4' => 'pe-xxl-4', 'pr-xxl-5' => 'pe-xxl-5',

        // XXL auto margins.
        'ml-xxl-auto' => 'ms-xxl-auto',
        'mr-xxl-auto' => 'me-xxl-auto',

        // XXL float utilities.
        'float-xxl-left' => 'float-xxl-start',
        'float-xxl-right' => 'float-xxl-end',

        // XXL text alignment.
        'text-xxl-left' => 'text-xxl-start',
        'text-xxl-right' => 'text-xxl-end',

        // Screen reader utilities (renamed).
        'sr-only' => 'visually-hidden',
        'sr-only-focusable' => 'visually-hidden-focusable',

        // Form utilities (removed/changed in BS5).
        'form-row' => 'row g-3',
        'form-inline' => 'd-flex align-items-center',

        // Badge classes (contextual colors changed to background utilities).
        'badge-primary' => 'bg-primary',
        'badge-secondary' => 'bg-secondary',
        'badge-success' => 'bg-success',
        'badge-danger' => 'bg-danger',
        'badge-warning' => 'bg-warning text-dark',
        'badge-info' => 'bg-info text-dark',
        'badge-light' => 'bg-light text-dark',
        'badge-dark' => 'bg-dark',

        // Dropdown menu improvements.
        'dropdown-menu-right' => 'dropdown-menu-end',
        'dropdown-menu-left' => 'dropdown-menu-start',
        'dropright' => 'dropend',
        'dropleft' => 'dropstart',

        // Badge pill (removed in BS5).
        'badge-pill' => 'rounded-pill',

        // Font italic utility (renamed in BS5).
        'font-italic' => 'fst-italic',

        // Rounded size utilities (renamed in BS5).
        'rounded-sm' => 'rounded-1',
        'rounded-lg' => 'rounded-3',
    ];

    /**
     * Get the Bootstrap 5 equivalent of a Bootstrap 4 class
     *
     * @param string $class The Bootstrap 4 class name
     * @return string The Bootstrap 5 class name, or original if no mapping exists
     */
    public static function translate_class(string $class): string {
        return self::$classmappings[$class] ?? $class;
    }

    /**
     * Get all class mappings
     *
     * @return array All Bootstrap 4 to Bootstrap 5 class mappings
     */
    public static function get_all_mappings(): array {
        return self::$classmappings;
    }

    /**
     * Check if a class needs translation.
     *
     * @param string $class The class name to check
     * @return bool True if the class has a Bootstrap 5 equivalent
     */
    public static function needs_translation(string $class): bool {
        return isset(self::$classmappings[$class]);
    }
}
