<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace filter_bsmigrate;

/**
 * bsmigrate filter - Bootstrap 4 to Bootstrap 5 class migration
 *
 * This filter automatically translates Bootstrap 4 CSS classes to their Bootstrap 5 equivalents
 * in HTML content. It processes class attributes in HTML tags and replaces deprecated classes
 * with their modern counterparts.
 *
 * Documentation: {@link https://moodledev.io/docs/apis/plugintypes/filter}
 *
 * @package    filter_bsmigrate
 * @copyright  2025 Bas Brands <bas@sonsbeekmedia.nl>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class text_filter extends \core_filters\text_filter {

    /**
     * Filter text to translate Bootstrap 4 classes to Bootstrap 5
     *
     * @param string $text some HTML content to process.
     * @param array $options options passed to the filters
     * @return string the HTML content after the filtering has been applied.
     */
    public function filter($text, array $options = []) {
        // Skip processing if text is empty or doesn't contain HTML.
        if (empty($text) || (stripos($text, 'class=') === false && stripos($text, 'data-') === false)) {
            return $text;
        }

        // First process class attributes.
        $text = $this->process_class_attributes($text);

        // Then process data attributes.
        $text = $this->process_data_attributes($text);

        return $text;
    }

    /**
     * Process class attributes to translate Bootstrap 4 classes to Bootstrap 5
     *
     * @param string $text HTML content to process
     * @return string HTML content with translated class attributes
     */
    private function process_class_attributes(string $text): string {
        if (stripos($text, 'class=') === false) {
            return $text;
        }

        // Use regex to find all class attributes in HTML tags.
        // Matches: class="..." or class='...' (preserves case of attribute name).
        $pattern = '/(class)\s*=\s*(["\'])(.*?)\2/i';

        return preg_replace_callback($pattern, function($matches) {
            $attributename = $matches[1]; // Preserve original case of 'class'.
            $quote = $matches[2]; // Preserve original quote type (" or ').
            $classes = $matches[3]; // The class list.

            // Split classes by whitespace and filter empty values.
            $classarray = array_filter(preg_split('/\s+/', trim($classes)));

            // Translate each class.
            $translatedclasses = [];
            foreach ($classarray as $class) {
                $translatedclasses[] = bs_translator::translate_class($class);
            }

            // Flatten and deduplicate classes (some translations add multiple classes).
            $allclasses = [];
            foreach ($translatedclasses as $translated) {
                $multipleclasses = preg_split('/\s+/', trim($translated));
                foreach ($multipleclasses as $singleclass) {
                    if (!empty($singleclass) && !in_array($singleclass, $allclasses)) {
                        $allclasses[] = $singleclass;
                    }
                }
            }

            // Rebuild the class attribute.
            $newclasses = implode(' ', $allclasses);
            return "{$attributename}={$quote}{$newclasses}{$quote}";
        }, $text);
    }

    /**
     * Process data attributes to translate Bootstrap 4 data attributes to Bootstrap 5
     *
     * @param string $text HTML content to process
     * @return string HTML content with translated data attributes
     */
    private function process_data_attributes(string $text): string {
        if (stripos($text, 'data-') === false) {
            return $text;
        }

        // Bootstrap 4 to 5 data attribute mappings.
        $datamappings = [
            'data-toggle' => 'data-bs-toggle',
            'data-target' => 'data-bs-target',
            'data-dismiss' => 'data-bs-dismiss',
            'data-slide' => 'data-bs-slide',
            'data-slide-to' => 'data-bs-slide-to',
            'data-parent' => 'data-bs-parent',
            'data-spy' => 'data-bs-spy',
            'data-offset' => 'data-bs-offset',
            'data-interval' => 'data-bs-interval',
            'data-pause' => 'data-bs-pause',
            'data-wrap' => 'data-bs-wrap',
            'data-keyboard' => 'data-bs-keyboard',
            'data-backdrop' => 'data-bs-backdrop',
            'data-placement' => 'data-bs-placement',
            'data-trigger' => 'data-bs-trigger',
            'data-delay' => 'data-bs-delay',
            'data-html' => 'data-bs-html',
            'data-selector' => 'data-bs-selector',
            'data-template' => 'data-bs-template',
            'data-title' => 'data-bs-title',
            'data-content' => 'data-bs-content',
            'data-boundary' => 'data-bs-boundary',
            'data-animation' => 'data-bs-animation',
            'data-container' => 'data-bs-container',
            'data-sanitize' => 'data-bs-sanitize',
            'data-whiteList' => 'data-bs-allowList',
            'data-ride' => 'data-bs-ride',
        ];

        // Process each data attribute mapping.
        foreach ($datamappings as $oldattr => $newattr) {
            $pattern = '/\b' . preg_quote($oldattr, '/') . '\b/i';
            $text = preg_replace($pattern, $newattr, $text);
        }

        return $text;
    }

    /**
     * Get statistics about translations performed
     *
     * @param string $text The text to analyze
     * @return array Statistics about potential translations
     */
    public function get_translation_stats(string $text): array {
        $stats = [
            'total_classes' => 0,
            'translated_classes' => 0,
            'translations' => [],
        ];

        if (empty($text) || strpos($text, 'class=') === false) {
            return $stats;
        }

        $pattern = '/class\s*=\s*(["\'])(.*?)\1/i';

        preg_replace_callback($pattern, function($matches) use (&$stats) {
            $classes = $matches[2];
            $classarray = array_filter(preg_split('/\s+/', trim($classes)));

            foreach ($classarray as $class) {
                $stats['total_classes']++;
                if (bs_translator::needs_translation($class)) {
                    $stats['translated_classes']++;
                    $translated = bs_translator::translate_class($class);
                    $stats['translations'][$class] = $translated;
                }
            }

            return $matches[0]; // Return unchanged for analysis.
        }, $text);

        return $stats;
    }
}
