<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace filter_bsmigrate;

/**
 * Unit tests for bs_translator class.
 *
 * Tests the Bootstrap 4 to Bootstrap 5 class translation mappings.
 *
 * @package    filter_bsmigrate
 * @copyright  2025 Bas Brands <bas@sonsbeekmedia.nl>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @covers     \filter_bsmigrate\bs_translator
 */
final class bs_translator_test extends \basic_testcase {

    /**
     * Test basic margin translations.
     */
    public function test_margin_translations(): void {
        $mappings = [
            'ml-0' => 'ms-0', 'ml-1' => 'ms-1', 'ml-2' => 'ms-2', 'ml-3' => 'ms-3', 'ml-4' => 'ms-4', 'ml-5' => 'ms-5',
            'mr-0' => 'me-0', 'mr-1' => 'me-1', 'mr-2' => 'me-2', 'mr-3' => 'me-3', 'mr-4' => 'me-4', 'mr-5' => 'me-5',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test basic padding translations.
     */
    public function test_padding_translations(): void {
        $mappings = [
            'pl-0' => 'ps-0', 'pl-1' => 'ps-1', 'pl-2' => 'ps-2', 'pl-3' => 'ps-3', 'pl-4' => 'ps-4', 'pl-5' => 'ps-5',
            'pr-0' => 'pe-0', 'pr-1' => 'pe-1', 'pr-2' => 'pe-2', 'pr-3' => 'pe-3', 'pr-4' => 'pe-4', 'pr-5' => 'pe-5',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test auto margin translations.
     */
    public function test_auto_margin_translations(): void {
        $mappings = [
            'ml-auto' => 'ms-auto',
            'mr-auto' => 'me-auto',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test responsive margin translations.
     */
    public function test_responsive_margin_translations(): void {
        $mappings = [
            'ml-sm-0' => 'ms-sm-0', 'ml-sm-auto' => 'ms-sm-auto',
            'mr-md-3' => 'me-md-3', 'mr-md-auto' => 'me-md-auto',
            'ml-lg-5' => 'ms-lg-5', 'ml-lg-auto' => 'ms-lg-auto',
            'mr-xl-2' => 'me-xl-2', 'mr-xl-auto' => 'me-xl-auto',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test responsive padding translations.
     */
    public function test_responsive_padding_translations(): void {
        $mappings = [
            'pl-sm-1' => 'ps-sm-1', 'pr-sm-2' => 'pe-sm-2',
            'pl-md-3' => 'ps-md-3', 'pr-md-4' => 'pe-md-4',
            'pl-lg-0' => 'ps-lg-0', 'pr-lg-5' => 'pe-lg-5',
            'pl-xl-2' => 'ps-xl-2', 'pr-xl-1' => 'pe-xl-1',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test float utility translations.
     */
    public function test_float_translations(): void {
        $mappings = [
            'float-left' => 'float-start',
            'float-right' => 'float-end',
            'float-sm-left' => 'float-sm-start',
            'float-sm-right' => 'float-sm-end',
            'float-md-left' => 'float-md-start',
            'float-md-right' => 'float-md-end',
            'float-lg-left' => 'float-lg-start',
            'float-lg-right' => 'float-lg-end',
            'float-xl-left' => 'float-xl-start',
            'float-xl-right' => 'float-xl-end',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test text alignment translations.
     */
    public function test_text_alignment_translations(): void {
        $mappings = [
            'text-left' => 'text-start',
            'text-right' => 'text-end',
            'text-sm-left' => 'text-sm-start',
            'text-sm-right' => 'text-sm-end',
            'text-md-left' => 'text-md-start',
            'text-md-right' => 'text-md-end',
            'text-lg-left' => 'text-lg-start',
            'text-lg-right' => 'text-lg-end',
            'text-xl-left' => 'text-xl-start',
            'text-xl-right' => 'text-xl-end',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test border utility translations.
     */
    public function test_border_translations(): void {
        $mappings = [
            'border-left' => 'border-start',
            'border-right' => 'border-end',
            'border-left-0' => 'border-start-0',
            'border-right-0' => 'border-end-0',
            'rounded-left' => 'rounded-start',
            'rounded-right' => 'rounded-end',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test grid system translations.
     */
    public function test_grid_translations(): void {
        $this->assertEquals('g-0', bs_translator::translate_class('no-gutters'));
        $this->assertTrue(bs_translator::needs_translation('no-gutters'));
    }

    /**
     * Test form control translations.
     */
    public function test_form_control_translations(): void {
        $mappings = [
            'form-control-file' => 'form-control',
            'form-control-range' => 'form-range',
            'custom-control' => 'form-check',
            'custom-control-input' => 'form-check-input',
            'custom-control-label' => 'form-check-label',
            'custom-checkbox' => 'form-check',
            'custom-radio' => 'form-check',
            'custom-switch' => 'form-check form-switch',
            'custom-select' => 'form-select',
            'custom-file' => 'form-control',
            'custom-file-input' => 'form-control',
            'custom-file-label' => 'form-label',
            'custom-range' => 'form-range',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test input group translations.
     */
    public function test_input_group_translations(): void {
        $mappings = [
            'input-group-prepend' => 'input-group-text',
            'input-group-append' => 'input-group-text',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test media object translations.
     */
    public function test_media_object_translations(): void {
        $mappings = [
            'media' => 'd-flex',
            'media-object' => 'flex-shrink-0',
            'media-body' => 'flex-grow-1 ms-3',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test card layout translations.
     */
    public function test_card_layout_translations(): void {
        $mappings = [
            'card-deck' => 'row row-cols-1 row-cols-md-3 g-4',
            'card-columns' => 'row row-cols-1 row-cols-md-2 row-cols-xl-3',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test jumbotron translations.
     */
    public function test_jumbotron_translations(): void {
        $mappings = [
            'jumbotron' => 'bg-light p-5 rounded-3',
            'jumbotron-fluid' => 'bg-light p-5',
        ];

        foreach ($mappings as $bs4 => $bs5) {
            $this->assertEquals($bs5, bs_translator::translate_class($bs4));
            $this->assertTrue(bs_translator::needs_translation($bs4));
        }
    }

    /**
     * Test close button translation.
     */
    public function test_close_button_translation(): void {
        $this->assertEquals('btn-close', bs_translator::translate_class('close'));
        $this->assertTrue(bs_translator::needs_translation('close'));
    }

    /**
     * Test non-Bootstrap classes are not translated.
     */
    public function test_non_bootstrap_classes_unchanged(): void {
        $nonbootstrapclasses = [
            'my-custom-class',
            'another-component',
            'btn-primary',
            'container',
            'row',
            'col-md-6',
            'text-center', // This is preserved in BS5.
            'd-none',
            'some-random-class',
        ];

        foreach ($nonbootstrapclasses as $class) {
            $this->assertEquals($class, bs_translator::translate_class($class));
            $this->assertFalse(bs_translator::needs_translation($class));
        }
    }

    /**
     * Test get_all_mappings returns array.
     */
    public function test_get_all_mappings(): void {
        $mappings = bs_translator::get_all_mappings();

        $this->assertIsArray($mappings);
        $this->assertNotEmpty($mappings);

        // Test a few known mappings.
        $this->assertArrayHasKey('ml-1', $mappings);
        $this->assertEquals('ms-1', $mappings['ml-1']);

        $this->assertArrayHasKey('float-left', $mappings);
        $this->assertEquals('float-start', $mappings['float-left']);

        $this->assertArrayHasKey('custom-control', $mappings);
        $this->assertEquals('form-check', $mappings['custom-control']);
    }

    /**
     * Test edge cases and special characters.
     */
    public function test_edge_cases(): void {
        // Empty string.
        $this->assertEquals('', bs_translator::translate_class(''));
        $this->assertFalse(bs_translator::needs_translation(''));

        // Class with numbers and hyphens that looks similar but isn't Bootstrap.
        $this->assertEquals('ml-custom-1', bs_translator::translate_class('ml-custom-1'));
        $this->assertFalse(bs_translator::needs_translation('ml-custom-1'));
    }

    /**
     * Test that all mappings are consistent.
     */
    public function test_mapping_consistency(): void {
        $mappings = bs_translator::get_all_mappings();

        foreach ($mappings as $bs4 => $bs5) {
            // Each BS4 class should need translation.
            $this->assertTrue(bs_translator::needs_translation($bs4),
                "Class '{$bs4}' should need translation");

            // Each BS4 class should translate to its BS5 equivalent.
            $this->assertEquals($bs5, bs_translator::translate_class($bs4),
                "Class '{$bs4}' should translate to '{$bs5}'");

            // BS5 classes should not need further translation.
            $this->assertFalse(bs_translator::needs_translation($bs5),
                "BS5 class '{$bs5}' should not need translation");
        }
    }
}
