<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace filter_bsmigrate;

use core\context\system as context_system;

/**
 * Unit tests for filter_bsmigrate text filter.
 *
 * Tests the Bootstrap 4 to Bootstrap 5 class migration functionality.
 *
 * @package    filter_bsmigrate
 * @copyright  2025 Bas Brands <bas@sonsbeekmedia.nl>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @covers     \filter_bsmigrate\text_filter
 */
final class text_filter_test extends \basic_testcase {

    /** @var text_filter The filter to test */
    protected text_filter $filter;

    protected function setUp(): void {
        parent::setUp();
        $this->filter = new text_filter(context_system::instance(), []);
    }

    /**
     * Test basic margin and padding translations.
     */
    public function test_basic_margin_padding(): void {
        $input = '<div class="ml-1 mr-2 pl-3 pr-4">Content</div>';
        $expected = '<div class="ms-1 me-2 ps-3 pe-4">Content</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test auto margin translations.
     */
    public function test_auto_margins(): void {
        $input = '<div class="ml-auto mr-auto">Centered content</div>';
        $expected = '<div class="ms-auto me-auto">Centered content</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test responsive class translations.
     */
    public function test_responsive_classes(): void {
        $input = '<div class="ml-sm-2 mr-md-3 text-lg-left">Responsive</div>';
        $expected = '<div class="ms-sm-2 me-md-3 text-lg-start">Responsive</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test float utility translations.
     */
    public function test_float_utilities(): void {
        $input = '<div class="float-left">Left</div><div class="float-right">Right</div>';
        $expected = '<div class="float-start">Left</div><div class="float-end">Right</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test text alignment translations.
     */
    public function test_text_alignment(): void {
        $input = '<p class="text-left">Left text</p><p class="text-right">Right text</p>';
        $expected = '<p class="text-start">Left text</p><p class="text-end">Right text</p>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test border utility translations.
     */
    public function test_border_utilities(): void {
        $input = '<div class="border-left border-right rounded-left">Borders</div>';
        $expected = '<div class="border-start border-end rounded-start">Borders</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test grid system translations.
     */
    public function test_grid_system(): void {
        $input = '<div class="row no-gutters"><div class="col-6">Column</div></div>';
        $expected = '<div class="row g-0"><div class="col-6">Column</div></div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test custom form control translations.
     */
    public function test_custom_forms(): void {
        $input = '<div class="custom-control custom-switch"><input class="custom-control-input" type="checkbox">' .
                '<label class="custom-control-label">Switch</label></div>';
        $expected = '<div class="form-check form-switch"><input class="form-check-input" type="checkbox">' .
                '<label class="form-check-label">Switch</label></div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test input group translations.
     */
    public function test_input_groups(): void {
        $input = '<div class="input-group"><div class="input-group-prepend"><span class="input-group-text">@</span></div></div>';
        $expected = '<div class="input-group"><div class="input-group-text"><span class="input-group-text">@</span></div></div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test media object translations.
     */
    public function test_media_object(): void {
        $input = '<div class="media"><img class="media-object" src="test.jpg">' .
                '<div class="media-body">Content</div></div>';
        $expected = '<div class="d-flex"><img class="flex-shrink-0" src="test.jpg">' .
                '<div class="flex-grow-1 ms-3">Content</div></div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test card deck translations.
     */
    public function test_card_deck(): void {
        $input = '<div class="card-deck"><div class="card">Card 1</div><div class="card">Card 2</div></div>';
        $expected = '<div class="row row-cols-1 row-cols-md-3 g-4">' .
                '<div class="card">Card 1</div><div class="card">Card 2</div></div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test jumbotron translations.
     */
    public function test_jumbotron(): void {
        $input = '<div class="jumbotron"><h1>Hero</h1></div>';
        $expected = '<div class="bg-light p-5 rounded-3"><h1>Hero</h1></div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test close button translations.
     */
    public function test_close_button(): void {
        $input = '<button class="close">×</button>';
        $expected = '<button class="btn-close">×</button>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test handling of single quotes in class attributes.
     */
    public function test_single_quotes(): void {
        $input = '<div class=\'ml-3 float-left\'>Single quotes</div>';
        $expected = '<div class=\'ms-3 float-start\'>Single quotes</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test complex example with multiple Bootstrap classes.
     */
    public function test_complex_example(): void {
        $input = '<div class="container ml-2 mr-auto"><div class="row no-gutters">' .
                '<div class="col-md-6 text-left border-right">' .
                '<button class="btn float-right ml-1">Button</button></div></div></div>';
        $expected = '<div class="container ms-2 me-auto"><div class="row g-0">' .
                '<div class="col-md-6 text-start border-end">' .
                '<button class="btn float-end ms-1">Button</button></div></div></div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test that non-Bootstrap classes are preserved.
     */
    public function test_non_bootstrap_classes(): void {
        $input = '<div class="my-custom-class another-class">Custom only</div>';
        $expected = '<div class="my-custom-class another-class">Custom only</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test handling of empty class attributes.
     */
    public function test_empty_class(): void {
        $input = '<div class="">Empty</div>';
        $expected = '<div class="">Empty</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test handling of elements without class attributes.
     */
    public function test_no_class_attribute(): void {
        $input = '<div>No class</div>';
        $expected = '<div>No class</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test that empty text is handled properly.
     */
    public function test_empty_text(): void {
        $this->assertEquals('', $this->filter->filter(''));
    }

    /**
     * Test that text without HTML is handled properly.
     */
    public function test_plain_text(): void {
        $input = 'This is plain text without HTML';
        $this->assertEquals($input, $this->filter->filter($input));
    }

    /**
     * Test HTML without class attributes.
     */
    public function test_html_without_classes(): void {
        $input = '<p>This <strong>HTML</strong> has no classes</p>';
        $this->assertEquals($input, $this->filter->filter($input));
    }

    /**
     * Test mixed Bootstrap and non-Bootstrap classes.
     */
    public function test_mixed_classes(): void {
        $input = '<div class="my-class ml-2 custom-component mr-auto">Mixed</div>';
        $expected = '<div class="my-class ms-2 custom-component me-auto">Mixed</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test whitespace handling in class attributes.
     */
    public function test_whitespace_handling(): void {
        $input = '<div class="  ml-1   mr-2  custom-class  ">Whitespace</div>';
        $expected = '<div class="ms-1 me-2 custom-class">Whitespace</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test case insensitive class attribute matching.
     */
    public function test_case_insensitive_class_attribute(): void {
        $input = '<div CLASS="ml-1 mr-2">Uppercase CLASS</div>';
        $expected = '<div CLASS="ms-1 me-2">Uppercase CLASS</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test multiple class attributes in different elements.
     */
    public function test_multiple_elements(): void {
        $input = '<div class="ml-1">First</div><span class="mr-2">Second</span><p class="text-left">Third</p>';
        $expected = '<div class="ms-1">First</div><span class="me-2">Second</span><p class="text-start">Third</p>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test form control specific translations.
     */
    public function test_form_controls(): void {
        $input = '<input class="form-control-file custom-select" type="file">';
        $expected = '<input class="form-control form-select" type="file">';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test that order of classes is preserved.
     */
    public function test_class_order_preservation(): void {
        $input = '<div class="first-class ml-1 middle-class mr-2 last-class">Order test</div>';
        $expected = '<div class="first-class ms-1 middle-class me-2 last-class">Order test</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test responsive float utilities.
     */
    public function test_responsive_floats(): void {
        $input = '<div class="float-sm-left float-md-right float-lg-left">Responsive floats</div>';
        $expected = '<div class="float-sm-start float-md-end float-lg-start">Responsive floats</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test responsive text alignment.
     */
    public function test_responsive_text_alignment(): void {
        $input = '<p class="text-sm-left text-md-center text-lg-right">Responsive text</p>';
        $expected = '<p class="text-sm-start text-md-center text-lg-end">Responsive text</p>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test font weight utility translations.
     */
    public function test_font_weight_utilities(): void {
        $input = '<p class="font-weight-bold font-weight-normal font-weight-light">Font weights</p>';
        $expected = '<p class="fw-bold fw-normal fw-light">Font weights</p>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test button block class translation.
     */
    public function test_button_block(): void {
        $input = '<button class="btn btn-primary btn-block">Block button</button>';
        $expected = '<button class="btn btn-primary d-grid">Block button</button>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test XXL breakpoint translations.
     */
    public function test_xxl_breakpoint_classes(): void {
        $input = '<div class="ml-xxl-3 mr-xxl-auto float-xxl-left text-xxl-right">XXL breakpoint</div>';
        $expected = '<div class="ms-xxl-3 me-xxl-auto float-xxl-start text-xxl-end">XXL breakpoint</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test screen reader utilities.
     */
    public function test_screen_reader_utilities(): void {
        $input = '<span class="sr-only sr-only-focusable">Screen reader content</span>';
        $expected = '<span class="visually-hidden visually-hidden-focusable">Screen reader content</span>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test form utility translations.
     */
    public function test_form_utilities(): void {
        $input = '<div class="form-row form-inline">Form utilities</div>';
        $expected = '<div class="row g-3 d-flex align-items-center">Form utilities</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test badge class translations.
     */
    public function test_badge_classes(): void {
        $input = '<span class="badge badge-primary badge-warning badge-light">Badges</span>';
        $expected = '<span class="badge bg-primary bg-warning text-dark bg-light">Badges</span>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test dropdown menu improvements.
     */
    public function test_dropdown_improvements(): void {
        $input = '<div class="dropdown-menu dropdown-menu-right dropright dropleft">Dropdown</div>';
        $expected = '<div class="dropdown-menu dropdown-menu-end dropend dropstart">Dropdown</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test data attribute translations.
     */
    public function test_data_attributes(): void {
        $input = '<button data-toggle="modal" data-target="#myModal" data-dismiss="modal">Modal button</button>';
        $expected = '<button data-bs-toggle="modal" data-bs-target="#myModal" data-bs-dismiss="modal">Modal button</button>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test complex data attributes with various Bootstrap components.
     */
    public function test_complex_data_attributes(): void {
        $input = '<div data-spy="scroll" data-offset="100" data-ride="carousel" data-slide="next">Complex</div>';
        $expected = '<div data-bs-spy="scroll" data-bs-offset="100" data-bs-ride="carousel" data-bs-slide="next">Complex</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test mixed class and data attribute translations.
     */
    public function test_mixed_class_and_data_attributes(): void {
        $input = '<div class="btn btn-block sr-only" data-toggle="tooltip" data-placement="top">Mixed</div>';
        $expected = '<div class="btn d-grid visually-hidden" data-bs-toggle="tooltip" data-bs-placement="top">Mixed</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test whiteList to allowList data attribute change.
     */
    public function test_whitelist_to_allowlist(): void {
        $input = '<div data-whiteList=\'["strong", "em"]\'>Sanitization</div>';
        $expected = '<div data-bs-allowList=\'["strong", "em"]\'>Sanitization</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test badge-pill class translation.
     */
    public function test_badge_pill(): void {
        $input = '<span class="badge badge-primary badge-pill">Pill badge</span>';
        $expected = '<span class="badge bg-primary rounded-pill">Pill badge</span>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test form-group class (no automatic replacement in filter).
     */
    public function test_form_group(): void {
        // Form-group should remain unchanged as it requires manual intervention.
        $input = '<div class="form-group"><input class="form-control"></div>';
        $expected = '<div class="form-group"><input class="form-control"></div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test font-italic utility translation.
     */
    public function test_font_italic(): void {
        $input = '<p class="font-italic">Italic text</p>';
        $expected = '<p class="fst-italic">Italic text</p>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test rounded size utilities.
     */
    public function test_rounded_sizes(): void {
        $input = '<div class="rounded-sm rounded-lg">Rounded elements</div>';
        $expected = '<div class="rounded-1 rounded-3">Rounded elements</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test custom control classes.
     */
    public function test_custom_controls(): void {
        $input = '<div class="custom-control custom-control-input custom-control-label">Custom control</div>';
        $expected = '<div class="form-check form-check-input form-check-label">Custom control</div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }

    /**
     * Test comprehensive Moodle deprecated classes coverage.
     */
    public function test_moodle_deprecated_classes_comprehensive(): void {
        // Test multiple deprecated classes from Moodle's bs4-compat.scss.
        $input = '<div class="media badge-pill font-italic rounded-sm sr-only">' .
                '<div class="media-body custom-control custom-select">' .
                '<span class="badge-primary close">Content</span>' .
                '</div></div>';
        $expected = '<div class="d-flex rounded-pill fst-italic rounded-1 visually-hidden">' .
                   '<div class="flex-grow-1 ms-3 form-check form-select">' .
                   '<span class="bg-primary btn-close">Content</span>' .
                   '</div></div>';
        $this->assertEquals($expected, $this->filter->filter($input));
    }
}
