<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace auth_outage\output\manage;

use auth_outage\local\outage;
use flexible_table;
use html_writer;
use moodle_url;

defined('MOODLE_INTERNAL') || die();
require_once($CFG->libdir . '/tablelib.php');

/**
 * base_table class.
 *
 * @package    auth_outage
 * @author     Daniel Thee Roperto <danielroperto@catalyst-au.net>
 * @copyright  2016 Catalyst IT
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class base_table extends flexible_table {
    /**
     * @var int Autogenerated id.
     */
    private static $autoid = 0;

    /**
     * Creates start time string.
     * @param string $starttime start time
     */
    protected static function create_starttime_string($starttime) {
        $absolute = userdate($starttime, get_string('datetimeformat', 'auth_outage'));
        $relative = $starttime - time();
        $format = ($relative < 0) ? 'tablestarttimepastformat' : 'tablestarttimefutureformat';
        $relative = format_time($relative);
        return get_string($format, 'auth_outage', compact('absolute', 'relative'));
    }

    /**
     * Constructor
     * @param string|null $id to be used by the table, autogenerated if null.
     */
    public function __construct($id = null) {
        global $PAGE;

        $id = (is_null($id) ? self::$autoid++ : $id);
        parent::__construct('auth_outage_manage_' . $id);

        $this->define_baseurl($PAGE->url);
        $this->set_attribute('class', 'generaltable admintable');
    }

    /**
     * Create the action buttons HTML code for a specific outage.
     * @param outage $outage The outage to generate the buttons.
     * @param bool $editdelete If it should display the edit and delete button.
     * @return string The HTML code of the action buttons.
     */
    protected function create_data_buttons(outage $outage, $editdelete) {
        global $OUTPUT;
        $buttons = '';

        // View button.
        $buttons .= html_writer::link(
            new moodle_url('/auth/outage/info.php', ['id' => $outage->id]),
            $OUTPUT->pix_icon('t/preview', get_string('view'), 'moodle', ['class' => 'iconsmall']),
            [
                'title' => get_string('view'),
                'target' => '_blank',
            ]
        );

        // Edit button if required.
        if ($editdelete) {
            $buttons .= html_writer::link(
                new moodle_url('/auth/outage/edit.php', ['edit' => $outage->id]),
                $OUTPUT->pix_icon('t/edit', get_string('edit'), 'moodle', ['class' => 'iconsmall']),
                ['title' => get_string('edit')]
            );
        }

        // Clone button.
        $buttons .= html_writer::link(
            new moodle_url('/auth/outage/edit.php', ['clone' => $outage->id]),
            $OUTPUT->pix_icon('t/copy', get_string('clone', 'auth_outage'), 'moodle', ['class' => 'iconsmall']),
            ['title' => get_string('clone', 'auth_outage')]
        );

        // Finish button if ongoing.
        if ($outage->is_ongoing()) {
            $buttons .= html_writer::link(
                new moodle_url('/auth/outage/finish.php', ['id' => $outage->id]),
                $OUTPUT->pix_icon('t/check', get_string('finish', 'auth_outage'), 'moodle', ['class' => 'iconsmall']),
                ['title' => get_string('finish', 'auth_outage')]
            );
        }

        // Delete button if required.
        if ($editdelete) {
            $buttons .= html_writer::link(
                new moodle_url('/auth/outage/delete.php', ['id' => $outage->id]),
                $OUTPUT->pix_icon('t/delete', get_string('delete'), 'moodle', ['class' => 'iconsmall']),
                ['title' => get_string('delete')]
            );
        }

        return html_writer::tag('nobr', $buttons);
    }
}
