<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Copyright (C) 2007-2011 Catalyst IT (http://www.catalyst.net.nz)
 * Copyright (C) 2011-2013 Totara LMS (http://www.totaralms.com)
 * Copyright (C) 2014 onwards Catalyst IT (http://www.catalyst-eu.net)
 *
 * @package    mod_facetoface
 * @copyright  2014 onwards Catalyst IT <http://www.catalyst-eu.net>
 * @author     Stacey Walker <stacey@catalyst-eu.net>
 * @author     Alastair Munro <alastair.munro@totaralms.com>
 * @author     Aaron Barnes <aaron.barnes@totaralms.com>
 * @author     Francois Marier <francois@catalyst.net.nz>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once($CFG->dirroot . '/mod/facetoface/lib.php');

// Face-to-face session ID.
$s = required_param('s', PARAM_INT);

$takeattendance = optional_param('takeattendance', false, PARAM_BOOL); // Take attendance.
$cancelform = optional_param('cancelform', false, PARAM_BOOL); // Cancel request.
$backtoallsessions = optional_param('backtoallsessions', 0, PARAM_INT); // Face-to-face activity to return to.
$download = optional_param('download', '', PARAM_ALPHA); // Download attendees.

// Load data.
if (!$session = facetoface_get_session($s)) {
    throw new moodle_exception('error:incorrectcoursemodulesession', 'facetoface');
}
if (!$facetoface = $DB->get_record('facetoface', ['id' => $session->facetoface])) {
    throw new moodle_exception('error:incorrectfacetofaceid', 'facetoface');
}
if (!$course = $DB->get_record('course', ['id' => $facetoface->course])) {
    throw new moodle_exception('error:coursemisconfigured', 'facetoface');
}
if (!$cm = get_coursemodule_from_instance('facetoface', $facetoface->id, $course->id)) {
    throw new moodle_exception('error:incorrectcoursemodule', 'facetoface');
}

// Load attendees.
$attendees = facetoface_get_attendees($session->id);

// Load cancellations.
$cancellations = facetoface_get_cancellations($session->id);

/*
 * Capability checks to see if the current user can view this page
 *
 * This page is a bit of a special case in this respect as there are four uses for this page.
 *
 * 1) Viewing attendee list
 *   - Requires mod/facetoface:viewattendees capability in the course
 *
 * 2) Viewing cancellation list
 *   - Requires mod/facetoface:viewcancellations capability in the course
 *
 * 3) Taking attendance
 *   - Requires mod/facetoface:takeattendance capabilities in the course
 */
$context = context_course::instance($course->id);
$contextmodule = context_module::instance($cm->id);
require_course_login($course, true, $cm);

// Actions the user can perform.
$canviewattendees = has_capability('mod/facetoface:viewattendees', $context);
$cantakeattendance = has_capability('mod/facetoface:takeattendance', $context);
$canviewcancellations = has_capability('mod/facetoface:viewcancellations', $context);
$canviewsession = $canviewattendees || $cantakeattendance || $canviewcancellations;
$canapproverequests = false;

$requests = [];
$declines = [];

// If a user can take attendance, they can approve staff's booking requests.
if ($cantakeattendance) {
    $requests = facetoface_get_requests($session->id);
}

// If requests found (but not in the middle of taking attendance), show requests table.
if ($requests && !$takeattendance) {
    $canapproverequests = true;
}

// Check the user is allowed to view this page.
if (!$canviewattendees && !$cantakeattendance && !$canapproverequests && !$canviewcancellations) {
    throw new moodle_exception('nopermissions', '', "{$CFG->wwwroot}/mod/facetoface/view.php?id={$cm->id}", get_string('view'));
}

// Check user has permissions to take attendance.
if ($takeattendance && !$cantakeattendance) {
    throw new moodle_exception('nopermissions', '', '', get_capability_string('mod/facetoface:takeattendance'));
}

if (!empty($download) && $canviewattendees) {
    // Download list of attendees.
    facetoface_download_attendees(format_string($facetoface->name), $session, $attendees, $download);
    exit();
}

/*
 * Handle submitted data
 */
if ($form = data_submitted()) {
    if (!confirm_sesskey()) {
        throw new moodle_exception('confirmsesskeybad', 'error');
    }

    $return = "{$CFG->wwwroot}/mod/facetoface/attendees.php?s={$s}&backtoallsessions={$backtoallsessions}";

    if ($cancelform) {
        redirect($return);
    } else if (!empty($form->requests)) {
        // Approve requests.
        if ($canapproverequests && facetoface_approve_requests($form)) {
            // Logging and events trigger.
            $params = [
                'context'  => $contextmodule,
                'objectid' => $session->id,
            ];
            $event = \mod_facetoface\event\approve_requests::create($params);
            $event->add_record_snapshot('facetoface_sessions', $session);
            $event->add_record_snapshot('facetoface', $facetoface);
            $event->trigger();
        }

        redirect($return);
    } else if ($takeattendance) {
        if (facetoface_take_attendance($form)) {
            // Logging and events trigger.
            $params = [
                'context'  => $contextmodule,
                'objectid' => $session->id,
            ];
            $event = \mod_facetoface\event\take_attendance::create($params);
            $event->add_record_snapshot('facetoface_sessions', $session);
            $event->add_record_snapshot('facetoface', $facetoface);
            $event->trigger();
        } else {
            // Logging and events trigger.
            $params = [
                'context'  => $contextmodule,
                'objectid' => $session->id,
            ];
            $event = \mod_facetoface\event\take_attendance_failed::create($params);
            $event->add_record_snapshot('facetoface_sessions', $session);
            $event->add_record_snapshot('facetoface', $facetoface);
            $event->trigger();
        }
        redirect($return . '&takeattendance=1');
    }
}

/*
 * Print page header
 */

// Logging and events trigger.
$params = [
    'context'  => $contextmodule,
    'objectid' => $session->id,
];
$event = \mod_facetoface\event\attendees_viewed::create($params);
$event->add_record_snapshot('facetoface_sessions', $session);
$event->add_record_snapshot('facetoface', $facetoface);
$event->trigger();

$pagetitle = format_string($facetoface->name);

$PAGE->set_url('/mod/facetoface/attendees.php', ['s' => $s]);
$PAGE->set_title($pagetitle);
$PAGE->set_heading($course->fullname);

echo $OUTPUT->header();

/*
 * Print page content
 */

// If taking attendance, make sure the session has already started.
if ($takeattendance && $session->datetimeknown && !facetoface_has_session_started($session, time())) {
    $link = "{$CFG->wwwroot}/mod/facetoface/attendees.php?s={$session->id}";
    throw new moodle_exception('error:canttakeattendanceforunstartedsession', 'facetoface', $link);
}

echo $OUTPUT->box_start();
echo $OUTPUT->heading(format_string($facetoface->name));

if ($canviewsession) {
    echo facetoface_print_session($session, true);
}

/*
 * Print attendees (if user able to view)
 */
if ($canviewattendees || $cantakeattendance) {
    if ($takeattendance) {
        $heading = get_string('takeattendance', 'facetoface');
    } else {
        $heading = get_string('attendees', 'facetoface');
    }

    echo $OUTPUT->heading($heading);

    if (empty($attendees)) {
        echo $OUTPUT->notification(get_string('nosignedupusers', 'facetoface'));
    } else {
        if ($takeattendance) {
            $attendeesurl = new moodle_url('attendees.php', ['s' => $s, 'takeattendance' => '1']);
            echo html_writer::start_tag('form', ['action' => $attendeesurl, 'method' => 'post']);
            echo html_writer::tag('p', get_string('attendanceinstructions', 'facetoface'));
            echo html_writer::empty_tag('input', ['type' => 'hidden', 'name' => 'sesskey', 'value' => $USER->sesskey]);
            echo html_writer::empty_tag('input', ['type' => 'hidden', 'name' => 's', 'value' => $s]);
            echo html_writer::empty_tag('input', [
                'type' => 'hidden', ' name' => 'backtoallsessions',
                'value' => $backtoallsessions,
            ]) . '</p>';

            // Prepare status options array.
            $statuses = facetoface_statuses();
            $statusoptions = [];
            foreach ($statuses as $key => $value) {
                if ($key <= MDL_F2F_STATUS_BOOKED) {
                    continue;
                }

                $statusoptions[$key] = get_string('status_' . $value, 'facetoface');
            }
        }

        $table = new html_table();
        $table->head = [get_string('name')];
        $table->align = ['left'];
        $table->size = ['100%'];

        if ($takeattendance) {
            $table->head[] = get_string('currentstatus', 'facetoface');
            $table->align[] = 'center';
            $table->head[] = get_string('attendedsession', 'facetoface');
            $table->align[] = 'center';
        } else {
            if (!get_config('facetoface', 'hidecost')) {
                $table->head[] = get_string('cost', 'facetoface');
                $table->align[] = 'center';
                if (!get_config('facetoface', 'hidediscount')) {
                    $table->head[] = get_string('discountcode', 'facetoface');
                    $table->align[] = 'center';
                }
            }

            $table->head[] = get_string('attendance', 'facetoface');
            $table->align[] = 'center';
        }

        foreach ($attendees as $attendee) {
            $data = [];
            $attendeeurl = new moodle_url('/user/view.php', ['id' => $attendee->id, 'course' => $course->id]);
            $data[] = html_writer::link($attendeeurl, format_string(fullname($attendee)));

            if ($takeattendance) {
                // Show current status.
                $data[] = get_string('status_' . facetoface_get_status($attendee->statuscode), 'facetoface');

                $optionid = 'submissionid_' . $attendee->submissionid;
                $status = $attendee->statuscode;
                $select = html_writer::select($statusoptions, $optionid, $status);
                $data[] = $select;
            } else {
                if (!get_config('facetoface', 'hidecost')) {
                    $data[] = facetoface_cost($attendee->id, $session->id, $session);
                    if (!get_config('facetoface', 'hidediscount')) {
                        $data[] = $attendee->discountcode;
                    }
                }
                $data[] = str_replace(
                    ' ',
                    '&nbsp;',
                    get_string('status_' . facetoface_get_status($attendee->statuscode), 'facetoface')
                );
            }
            $table->data[] = $data;
        }

        echo html_writer::table($table);

        if ($takeattendance) {
            echo html_writer::start_tag('p');
            echo html_writer::empty_tag('input', ['type' => 'submit', 'value' => get_string('saveattendance', 'facetoface')]);
            echo '&nbsp;' . html_writer::empty_tag('input', [
                'type' => 'submit', 'name' => 'cancelform',
                'value' => get_string('cancel'),
            ]);
            echo html_writer::end_tag('p') . html_writer::end_tag('form');
        } else {
            // Actions.
            print html_writer::start_tag('p');
            if ($cantakeattendance && $session->datetimeknown && facetoface_has_session_started($session, time())) {
                // Take attendance.
                $attendanceurl = new moodle_url('attendees.php', [
                    's' => $session->id, 'takeattendance' => '1',
                    'backtoallsessions' => $backtoallsessions,
                ]);
                echo html_writer::link($attendanceurl, get_string('takeattendance', 'facetoface')) . ' - ';
            }
        }
    }

    if (
        !$takeattendance
        && (has_capability('mod/facetoface:addattendees', $context)
        || has_capability('mod/facetoface:removeattendees', $context))
    ) {
        // Add/remove attendees.
        $editattendeeslink = new moodle_url('editattendees.php', ['s' => $session->id, 'backtoallsessions' => $backtoallsessions]);
        echo html_writer::link($editattendeeslink, get_string('addremoveattendees', 'facetoface')) . ' - ';
    }
    echo html_writer::link(
        "attendees.php?s=$session->id&backtoallsessions=$session->facetoface&download=ods",
        get_string('downloadods')
    ) . ' - ';
    echo html_writer::link(
        "attendees.php?s=$session->id&backtoallsessions=$session->facetoface&download=xls",
        get_string('downloadexcel')
    ) . ' - ';
}

// Go back.
$url = new moodle_url('/course/view.php', ['id' => $course->id]);
if ($backtoallsessions) {
    $url = new moodle_url('/mod/facetoface/view.php', ['f' => $facetoface->id, 'backtoallsessions' => $backtoallsessions]);
}
echo html_writer::link($url, get_string('goback', 'facetoface')) . html_writer::end_tag('p');

/*
 * Print unapproved requests (if user able to view)
 */
if ($canapproverequests) {
    echo html_writer::empty_tag('br', ['id' => 'unapproved']);
    if (!$requests) {
        echo $OUTPUT->notification(get_string('noactionableunapprovedrequests', 'facetoface'));
    } else {
        $canbookuser = (facetoface_session_has_capacity($session, $contextmodule) || $session->allowoverbook);

        $OUTPUT->heading(get_string('unapprovedrequests', 'facetoface'));

        if (!$canbookuser) {
            echo html_writer::tag('p', get_string('cannotapproveatcapacity', 'facetoface'));
        }

        $action = new moodle_url('attendees.php', ['s' => $s]);
        echo html_writer::start_tag('form', ['action' => $action->out(), 'method' => 'post']);
        echo html_writer::empty_tag('input', ['type' => 'hidden', 'name' => 'sesskey', 'value' => $USER->sesskey]);
        echo html_writer::empty_tag('input', ['type' => 'hidden', 'name' => 's', 'value' => $s]);
        echo html_writer::empty_tag('input', [
            'type' => 'hidden', 'name' => 'backtoallsessions',
            'value' => $backtoallsessions,
        ]) . html_writer::end_tag('p');

        $table = new html_table();
        $table->head = [
            get_string('name'), get_string('timerequested', 'facetoface'),
            get_string('decidelater', 'facetoface'), get_string('decline', 'facetoface'),
            get_string('approve', 'facetoface'),
        ];
        $table->align = ['left', 'center', 'center', 'center', 'center'];

        foreach ($requests as $attendee) {
            $data = [];
            $attendeelink = new moodle_url('/user/view.php', ['id' => $attendee->id, 'course' => $course->id]);
            $data[] = html_writer::link($attendeelink, format_string(fullname($attendee)));
            $data[] = userdate($attendee->timerequested, get_string('strftimedatetime'));
            $data[] = html_writer::empty_tag('input', [
                'type' => 'radio', 'name' => 'requests[' . $attendee->id . ']',
                'value' => '0', 'checked' => 'checked',
            ]);
            $data[] = html_writer::empty_tag('input', [
                'type' => 'radio', 'name' => 'requests[' . $attendee->id . ']',
                'value' => '1',
            ]);
            $disabled = ($canbookuser) ? [] : ['disabled' => 'disabled'];
            $data[] = html_writer::empty_tag('input', array_merge([
                'type' => 'radio', 'name' => 'requests[' . $attendee->id . ']',
                'value' => '2',
            ], $disabled));
            $table->data[] = $data;
        }

        echo html_writer::table($table);

        echo html_writer::tag('p', html_writer::empty_tag('input', [
            'type' => 'submit',
            'value' => get_string('updaterequests', 'facetoface'),
        ]));
        echo html_writer::end_tag('form');
    }
}

/*
 * Print cancellations (if user able to view)
 */
if (!$takeattendance && $canviewcancellations && $cancellations) {
    echo html_writer::empty_tag('br');
    echo $OUTPUT->heading(get_string('cancellations', 'facetoface'));

    $table = new html_table();
    $table->summary = get_string('cancellationstablesummary', 'facetoface');
    $table->head = [
        get_string('name'), get_string('timesignedup', 'facetoface'),
        get_string('timecancelled', 'facetoface'), get_string('cancelreason', 'facetoface'),
    ];
    $table->align = ['left', 'center', 'center'];

    foreach ($cancellations as $attendee) {
        $data = [];
        $attendeelink = new moodle_url('/user/view.php', ['id' => $attendee->id, 'course' => $course->id]);
        $data[] = html_writer::link($attendeelink, format_string(fullname($attendee)));
        $data[] = userdate($attendee->timesignedup, get_string('strftimedatetime'));
        $data[] = userdate($attendee->timecancelled, get_string('strftimedatetime'));
        $data[] = format_string($attendee->cancelreason);
        $table->data[] = $data;
    }
    echo html_writer::table($table);
}

/*
 * Print page footer
 */
echo $OUTPUT->box_end();
echo $OUTPUT->footer($course);
