<?php
// This file is part of the Certify Certificate module for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace mod_certify\local;

use mod_certify\apirest\apirest;

/**
 * Local functions related to certificates.
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class certificates
{
    /**
     * The apirest object used to call API requests.
     * @var apirest
     */
    private $apirest;

    /**
     * A random value for a new certificate name.
     * @var int
     */
    private $rand;

    /**
     * Constructor method.
     *
     * @param stdObject $apirest a mock apirest for testing.
     * @param int $rand a random number to avoid duplicated names when creating certificates.
     */
    public function __construct($apirest = null, $rand = null)
    {
        // An apirest with a mock client is passed when unit testing.
        if ($apirest) {
            $this->apirest = $apirest;
        } else {
            $this->apirest = new apirest();
        }

        // A fixed value is passed when unit testing.
        if ($rand) {
            $this->rand = $rand;
        } else {
            $this->rand = mt_rand();
        }
    }

    /**
     * Get the certificates for the issuer
     * @return array[stdClass] $certificates
     */
    public function get_certificates()
    {
        $pagesize = 30;
        $page = 1;

        try {
            $certificates = [];

            // Query the Certify API and loop until it returns that there is no next page.
            do {
                $response = $this->apirest->get_groups($pagesize, $page);

                if (isset($response->data) && is_array($response->data)) {
                    foreach ($response->data as $certificate) {
                        $certificates[$certificate->uuid] = $certificate->name . ' - ' . $certificate->identifier;
                    }
                }

                $page++;
                $total_pages = $response->meta->pagination->total_pages ?? 0;

            } while ($page <= $total_pages);

            natcasesort($certificates);

            return $certificates;
        } catch (\Exception $e) {
            throw new \moodle_exception('getcertificateserror', 'certify', 'https://help.certify.one/hc/en-us');
        }
    }

    /**
     * List all of the issuer's templates
     *
     * @return array[stdClass] $templates
     */
    public function get_templates()
    {
        $pagesize = 50;
        $page = 1;

        try {
            $templates = [];
            // Query the Certify API and loop until it returns that there is no next page.
            for ($i = 0; $i <= 100; $i++) {
                $response = $this->apirest->search_groups($pagesize, $page);
                foreach ($response->groups as $group) {
                    $templates[$group->name] = $group->name;
                }

                $page++;
                if ($response->meta->next_page === null) {
                    // If the Certify API returns that there
                    // is no next page, end the loop.
                    break;
                }
            }
            return $templates;
        } catch (\Exception $e) {
            throw new \moodle_exception('gettemplateserror', 'certify', 'https://help.certify.one/hc/en-us');
        }
    }
}
