<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Contains class mod_certify\update\update
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace mod_certify\update;

class update {

    /**
     * Check if the current plugin version is the latest.
     *
     * @return bool True if an update is available, false otherwise.
     */
    public function check_for_update(): bool {
        $currentversion = $this->get_current_version();
        $latestversion = $this->get_latest_version();

        if ($latestversion === null) {
            mtrace('Certify: Unable to fetch the latest version from API.');
            return false;
        }

        return $latestversion !== $currentversion;
    }

    /**
     * Get the current plugin version from version.php.
     *
     * @return string The current plugin release version.
     */
    private function get_current_version(): string {
        global $CFG;

        $plugin = new \stdClass();
        require($CFG->dirroot . '/mod/certify/version.php');

        return $plugin->release ?? '0.0.0';
    }

    /**
     * Get the latest version from the Certify API.
     *
     * @return string|null The latest version string, or null on failure.
     */
    private function get_latest_version(): ?string {
        $api = new \mod_certify\apirest\apirest();

        try {
            $response = $api->version();

            if ($response && isset($response->version)) {
                return $response->version;
            }

            return null;
        } catch (\Exception $e) {
            mtrace('Certify: Error fetching version from API - ' . $e->getMessage());
            return null;
        }
    }
}