<?php
// This file is part of the Certify Certificate module for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Certificate module core interaction API
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
defined('MOODLE_INTERNAL') || die();

require_once($CFG->dirroot . '/mod/quiz/locallib.php');

use mod_certify\apirest\apirest;
use mod_certify\Html2Text\Html2Text;
use mod_certify\local\credentials;
use mod_certify\local\evidenceitems;
use mod_certify\local\users;
use mod_certify\local\certify;

/**
 * Checks if a user has earned a specific credential according to the activity settings
 * @param stdClass $record An Certify activity record
 * @param array $user
 * @return bool
 */
function certify_check_if_cert_earned($record, $user)
{
    global $DB;

    $earned = false;
    return $earned;
}

/**
 * Get the SSO link for a recipient
 * @param string $credentialuuid
 * @param string $firstname
 * @param string $lastname
 * @param string $email
 */
function certify_get_recipient_sso_link($credentialuuid, $firstname, $lastname, $email)
{
    global $CFG;

    $apirest = new apirest();

    $fullname = $firstname . ' ' . $lastname;

    try {
        $response = $apirest->recipient_sso_link($credentialuuid, $fullname, $email);
        return $response->url->authenticated;

    } catch (Exception $e) {
        return null;
    }
}

function get_credential($credentialuuid)
{
    global $CFG;

    $apirest = new apirest();

    try {
        $response = $apirest->credential($credentialuuid);
        return $response->data;

    } catch (Exception $e) {
        return null;
    }
}

/**
 * Issue a certificate
 *
 * @param int $userid
 * @param int $certificateid
 * @param string $name
 * @param string $email
 * @param string $grade
 * @param string $quizname
 * @param date|null $completedtimestamp
 * @param array $customattributes
 */
function certify_issue_default_certificate(
    $userid,
    $certificateid,
    $name,
    $email,
    $grade,
    $quizname,
    $completedtimestamp = null,
    $customattributes = null
) {
    global $DB, $CFG;

    if (!isset($completedtimestamp)) {
        $completedtimestamp = time();
    }
    $issuedon = date('Y-m-d', (int) $completedtimestamp);

    // Issue certs.
    $certifycertificate = $DB->get_record('certify', ['id' => $certificateid]);

    $courseurl = new moodle_url('/course/view.php', ['id' => $certifycertificate->course]);
    $courselink = $courseurl->__toString();

    $restapi = new apirest();
    $credential = $restapi->create_credential_legacy(
        $name,
        $email,
        $certifycertificate->achievementid,
        $issuedon,
        null,
        $certifycertificate->certificatename,
        $certifycertificate->description,
        $courselink,
        $customattributes
    );

    // Evidence item posts.
    $credentialid = $credential->credential->id;
    if ($grade) {
        if ($grade < 50) {
            $hidden = true;
        } else {
            $hidden = false;
        }

        $response = $restapi->create_evidence_item_grade($grade, $quizname, $credentialid, $hidden);
    }

    $evidenceitem = new evidenceitem();

    if ($transcript = certify_get_transcript($certifycertificate->course, $userid, $certifycertificate->finalquiz)) {
        $evidenceitem->post_evidence($credentialid, $transcript, false);
    }

    $evidenceitem->post_essay_answers($userid, $certifycertificate->course, $credentialid);
    $evidenceitem->course_duration_evidence($userid, $certifycertificate->course, $credentialid, $completedtimestamp);

    return json_decode($response);
}

/**
 * Log message when certificate is created.
 *
 * @param int $certificateid ID of the certificate.
 * @param int $userid ID of the user.
 * @param int $courseid ID of the course.
 * @param int $cmid ID of the couse module.
 */
function certify_log_creation($certificateid, $userid, $courseid, $cmid)
{
    global $DB;

    // Get context.
    $certifymod = $DB->get_record('modules', ['name' => 'certify'], '*', MUST_EXIST);
    if ($cmid) {
        $cm = $DB->get_record('course_modules', ['id' => (int) $cmid], '*');
    } else { // This is an activity add, so we have to use $courseid.
        $coursemodules = $DB->get_records('course_modules', ['course' => $courseid, 'module' => $certifymod->id]);
        $cm = end($coursemodules);
    }
    $context = context_module::instance($cm->id);

    return \mod_certify\event\certificate_created::create([
        'objectid' => $certificateid,
        'context' => $context,
        'relateduserid' => $userid,
    ]);
}

/**
 * Quiz submission handler (checks for a completed course)
 *
 * @param core/event $event quiz mod attempt_submitted event
 */
function certify_quiz_submission_handler($event)
{

}


/**
 * Course completion handler
 *
 * @param core/event $event
 */
function certify_course_completed_handler($event)
{
    global $DB, $CFG;

    $localcredentials = new credentials();
    $usersclient = new users();
    $certify = new certify();

    $user = $DB->get_record('user', ['id' => $event->relateduserid]);

    // Check we have a course record.
    if ($certifycertificaterecords = $DB->get_records('certify', ['course' => $event->courseid])) {
        foreach ($certifycertificaterecords as $record) {
            // Load user grade to attach in the credential.
            $gradeattributes = $usersclient->get_user_grades($record, $user->id);
            $gradeattributemapping = $usersclient->load_user_grade_as_custom_attributes($record, $gradeattributes, $user->id);
            $additionalattributemapping = $certify->load_credential_custom_attributes($record, $user->id);
            $customattributes = array_merge($gradeattributemapping, $additionalattributemapping);

            if ($record->credentialuuid) {
                $localcredentials->create_credential($user, $record->credentialuuid, null, $customattributes);

            } else {
                $apiresponse = certify_issue_default_certificate(
                    $user->id,
                    $record->id,
                    fullname($user),
                    $user->email,
                    null,
                    null,
                    null,
                    $customattributes
                );
                $certificateevent = \mod_certify\event\certificate_created::create([
                    'objectid' => $apiresponse->credential->id,
                    'context' => context_module::instance($event->contextinstanceid),
                    'relateduserid' => $event->relateduserid,
                ]);
                $certificateevent->trigger();
            }
        }
    }
}


/**
 * Make a transcript if user has completed 2/3 of the quizes.
 *
 * @param int $courseid
 * @param int $userid
 * @param int $finalquizid
 */
function certify_get_transcript($courseid, $userid, $finalquizid)
{
    global $DB, $CFG;

    $totalitems = 0;
    $totalscore = 0;
    $itemscompleted = 0;
    $transcriptitems = [];
    $quizes = $DB->get_records_select('quiz', 'course = :course_id', ['course_id' => $courseid], 'id ASC');

    // Grab the grades for all quizes.
    foreach ($quizes as $quiz) {
        if ($quiz->id !== $finalquizid) {
            $highestgrade = quiz_get_best_grade($quiz, $userid);
            if ($highestgrade) {
                $itemscompleted += 1;
                array_push($transcriptitems, [
                    'category' => $quiz->name,
                    'percent' => min(($highestgrade / $quiz->grade) * 100, 100),
                ]);
                $totalscore += min(($highestgrade / $quiz->grade) * 100, 100);
            }
            $totalitems += 1;
        }
    }

    // If they've completed over 2/3 of items
    // and have a passing average, make a transcript.
    if (
        $totalitems !== 0 && $itemscompleted !== 0 && $itemscompleted / $totalitems >= 0.66 &&
        $totalscore / $itemscompleted > 50
    ) {
        return [
            'description' => 'Course Transcript',
            'string_object' => json_encode($transcriptitems),
            'category' => 'transcript',
            'custom' => true,
            'hidden' => true,
        ];
    } else {
        return false;
    }
}

/**
 * Serialize completion array
 *
 * @param Array $completionarray
 */
function serialize_completion_array($completionarray)
{
    return base64_encode(serialize((array) $completionarray));
}

/**
 * Unserialize completion array
 *
 * @param stdClass $completionobject
 */
function unserialize_completion_array($completionobject)
{
    return (array) unserialize(base64_decode($completionobject));
}

/**
 * Get a timestamp for when a student completed a course. This is
 * used when manually issuing certs to get a proper issue date and
 * for the course duration item. Currently checking for the date of
 * the highest quiz attempt for the final quiz specified for that
 * certify activity.
 *
 * @param stdClass $certifyrecord
 * @param stdClass $user
 */
function certify_manual_issue_completion_timestamp($certifyrecord, $user)
{
    global $DB;

    $completedtimestamp = false;

    // Set timestamp to now if no good timestamp was found.
    if ($completedtimestamp === false) {
        $completedtimestamp = time();
    }

    return (int) $completedtimestamp;
}

/**
 * Return 's' when number is bigger than 1
 *
 * @param int $number
 * @return string
 */
function number_ending($number)
{
    return ($number > 1) ? 's' : '';
}

/**
 * Convert number of seconds in a string
 *
 * @param int $seconds
 * @return string
 */
function seconds_to_str($seconds)
{
    $hours = floor(($seconds %= 86400) / 3600);
    if ($hours) {
        return $hours . ' hour' . number_ending($hours);
    }
    $minutes = floor(($seconds %= 3600) / 60);
    if ($minutes) {
        return $minutes . ' minute' . number_ending($minutes);
    }
    return $seconds . ' second' . number_ending($seconds);
}
