<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * api.php - Contains methods to communicate with Compilatio REST API.
 *
 * @package    plagiarism_compilatio
 * @author     Compilatio <support@compilatio.net>
 * @copyright  2023 Compilatio.net {@link https://www.compilatio.net}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace plagiarism_compilatio\compilatio;

/**
 * api class
 */
class api {
    /**
     * @var string $apikey API key
     */
    private string $apikey;

    /**
     * @var string $urlrest Base REST url
     */
    private string $urlrest;

    /**
     * @var string $userid Compilatio user ID
     */
    private ?string $userid;

    /**
     * @var string $recipe Analysis recipe
     */
    private string $recipe;

    /**
     * User ID setter
     * @param  string $userid
     * @return void
     */
    public function set_user_id($userid): void {
        $this->userid = $userid;
    }

    /**
     * Class constructor
     * @param  string $userid User ID
     * @param  string $apikey API key
     */
    public function __construct($userid = null, $apikey = null) {
        if (null === $apikey) {
            $apikey = get_config('plagiarism_compilatio', 'apikey');
        }

        $this->urlrest = 'https://app.compilatio.net';
        $this->userid = $userid;

        if (isset($apikey) && $apikey !== '') {
            $this->apikey = $apikey;
        } else {
            return 'API key not available';
        }
    }

    /**
     * Get Compilatio configuration
     * @return stdClass|false Returns an object containing Compilatio configuration or false if an error occurs
     */
    public function get_config() {
        $endpoint = '/api/public/config/config';
        $response = json_decode($this->build_curl($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            return $response->data;
        }
        return false;
    }

    /**
     * Check if the api key is valid
     *
     * @return boolean Return true if valid, an error message otherwise
     */
    public function check_apikey() {
        $endpoint = '/api/private/authentication/check-api-key';
        $response = json_decode($this->build_curl($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            $readonly = $response->data->user->current_api_key->read_only ?? false;
            set_config('read_only_apikey', (int) $readonly, 'plagiarism_compilatio');

            if (!isset($response->data->user->managed_bundle->name)) {
                return 'Forbidden';
            }

            $recipe = $response->data->user->managed_bundle->name === 'magister-premium' ? 'anasim-premium' : 'anasim';
        }

        set_config('recipe', $recipe ?? 'anasim', 'plagiarism_compilatio');

        $endpoint = '/api/private/user/lms/23a3a6980c0f49d98c5dc1ec03478e9161ad5d352cb4651b14865d21d0e81be';

        $response = json_decode($this->build_curl($endpoint));

        $error = $this->get_error_response($response, 404);
        if ($error === false) {
            return true;
        }
        return $error;
    }

    /**
     * Get Compilatio user ID of legacy account attached to Moodle instance
     *
     * @param  boolean $updateapikey
     * @return string|false Returns user ID on success, or false otherwise
     */
    public function get_apikey_user_id($updateapikey = true) {
        $endpoint = '/api/private/authentication/check-api-key';

        $response = json_decode($this->build_curl($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            $oldmoodleownerid = $response->data->user->current_api_key->old_moodle_owner_id ?? null;

            if (!empty($oldmoodleownerid) || !$updateapikey) {
                return $oldmoodleownerid;
            }

            $this->update_apikey();

            return $response->data->user->id;
        }
        return false;
    }

    /**
     * Update API key for plugin v3 use
     * @return boolean Returns true on success, false otherwise
     */
    public function update_apikey() {
        $endpoint = '/api/private/moodle-configuration/update-api-key';

        $response = json_decode($this->build_curl($endpoint, 'post'));

        if ($this->get_error_response($response, 200) === false) {
            return true;
        }

        return false;
    }

    /**
     * Check if the API key has access rights to the analyses by students.
     *
     * @return bool return true if api key has access to student analyses, false otherwise.
     */
    public function check_allow_student_analyses() {
        $endpoint = '/api/private/authentication/check-api-key';

        $response = json_decode($this->build_curl($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            $bundle = $response->data->user->current_bundle;

            foreach ($bundle->accesses as $access) {
                if ($access->resource == 'api') {
                    if (isset($access->config, $access->config->allow_student_analysis_from_lms)) {
                        return $access->config->allow_student_analysis_from_lms;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Get or create Compilatio user from current Moodle user id
     *
     * @param  mixed $teacher
     * @return mixed return the user if find, null instead.
     */
    public function get_or_create_user($teacher = null) {
        global $USER, $DB;

        if (!isset($teacher)) {
            $teacher = $USER;
        }

        // Check if user already exists in Compilatio.
        $compilatioid = $this->get_user_by_email($teacher->email);

        // Create the user if doesn't exists.
        if ($compilatioid == 404) {
            $compilatioid = $this->set_user($teacher->firstname, $teacher->lastname, $teacher->email);
        }

        if (!preg_match('/^[a-f0-9]{40}$/', $compilatioid)) {
            return null;
        }

        $user = new \stdClass();
        $user->compilatioid = $compilatioid;
        $user->userid = $teacher->id;
        $user->id = $DB->insert_record('plagiarism_compilatio_user', $user);

        return $user;
    }

    /**
     * Create Elastisafe user
     *
     * @param   string  $firstname      User's firstname
     * @param   string  $lastname       User's lastname
     * @param   string  $email          User's email
     * @return  string|false            Return the user's ID, an error message otherwise or false
     */
    private function set_user($firstname, $lastname, $email) {
        $lang = substr(current_language(), 0, 2);

        $endpoint = '/api/private/user/create';
        $params = [
            'firstname' => $firstname,
            'lastname' => $lastname,
            'email' => $email,
            'locale' => [
                'timezone' => date_default_timezone_get(),
                'lang' => $lang,
            ],
            'origin' => 'LMS-Moodle',
        ];

        $response = json_decode($this->build_curl($endpoint, 'post', json_encode($params)));

        if ($this->get_error_response($response, 201) === false) {
            return $response->data->user->id;
        }
        return false;
    }

    /**
     * Get user
     *
     * @param   string  $userid
     * @return  mixed   Return the user if succeed, an error message otherwise
     */
    public function get_user($userid) {
        $endpoint = '/api/private/user/' . $userid;

        $response = json_decode($this->build_curl($endpoint));

        $error = $this->get_error_response($response, 200);
        if ($error === false) {
            return $response->data->user;
        }
        return false;
    }

    /**
     * Get user if exist for email
     *
     * @param   string  $email          Teacher's moodle email
     * @return  string|false            Return the user's ID if exist, the status error code or false
     */
    private function get_user_by_email($email) {
        $endpoint = '/api/private/user/lms/' . strtolower($email);

        $response = json_decode($this->build_curl($endpoint));

        $error = $this->get_error_response($response, 200);
        if ($error === false) {
            return $response->data->user->id;
        }
        return $response->status->code ?? false;
    }

    /**
     * Update Elastisafe user
     *
     * @param   string  $userid         User's identifier
     * @param   string  $firstname      User's firstname
     * @param   string  $lastname       User's lastname
     * @param   string  $email          User's email
     * @return  string                  Return true if succeed, false otherwise
     */
    public function update_user($userid, $firstname, $lastname, $email) {
        $endpoint = '/api/private/user/' . $userid;
        $params = [
            'firstname' => $firstname,
            'lastname' => $lastname,
            'email' => $email,
        ];

        $response = json_decode($this->build_curl($endpoint, 'patch', json_encode($params)));

        if ($this->get_error_response($response, 200) === false) {
            return true;
        }
        return false;
    }

    /**
     * Load document on Compilatio account
     *
     * @param   string  $filename       Filename
     * @param   string  $folderid       Document's folder ID
     * @param   string  $filepath       File path
     * @param   boolean $indexed        Document's indexing state
     * @param   object  $depositor      Document's depositor
     * @param   array   $authors        Document's authors
     * @return  string                  Return the document's ID, an error message otherwise
     */
    public function set_document($filename, $folderid, $filepath, $indexed, $depositor, $authors) {
        $endpoint = '/api/private/document/';
        $params = [
            'file' => new \CURLFile($filepath),
            'filename' => $filename,
            'title' => $filename,
            'folder_id' => $folderid,
            'indexed' => $indexed,
            'origin' => 'moodle',
        ];

        $params['depositor'] = [
            'firstname' => $this->sanitize($depositor->firstname),
            'lastname' => $this->sanitize($depositor->lastname),
            'email_address' => $this->validate_email($depositor->email),
        ];

        foreach ($authors as $author) {
            $params['authors'][] = [
                'firstname' => $this->sanitize($author->firstname),
                'lastname' => $this->sanitize($author->lastname),
                'email_address' => $this->validate_email($author->email),
            ];
        }

        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint, 'upload', $params));

        $error = $this->get_error_response($response, 201);
        if ($error === false) {
            if (in_array('extraction_error', $response->data->document->tags, true)) {
                return false;
            }
            return $response->data->document->id;
        }
        return $error;
    }

    /**
     * Replace forbidden characters
     *
     * @param  string $value
     * @return string Returns sanitized string
     */
    private function sanitize($value) {
        $forbiddencharacters = [
            ".", "!", "?", " => ", "%", "&", "*", "=", "#", "$", "@", "/", "\\", "<", ">", "(", ")", "[", "]", "{", "}",
        ];

        if (!is_string($value) || '' === $value) {
            return null;
        }

        $value = trim($value, " \n\r\t\v\x00" . implode('', $forbiddencharacters));

        return str_replace($forbiddencharacters, '_', $value);
    }

    /**
     * Validate e-mail string
     *
     * @param  string $email
     * @return string|null Returns e-mail if it's valid, null otherwise
     */
    private function validate_email($email) {
        $email = filter_var($email, FILTER_SANITIZE_EMAIL);
        return filter_var($email, FILTER_VALIDATE_EMAIL) ? $email : null;
    }

    /**
     * Get back information about a document
     *
     * @param string   $docid  Document ID
     * @return mixed           Return the document if succeed, an error message otherwise
     */
    public function get_document($docid) {
        $endpoint = '/api/private/document/' . $docid;
        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint));
        $error = $this->get_error_response($response, 200);
        if ($error === false) {
            return $response->data->document;
        }
        return $error;
    }

    /**
     * Delete a document on the Compilatio account
     *
     * @param  string   $docid  Document ID
     * @return boolean          Return true if succeed, an error message otherwise
     */
    public function delete_document($docid) {
        $endpoint = '/api/private/document/' . $docid;
        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint, 'delete'));

        if ($this->get_error_response($response, 200) === false) {
            return true;
        }
        return false;
    }

    /**
     * Create folder on Compilatio account
     *
     * @param   string         $name              Folder's name
     * @param   boolean        $defaultindexing   Folder's default indexing
     * @param   string         $analysistype      Analysis type
     * @param   string         $analysistime      Date for scheduled analysis
     * @param   int            $warningthreshold  Folder's warning threshold
     * @param   int            $criticalthreshold Folder's critical threshold
     * @return  string|false   Return the folder's ID, an error message otherwise, or false
     */
    public function set_folder(
        $name,
        $defaultindexing,
        $analysistype,
        $analysistime,
        $warningthreshold = 10,
        $criticalthreshold = 25
    ) {
        $endpoint = '/api/private/folder/create';
        $params = [
            'name' => $name,
            'thresholds' => [
                'warning' => $warningthreshold,
                'critical' => $criticalthreshold,
            ],
            'default_indexing' => $defaultindexing,
            'auto_analysis' => false,
            'scheduled_analysis_enabled' => false,
            'origin' => 'LMS-Moodle',
        ];

        if ($analysistype == 'auto') {
            $params['auto_analysis'] = true;
        } else if ($analysistype == 'planned') {
            $params['scheduled_analysis_enabled'] = true;
            $params['scheduled_analysis_date'] = $analysistime;
        }

        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint, 'post', json_encode($params)));

        if ($this->get_error_response($response, 201) === false) {
            return $response->data->folder->id;
        }
        return false;
    }

    /**
     * Update folder on Compilatio account
     *
     * @param   int      $folderid          Folder ID
     * @param   string   $name              Folder's name
     * @param   boolean  $defaultindexing   Folder's default indexing
     * @param   string   $analysistype      Analysis type
     * @param   string   $analysistime      Date for scheduled analysis
     * @param   int      $warningthreshold  Folder's warning threshold
     * @param   int      $criticalthreshold Folder's critical threshold
     * @return  string   Return true if succeed, an error message otherwise
     */
    public function update_folder(
        $folderid,
        $name,
        $defaultindexing,
        $analysistype,
        $analysistime,
        $warningthreshold = 10,
        $criticalthreshold = 25
    ) {
        $endpoint = '/api/private/folder/' . $folderid;

        $params = [
            'name' => $name,
            'thresholds' => [
                'warning' => $warningthreshold,
                'critical' => $criticalthreshold,
            ],
            'default_indexing' => $defaultindexing,
            'auto_analysis' => false,
            'scheduled_analysis_enabled' => false,
        ];

        if ($analysistype == 'auto') {
            $params['auto_analysis'] = true;
        } else if ($analysistype == 'planned') {
            $params['scheduled_analysis_enabled'] = true;
            $params['scheduled_analysis_date'] = $analysistime;
        }

        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint, 'patch', json_encode($params)));

        if ($this->get_error_response($response, 200) === false) {
            return true;
        }
        return false;
    }

    /**
     * Delete a folder on the Compilatio account
     *
     * @param string   $folderid  Folder ID
     * @return boolean            Return true if succeed, an error message otherwise
     */
    public function delete_folder($folderid) {
        $endpoint = '/api/private/folder/' . $folderid;
        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint, 'delete'));

        if ($this->get_error_response($response, 200) === false) {
            return true;
        }
        return false;
    }

    /**
     * Set the indexing state of a document
     *
     * @param   string  $docid      Document ID
     * @param   bool    $indexed    Indexing state
     * @return  mixed               Return true if succeed, an error message otherwise
     */
    public function set_indexing_state($docid, $indexed) {
        $endpoint = '/api/private/document/' . $docid;

        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint, 'patch', json_encode(['indexed' => $indexed])));

        if ($this->get_error_response($response, 200) === false) {
            return true;
        }
        return false;
    }

    /**
     * Get JWT to access a document report.
     *
     * @param  string $docid Document ID
     * @return string Return a JWT if succeed, an error otherwise
     */
    public function get_report_token($docid) {
        $endpoint = '/api/private/documents/' . $docid . '/report/jwt';

        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint, 'post'));

        if ($this->get_error_response($response, 201) === false) {
            return $response->data->jwt;
        }
        return false;
    }

    /**
     * Get back the PDF of a report
     *
     * @param  string $idreport Report ID
     * @param  string $lang     Language
     * @param  string $type     Report type
     * @return string           Return the PDF if succeed, an error message otherwise
     */
    public function get_pdf_report($idreport, $lang = 'en', $type = 'detailed') {
        global $CFG;

        $endpoint = '/api/private/report/anasim/' . $idreport . '/pdf/' . $lang . '/' . $type . '/';
        $filepath = $CFG->dataroot . '/temp/compilatio/' . $idreport . '_' . $lang . '_' . $type . '.pdf';

        $handle = fopen($filepath, 'wb');

        if ($this->build_curl_on_behalf_of_user($endpoint, 'download', null, $handle) == 200) {
            return $filepath;
        } else {
            return false;
        }
    }

    /**
     * Update ignored scores in report
     *
     * @param  string   $analysisid   Analysis ID
     * @param  array    $ignoredtypes Ignored scores
     * @return mixed    Return update_task_id if succeed, false otherwise
     */
    public function update_and_rebuild_report($analysisid, $ignoredtypes) {
        $endpoint = '/api/private/anasim/report/' . $analysisid;
        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint, 'patch', $ignoredtypes));

        if ($this->get_error_response($response, 200) === false) {
            $endpoint = '/api/private/anasim/report/' . $analysisid . '/rebuild';
            $response = json_decode($this->build_curl_on_behalf_of_user($endpoint, 'post'));

            if ($this->get_error_response($response, 200) === false) {
                return $response->data->update_task_id;
            }
        }
        return false;
    }

    /**
     * Get updated report
     *
     * @param  string   $analysisid    Analysis ID
     * @param  array    $updatetaskid  Update task ID
     * @return mixed    Return report if succeed, false otherwise
     */
    public function get_updated_report($analysisid, $updatetaskid) {
        $endpoint = '/api/private/report/anasim/' . $analysisid . '/is-updated/' . $updatetaskid;
        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            return $response->data->report;
        } else if ($response->status->code == 202) {
            sleep(1);
            return $this->get_updated_report($analysisid, $updatetaskid);
        }
        return false;
    }

    /**
     * Start an analyse of a document
     *
     * @param  string   $docid  Document ID
     * @return mixed    Return true if succeed, an error message otherwise
     */
    public function start_analyse($docid) {
        $endpoint = '/api/private/analysis/';
        $params = [
            'doc_id' => $docid,
            'tags' => [
                'stable',
            ],
            'params' => [
                'origin' => 'LMS-Moodle',
            ],
        ];

        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint, 'post', json_encode($params)));

        $error = $this->get_error_response($response, 201);
        if ($error === false) {
            return true;
        } else if (isset($response->errors->form[0])) {
            return $response->errors->form[0];
        }
        return $error;
    }

    /**
     * Get a list of the allowed file types by Compilatio.
     *
     * @return  array   Return an array of the different allowed file types
     */
    public function get_allowed_file_types() {
        $endpoint = '/api/public/file/allowed-extensions';

        $response = json_decode($this->build_curl($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            return $response->data;
        }
        return false;
    }

    /**
     * Post Moodle Configuration to Compilatio
     *
     * @param  string   $releasephp     PHP version
     * @param  string   $releasemoodle  Moodle version
     * @param  string   $releaseplugin  Plugin version
     * @param  string   $language       Language
     * @param  int      $cronfrequency  CRON frequency
     * @param  int      $instancekey    Instance key
     * @return mixed                    Return true if succeed, an error message otherwise
     */
    public function set_moodle_configuration($releasephp, $releasemoodle, $releaseplugin, $language, $cronfrequency, $instancekey) {
        $endpoint = '/api/private/moodle-configuration/';
        $params = [
            'php_version' => $releasephp,
            'moodle_version' => $releasemoodle,
            'compilatio_plugin_version' => $releaseplugin,
            'language' => $language,
            'cron_frequency' => $cronfrequency,
            'instance_key' => $instancekey,
        ];

        $response = json_decode($this->build_curl($endpoint, 'post', json_encode($params)));

        if ($this->get_error_response($response, 200) === false) {
            return true;
        }
        return false;
    }

    /**
     * Validate user's terms of service.
     *
     * @return boolean Return true if terms of service has been validated, false otherwise
     */
    public function validate_terms_of_service() {
        $endpoint = '/api/private/terms-of-service/validate';

        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            return $response->data->termsOfService_validated;
        }
        return false;
    }

    /**
     * Get zendesk jwt to authenticate user to help center.
     *
     * @return boolean Return jwt if succeed, false otherwise
     */
    public function get_zendesk_jwt() {
        $endpoint = '/api/private/user/zendesk/jwt';

        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            return $response->data->token;
        }
        return false;
    }

    /**
     * Get a list of Compilatio alerts.
     *
     * @return  array   Return an array of alerts
     */
    public function get_alerts() {
        $endpoint = '/api/public/alerts/moodle/' . get_config('plagiarism_compilatio', 'version');

        $response = json_decode($this->build_curl($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            return $response->data->alerts;
        }
        return [];
    }

    /**
     * Get a list of Compilatio marketing notifications.
     *
     * @param  string  $lang lang
     * @return array   Return an array of marketing notifications
     */
    public function get_marketing_notifications($lang) {
        $endpoint = '/api/private/marketing-notifications/' . $lang;

        $response = json_decode($this->build_curl_on_behalf_of_user($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            return $response->data->notifications;
        }

        return [];
    }

    /**
     * Get subscription info.
     *
     * @return  stdClass   Return subscription info.
     */
    public function get_subscription_info() {
        $endpoint = '/api/private/authentication/check-api-key';
        $response = json_decode($this->build_curl($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            foreach ($response->data->user->current_bundle->accesses as $access) {
                if ($access->resource == 'management') {
                    $magisterstandardbundleid = $access->bundle_id;
                }
            }
        }

        if (!isset($magisterstandardbundleid)) {
            return false;
        }

        $endpoint = '/api/private/subscription/bundles/' . $magisterstandardbundleid . '/last-subscription';

        $response = json_decode($this->build_curl($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            return $response->data->subscription;
        }

        return false;
    }

    /**
     * Get a Compilatio translation.
     *
     * @param  string  $lang  Language
     * @param  string  $key   Translation Key
     * @return string  Return the translation string
     */
    public function get_translation($lang, $key) {
        $endpoint = '/api/public/translation/last-version/' . $lang . '/key/' . $key;

        $response = json_decode($this->build_curl($endpoint));

        if ($this->get_error_response($response, 200) === false) {
            $translation = $response->data;
            foreach (explode('.', $key) as $object) {
                $translation = $translation->{$object};
            }
            return $translation;
        }
        return false;
    }

    /**
     * Returns a boolean to know if the API is under maintenance
     *
     * @return boolean
     */
    public function is_in_maintenance() {
        return get_config('plagiarism_compilatio', 'compilatio_maintenance') === '1';
    }

    /**
     * Get an eventually error message from an API response
     * Compare API response with expected HTTP code
     * @param  mixed $response           curl response
     * @param  int   $expectedstatuscode Expected HTTP code
     * @return false|string Returns false if expected HTTP code is found in API response, or a message otherwise
     */
    private function get_error_response($response, int $expectedstatuscode) {
        if (!isset($response->status->code, $response->status->message)) {
            return 'Error response status not found';
        } else if ($response->status->code === $expectedstatuscode) {
            return false;
        } else if ($response->status->code === 403) {
            foreach (($response->errors ?? []) as $error) {
                if (isset($error->key) && $error->key === 'need_terms_of_service_validation') {
                    if (!empty($this->userid)) {
                        $this->validate_terms_of_service();
                    }
                    return $error->key;
                }
            }
        } else if ($response->status->message === 'Forbidden ! Your read only API key cannot modify this resource') {
            set_config('read_only_apikey', 1, 'plagiarism_compilatio');
        }

        return $response->status->message;
    }

    /**
     * Execute curl request with the custom HTTP header X-LMS-USER-ID
     *
     * @param  string   $endpoint API endpoint
     * @param  string   $method   Desired action on endpoint
     * @param  string   $data     Data to be send in CURLOPT_POSTFIELDS
     * @param  resource $handle   File handler
     * @return string curl response
     */
    private function build_curl_on_behalf_of_user($endpoint, $method = null, $data = null, $handle = null) {
        global $DB, $USER;

        $header = [];

        $userid = $this->userid;

        if ($userid === null || $userid === '') {
            $userid = $DB->get_field('plagiarism_compilatio_user', 'compilatioid', ['userid' => 0]);
            if ($userid === false) {
                $user0compilatioemail = 'moodle-' . substr($this->apikey, 0, 10) . '@' . preg_replace('/^.*@/', '', $USER->email);

                $userid = $this->get_user_by_email($user0compilatioemail);

                if (!preg_match('/^[a-f0-9]{40}$/', $userid)) {
                    $userid = $this->set_user($USER->firstname, $USER->lastname, $user0compilatioemail);
                }

                if ($userid === false) {
                    return json_encode(['status' => ['code' => 500, 'message' => 'User could not be created']]);
                }

                $DB->insert_record('plagiarism_compilatio_user', (object) ['userid' => 0, 'compilatioid' => $userid]);
            }
        }

        // Plugin v2 docs management.
        $header[] = 'X-LMS-USER-ID: ' . $userid;
        return $this->build_curl($endpoint, $method, $data, $handle, $header);
    }

    /**
     * Execute curl request
     *
     * @param  string   $endpoint API endpoint
     * @param  string   $method   Desired action on endpoint
     * @param  string   $data     Data to be send in CURLOPT_POSTFIELDS
     * @param  resource $handle   File handler
     * @param  array    $header   HTTP headers
     * @return string curl response
     */
    private function build_curl($endpoint, $method = null, $data = null, $handle = null, $header = []) {
        global $CFG;

        $ch = curl_init();

        $params = [
            CURLOPT_URL => $this->urlrest . $endpoint,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
        ];

        $header[] = 'X-Auth-Token: ' . $this->apikey;

        if ($method !== 'upload') {
            $header[] = 'Content-Type: application/json';
        }
        $params[CURLOPT_HTTPHEADER] = $header;

        // Proxy settings.
        if (!empty($CFG->proxyhost)) {
            $params[CURLOPT_PROXY] = $CFG->proxyhost;

            $params[CURLOPT_HTTPPROXYTUNNEL] = false;

            if (!empty($CFG->proxytype) && ($CFG->proxytype == 'SOCKS5')) {
                $params[CURLOPT_PROXYTYPE] = CURLPROXY_SOCKS5;
            }

            if (!empty($CFG->proxyport)) {
                $params[CURLOPT_PROXYPORT] = $CFG->proxyport;
            }

            if (!empty($CFG->proxyuser) && !empty($CFG->proxypassword)) {
                $params[CURLOPT_PROXYUSERPWD] = $CFG->proxyuser . ':' . $CFG->proxypassword;
            }
        }

        // SSL certificate verification.
        if (get_config('plagiarism_compilatio', 'disable_ssl_verification') == 1) {
            $params[CURLOPT_SSL_VERIFYPEER] = false;
        }

        switch ($method) {
            case 'post':
                $params[CURLOPT_POST] = true;
                $params[CURLOPT_POSTFIELDS] = $data;
                break;
            case 'upload':
                $params[CURLOPT_POST] = true;
                $params[CURLOPT_POSTFIELDS] = $this->build_post_fields($data);
                break;
            case 'patch':
                $params[CURLOPT_CUSTOMREQUEST] = 'PATCH';
                $params[CURLOPT_POSTFIELDS] = $data;
                break;
            case 'delete':
                $params[CURLOPT_CUSTOMREQUEST] = 'DELETE';
                break;
            case 'download':
                $params[CURLOPT_FILE] = $handle;
                $params[CURLOPT_TIMEOUT] = 20;
                $params[CURLOPT_FOLLOWLOCATION] = true;
                break;
        }

        curl_setopt_array($ch, $params);

        $result = curl_exec($ch);

        if ($method !== 'download' && $this->check_if_under_maintenance($result)) {
            return json_encode(['status' => ['code' => 503, 'message' => 'Compilation services undergoing maintenance']]);
        }

        if ($method == 'download') {
            $result = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        }
        curl_close($ch);

        return $result;
    }

    /**
     * Returns an array containing data for curl request
     *
     * @param  mixed  $data
     * @param  string $existingkeys
     * @param  array  $returnarray
     * @return array
     */
    private function build_post_fields($data, $existingkeys = '', &$returnarray = []) {
        if (($data instanceof \CURLFile) || !(is_array($data) || is_object($data))) {
            $returnarray[$existingkeys] = $data;
            return $returnarray;
        } else {
            foreach ($data as $key => $item) {
                $this->build_post_fields($item, $existingkeys ? $existingkeys . "[$key]" : $key, $returnarray);
            }
            return $returnarray;
        }
    }

    /**
     * Check if Compilatio API is under maintenance by the CURL result.
     *
     * @param  mixed  $result
     * @return boolean
     */
    private function check_if_under_maintenance($result) {

        $decodedresult = json_decode($result);

        $isinmaintenance = $decodedresult->status->code === 503
            && isset($decodedresult->data->maintenance)
            && $decodedresult->data->maintenance;

        if ($isinmaintenance) {
            set_config('compilatio_maintenance', '1', 'plagiarism_compilatio');
            return true;
        }

        if (!$isinmaintenance &&
                (get_config('plagiarism_compilatio', 'compilatio_maintenance') === '1' ||
                !get_config('plagiarism_compilatio', 'compilatio_maintenance')
        )) {
            set_config('compilatio_maintenance', '0', 'plagiarism_compilatio');
        }

        return false;
    }
}
