// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Commands helper for the Moodle tiny_studietube plugin.
 *
 * @module      tiny_studietube/commands
 * @copyright   Webtjenesten LærIT ApS <kontakt@laerit.dk>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

import {getButtonImage} from 'editor_tiny/utils';
import {get_string as getString} from 'core/str';
import {
component,
        studietubeButtonName,
        icon,
        } from './common';

const handleAction = (editor) => {
    const handler = messageHandler(editor);
    window.addEventListener("message", handler, false);

    editor.windowManager.openUrl(
            {
                title: 'Skoletube/Studietube import',
                url: 'https://www.studietube.dk/app/bluebird/?passtype=complete&fitscale=true&urlref=' + encodeURI(window.location),
                width: 630,
                height: 540,
                buttons: [],
                onClose: function () {
                    window.removeEventListener("message", handler, false);
                }
            });
};

const messageHandler = (ed) => {
    return function handler(e) {
        if (e.origin !== "https://www.studietube.dk" &&
                e.origin !== "http://www.studietube.dk"
                && e.origin !== "http://www.skoletube.dk"
                && e.origin !== "https://www.skoletube.dk")
        {
            return false;
        }

        if (!/(ftp|http|https):\/\/(\w+:{0,1}\w*@)?(\S+)(:[0-9]+)?(\/|\/([\w#!:.?+=&%@!\-\/]))?/.test(e.data))
        {
            return false;
        }

        const content = JSON.parse(e.data);
        ed.insertContent(templateContent(content));
        ed.windowManager.close();
    };
};

const templateContent = (content) => {
    return '' +
            `<div class="stdsts_holder ` +
            `stdconfig-${content.vkey}-${content.extra.width}-${content.extra.height}-1 stdsts_fullsize" ` +
            `contenteditable="false" ` +
            `data-studietube-id="${content.vkey}" ` +
            `data-studietube-config="${content.extra.width}:${content.extra.height}:1">` +
            `<iframe src="//www.skoletube.dk/e/${content.vkey}/0" ` +
            `style="width:${content.extra.width}px;height:${content.extra.height}px;" ` +
            `allowfullscreen webkitallowfullscreen mozAllowFullScreen ` +
            `frameborder="0" allow="encrypted-media">` +
            `</iframe>` +
            `</div>`;

};


/**
 * Get the setup function for the buttons.
 *
 * This is performed in an async function which ultimately returns the registration function as the
 * Tiny.AddOnManager.Add() function does not support async functions.
 *
 * @returns {function} The registration function to call within the Plugin.add function.
 */
export const getSetup = async() => {
    const [
        studietubeButtonNameTitle,
        buttonImage,
    ] = await Promise.all([
        getString('button_studietube', component),
        getButtonImage('icon', component),
    ]);

    return (editor) => {
        // Register the Moodle SVG as an icon suitable for use as a TinyMCE toolbar button.
        editor.ui.registry.addIcon(icon, buttonImage.html);

        // Register the studietube Toolbar Button.
        editor.ui.registry.addButton(studietubeButtonName, {
            icon,
            tooltip: studietubeButtonNameTitle,
            onAction: () => handleAction(editor)
        });

        editor.on('BeforeSetContent', function (e) {
            if (e.content) {
                e.content = e.content.replace(
                        /<div class="(?:.*)stdsts_holder stdconfig-([A-Za-z0-9]+)-([0-9]+)-([0-9]+)-([01])([ A-Za-z0-9_\-=]+)"([ A-Za-z0-9="\-_]+)>(?:.*)<\/div>/g,
                        function (original, vkey, width, height, fs, classes, more) {
                            return ''+
                                    `<div class="stdsts_holder ` +
                                    `stdconfig-${vkey}-${width}-${height}-${fs}` +
                                    ` ${classes}" ${more}><iframe src="//www.skoletube.dk/e/${vkey}/0" ` +
                                    `style="width:${width}px;height:${height}px;"` +
                                    `allowfullscreen webkitallowfullscreen mozAllowFullScreen` +
                                    `frameborder="0" allow="encrypted-media">` +
                                    `</iframe></div>`;
                        }
                );
            }
        });
    };
};
