<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Participants metrics for a course.
 *
 * @package   report_edumetrics
 * @copyright 2026, Paulo Júnior <pauloa.junior@ufla.br>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require(__DIR__ . '/../../config.php');
require_once($CFG->libdir . '/adminlib.php');
require_once(__DIR__ . '/classes/filter_form.php');

$courseid = required_param('courseid', PARAM_INT);
$course = get_course($courseid);
require_login($course);

// Check if user has capability to view participants.
$context = context_course::instance($courseid);
require_capability('moodle/course:viewparticipants', $context);

// Set up the page.
$pageurl = new moodle_url($CFG->wwwroot . '/report/edumetrics/participants_metrics.php', ['courseid' => $courseid]);
$PAGE->set_url($pageurl);
$PAGE->set_context($context);
$PAGE->set_pagelayout('incourse');
$PAGE->set_title(get_string('pluginname', 'report_edumetrics') . ' | ' . get_string('course_report', 'report_edumetrics', format_string($course->fullname)));
echo $OUTPUT->header();

// Show heading.
$heading = get_string('course_report', 'report_edumetrics', format_string($course->fullname));
echo $OUTPUT->heading($heading);

// Show filters heading.
echo $OUTPUT->heading(get_string('filters', 'report_edumetrics'), 3);

// Create and process filter form.
$filterform = new report_edumetrics_filter_form(null, ['courseid' => $courseid, 'context' => $context]);

// Get filter data.
$filterdata = $filterform->get_data();
$selectedrole = 0;
$datefrom = 0;
$dateto = 0;

if ($filterdata) {
    if (isset($filterdata->role)) {
        $selectedrole = $filterdata->role;
    }
    if (isset($filterdata->datefrom)) {
        $datefrom = $filterdata->datefrom;
    }
    if (isset($filterdata->dateto)) {
        $dateto = $filterdata->dateto;
    }
}

// Display the form (after processing to show validation errors).
$filterform->display();

// Build participants list (includes students and teachers/tutors).
$users = get_enrolled_users($context, '', 0, 'u.id, u.firstname, u.lastname', 'u.lastname ASC, u.firstname ASC');

$participants = [];
$active_count = 0;
$inactive_count = 0;

// Get participation data from the plugin table
global $DB;
$participationdata = [];

// Build dynamic filters for participation records.
$params = ['courseid' => $courseid];
$select = 'courseid = :courseid';
if ($selectedrole > 0) {
    $select .= ' AND roleid = :roleid';
    $params['roleid'] = $selectedrole;
}
if (!empty($datefrom)) {
    $select .= ' AND event_date >= :datefrom';
    $params['datefrom'] = $datefrom;
}
if (!empty($dateto)) {
    $select .= ' AND event_date <= :dateto';
    $params['dateto'] = $dateto;
}

$records = $DB->get_records_select('report_edumetrics_participation', $select, $params);

foreach ($records as $record) {
    if (!isset($participationdata[$record->userid])) {
        $participationdata[$record->userid] = (object) [
            'resource_access' => 0,
            'forum_interactions' => 0,
            'activity_submissions' => 0,
        ];
    }
    $participationdata[$record->userid]->resource_access += $record->resource_access;
    $participationdata[$record->userid]->forum_interactions += $record->forum_interactions;
    $participationdata[$record->userid]->activity_submissions += $record->activity_submissions;
}

foreach ($users as $user) {
    // Filter by role if selected.
    if ($selectedrole > 0) {
        $userroles = get_user_roles($context, $user->id);
        $hasrole = false;
        foreach ($userroles as $userrole) {
            if ($userrole->roleid == $selectedrole) {
                $hasrole = true;
                break;
            }
        }
        if (!$hasrole) {
            continue;
        }
    }
    
    // Get real data from participation table
    if (isset($participationdata[$user->id])) {
        $resource_access = $participationdata[$user->id]->resource_access;
        $forum_interactions = $participationdata[$user->id]->forum_interactions;
        $activity_submissions = $participationdata[$user->id]->activity_submissions;
    } else {
        $resource_access = 0;
        $forum_interactions = 0;
        $activity_submissions = 0;
    }
    
    // Determine if participant is active (has at least one interaction of any type)
    $is_active = ($resource_access > 0 || $forum_interactions > 0 || $activity_submissions > 0);
    
    if ($is_active) {
        $active_count++;
    } else {
        $inactive_count++;
    }
    
    $participants[] = (object) [
        'id' => $user->id,
        'firstname' => $user->firstname,
        'lastname' => $user->lastname,
        'resource_access' => $resource_access,
        'forum_interactions' => $forum_interactions,
        'activity_submissions' => $activity_submissions,
        'is_active' => $is_active,
        'message_url' => $is_active ? '' : (new moodle_url('/message/index.php', ['user1' => $USER->id, 'user2' => $user->id]))->out(false),
    ];
}

// Prepare template context.
$templatecontext = [
    'participants' => $participants,
];

// Render participants list template.
echo $OUTPUT->render_from_template('report_edumetrics/participants_metrics', $templatecontext);

echo $OUTPUT->footer();
