<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Event observer for tracking participant activity in courses.
 *
 * @package   report_edumetrics
 * @copyright 2026, Paulo Júnior <pauloa.junior@ufla.br>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_edumetrics\event;

defined('MOODLE_INTERNAL') || die();

/**
 * Observer class for course participation events.
 */
class observer {

    /**
     * Get the primary role of a user in a course context.
     * If user has multiple roles, returns the first one found.
     *
     * @param int $userid The user ID
     * @param int $courseid The course ID
     * @return int The role ID, or 0 if no role found
     */
    private static function get_user_primary_role($userid, $courseid) {
        global $DB;

        $context = \context_course::instance($courseid);
        $userroles = get_user_roles($context, $userid);

        if (!empty($userroles)) {
            $role = reset($userroles);
            return $role->roleid;
        }

        return 0;
    }

    /**
     * Handle course_module_viewed event for resource modules (url, file, etc).
     *
     * @param \core\event\course_module_viewed $event The event object
     */
    public static function course_module_viewed(\core\event\course_module_viewed $event) {
        global $DB;

        try {
            // Get the course module to determine the module type.
            $cm = get_coursemodule_from_id('', $event->contextinstanceid, 0, false, MUST_EXIST);
            $modulename = $cm->modname;

            // Only track resource modules (url, file).
            $resourcemodules = ['url', 'resource'];
            if (!in_array($modulename, $resourcemodules)) {
                return;
            }
            $userid = $event->userid;
            $courseid = $event->courseid;

            // Get course and its category.
            $course = get_course($courseid);
            $categoryid = $course->category;

            // Get user's primary role
            $roleid = self::get_user_primary_role($userid, $courseid);
            if ($roleid === 0) {
                return;
            }

            // Get the date at midnight (start of day).
            $eventdate = strtotime('midnight', $event->timecreated);
            $now = time();

            $record = $DB->get_record('report_edumetrics_participation', [
                'courseid' => $courseid,
                'userid' => $userid,
                'roleid' => $roleid,
                'event_date' => $eventdate
            ], '*', IGNORE_MISSING);

            if ($record) {
                // Update existing record - increment and update version.
                $record->resource_access++;
                $record->version++;
                $record->timemodified = $now;
                $DB->update_record('report_edumetrics_participation', $record);
            } else {
                // Create new record.
                $newrecord = (object) [
                    'courseid' => $courseid,
                    'categoryid' => $categoryid,
                    'userid' => $userid,
                    'roleid' => $roleid,
                    'event_date' => $eventdate,
                    'resource_access' => 1,
                    'forum_interactions' => 0,
                    'activity_submissions' => 0,
                    'version' => 1,
                    'timecreated' => $now,
                    'timemodified' => $now,
                ];
                $DB->insert_record('report_edumetrics_participation', $newrecord);
            }
        } catch (\Exception $e) {
            // Ignore errors - if we can't save the event, just skip it.
            return;
        }
    }

    /**
     * Handle forum interaction events (discussion_created, post_created).
     *
     * @param \core\event\base $event The event object
     */
    public static function forum_interaction(\core\event\base $event) {
        global $DB;

        try {
            $userid = $event->userid;
            $courseid = $event->courseid;

            // Get course and its category.
            $course = get_course($courseid);
            $categoryid = $course->category;

            // Get user's primary role
            $roleid = self::get_user_primary_role($userid, $courseid);
            if ($roleid === 0) {
                return;
            }

            // Get the date at midnight (start of day).
            $eventdate = strtotime('midnight', $event->timecreated);
            $now = time();

            $record = $DB->get_record('report_edumetrics_participation', [
                'courseid' => $courseid,
                'userid' => $userid,
                'roleid' => $roleid,
                'event_date' => $eventdate
            ], '*', IGNORE_MISSING);

            if ($record) {
                // Update existing record - increment and update version.
                $record->forum_interactions++;
                $record->version++;
                $record->timemodified = $now;
                $DB->update_record('report_edumetrics_participation', $record);
            } else {
                // Create new record.
                $newrecord = (object) [
                    'courseid' => $courseid,
                    'categoryid' => $categoryid,
                    'userid' => $userid,
                    'roleid' => $roleid,
                    'event_date' => $eventdate,
                    'resource_access' => 0,
                    'forum_interactions' => 1,
                    'activity_submissions' => 0,
                    'version' => 1,
                    'timecreated' => $now,
                    'timemodified' => $now,
                ];
                $DB->insert_record('report_edumetrics_participation', $newrecord);
            }
        } catch (\Exception $e) {
            return;
        }
    }

    /**
     * Handle assign_submission_created event for activity submissions.
     *
     * @param \mod_assign\event\submission_created $event The event object
     */
    public static function assign_submission_created(\mod_assign\event\submission_created $event) {
        global $DB;

        try {
            $userid = $event->relateduserid;
            $courseid = $event->courseid;

            // Get course and its category.
            $course = get_course($courseid);
            $categoryid = $course->category;

            // Get user's primary role
            $roleid = self::get_user_primary_role($userid, $courseid);
            if ($roleid === 0) {
                return;
            }

            // Get the date at midnight (start of day).
            $eventdate = strtotime('midnight', $event->timecreated);
            $now = time();

            $record = $DB->get_record('report_edumetrics_participation', [
                'courseid' => $courseid,
                'userid' => $userid,
                'roleid' => $roleid,
                'event_date' => $eventdate
            ], '*', IGNORE_MISSING);

            if ($record) {
                // Update existing record - increment and update version.
                $record->activity_submissions++;
                $record->version++;
                $record->timemodified = $now;
                $DB->update_record('report_edumetrics_participation', $record);
            } else {
                // Create new record.
                $newrecord = (object) [
                    'courseid' => $courseid,
                    'categoryid' => $categoryid,
                    'userid' => $userid,
                    'roleid' => $roleid,
                    'event_date' => $eventdate,
                    'resource_access' => 0,
                    'forum_interactions' => 0,
                    'activity_submissions' => 1,
                    'version' => 1,
                    'timecreated' => $now,
                    'timemodified' => $now,
                ];
                $DB->insert_record('report_edumetrics_participation', $newrecord);
            }
        } catch (\Exception $e) {
            // Ignore errors - if we can't save the event, just skip it.
            return;
        }
    }
}
