/* Copyright 2013 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* globals DEFAULT_PREFERENCES, chrome */

'use strict';

(function(root, factory) {
    if (typeof define === 'function' && define.amd) {
        define('pdfjs-web/preferences', ['exports'], factory);
    } else if (typeof exports !== 'undefined') {
        factory(exports);
    } else {
        factory((root.pdfjsWebPreferences = {}));
    }
}(this, function(exports) {

//#if PRODUCTION
//var defaultPreferences = Promise.resolve(
//#include $ROOT/web/default_preferences.json
//);
//#else
    var defaultPreferences = new Promise(function(resolve) {
        if (DEFAULT_PREFERENCES) {
            resolve(DEFAULT_PREFERENCES);
            return;
        }
        document.addEventListener('defaultpreferencesloaded', function loaded() {
            resolve(DEFAULT_PREFERENCES);
            document.removeEventListener('defaultpreferencesloaded', loaded);
        });
    });

//#endif

    function cloneObj(obj) {
        var result = {};
        for (var i in obj) {
            if (Object.prototype.hasOwnProperty.call(obj, i)) {
                result[i] = obj[i];
            }
        }
        return result;
    }

    /**
     * Preferences - Utility for storing persistent settings.
     *   Used for settings that should be applied to all opened documents,
     *   or every time the viewer is loaded.
     */
    var Preferences = {
        prefs                        : null,
        isInitializedPromiseResolved : false,
        initializedPromise           : null,

        /**
         * Initialize and fetch the current preference values from storage.
         * @return {Promise} A promise that is resolved when the preferences
         *                   have been initialized.
         */
        initialize : function preferencesInitialize() {
            return this.initializedPromise = defaultPreferences.then(
                function(defaults) {

                    Object.defineProperty(this, 'defaults', {
                        value        : Object.freeze(defaults),
                        writable     : false,
                        enumerable   : true,
                        configurable : false
                    });

                    this.prefs = cloneObj(defaults);
                    return this._readFromStorage(defaults);
                }.bind(this)).then(function(prefObj) {
                this.isInitializedPromiseResolved = true;
                if (prefObj) {
                    this.prefs = prefObj;
                }
            }.bind(this));
        },

        /**
         * Stub function for writing preferences to storage.
         * NOTE: This should be overridden by a build-specific function defined below.
         * @param {Object} prefObj The preferences that should be written to storage.
         * @return {Promise} A promise that is resolved when the preference values
         *                   have been written.
         */
        _writeToStorage : function preferences_writeToStorage(prefObj) {
            return Promise.resolve();
        },

        /**
         * Stub function for reading preferences from storage.
         * NOTE: This should be overridden by a build-specific function defined below.
         * @param {Object} prefObj The preferences that should be read from storage.
         * @return {Promise} A promise that is resolved with an {Object} containing
         *                   the preferences that have been read.
         */
        _readFromStorage : function preferences_readFromStorage(prefObj) {
            return Promise.resolve();
        },

        /**
         * Reset the preferences to their default values and update storage.
         * @return {Promise} A promise that is resolved when the preference values
         *                   have been reset.
         */
        reset : function preferencesReset() {
            return this.initializedPromise.then(function() {
                this.prefs = cloneObj(this.defaults);
                return this._writeToStorage(this.defaults);
            }.bind(this));
        },

        /**
         * Replace the current preference values with the ones from storage.
         * @return {Promise} A promise that is resolved when the preference values
         *                   have been updated.
         */
        reload : function preferencesReload() {
            return this.initializedPromise.then(function() {
                this._readFromStorage(this.defaults).then(function(prefObj) {
                    if (prefObj) {
                        this.prefs = prefObj;
                    }
                }.bind(this));
            }.bind(this));
        },

        /**
         * Set the value of a preference.
         * @param {string} name The name of the preference that should be changed.
         * @param {boolean|number|string} value The new value of the preference.
         * @return {Promise} A promise that is resolved when the value has been set,
         *                   provided that the preference exists and the types match.
         */
        set : function preferencesSet(name, value) {
            return this.initializedPromise.then(function() {
                if (this.defaults[name] === undefined) {
                    throw new Error('preferencesSet: \'' + name + '\' is undefined.');
                } else if (value === undefined) {
                    throw new Error('preferencesSet: no value is specified.');
                }
                var valueType = typeof value;
                var defaultType = typeof this.defaults[name];

                if (valueType !== defaultType) {
                    if (valueType === 'number' && defaultType === 'string') {
                        value = value.toString();
                    } else {
                        throw new Error('Preferences_set: \'' + value + '\' is a \"' +
                            valueType + '\", expected \"' + defaultType + '\".');
                    }
                } else {
                    if (valueType === 'number' && (value | 0) !== value) {
                        throw new Error('Preferences_set: \'' + value +
                            '\' must be an \"integer\".');
                    }
                }
                this.prefs[name] = value;
                return this._writeToStorage(this.prefs);
            }.bind(this));
        },

        /**
         * Get the value of a preference.
         * @param {string} name The name of the preference whose value is requested.
         * @return {Promise} A promise that is resolved with a {boolean|number|string}
         *                   containing the value of the preference.
         */
        get : function preferencesGet(name) {
            return this.initializedPromise.then(function() {
                var defaultValue = this.defaults[name];

                if (defaultValue === undefined) {
                    throw new Error('preferencesGet: \'' + name + '\' is undefined.');
                } else {
                    var prefValue = this.prefs[name];

                    if (prefValue !== undefined) {
                        return prefValue;
                    }
                }
                return defaultValue;
            }.bind(this));
        }
    };

//#if !(FIREFOX || MOZCENTRAL || CHROME)
    Preferences._writeToStorage = function(prefObj) {
        return new Promise(function(resolve) {
            localStorage.setItem('pdfjs.preferences', JSON.stringify(prefObj));
            resolve();
        });
    };

    Preferences._readFromStorage = function(prefObj) {
        return new Promise(function(resolve) {
            var readPrefs = JSON.parse(localStorage.getItem('pdfjs.preferences'));
            resolve(readPrefs);
        });
    };
//#endif

    exports.Preferences = Preferences;
}));
