<?php
// This file is part of Level Up XP.
//
// Level Up XP is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Level Up XP is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Level Up XP.  If not, see <https://www.gnu.org/licenses/>.
//
// https://levelup.plus

/**
 * Admin config.
 *
 * @package    block_xp
 * @copyright  2017 Frédéric Massart
 * @author     Frédéric Massart <fred@branchup.tech>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_xp\local\config;

/**
 * Admin config.
 *
 * Note that the settings page generated by Moodle core may not reflect
 * the values returned by this class, and this is because they do not fall
 * back on the defaults when values are not set, but we do. And we do it right.
 *
 * We override the standard mdl_config implementation in order to cater
 * for the legacy value $CFG->block_xp_context which sits in a different namespace.
 *
 * @package    block_xp
 * @copyright  2017 Frédéric Massart
 * @author     Frédéric Massart <fred@branchup.tech>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class admin_config extends mdl_config {

    /** @var config The defaults. */
    private $defaults;

    /**
     * Constructor.
     *
     * @param config $defaults The defaults.
     */
    public function __construct(config $defaults) {
        parent::__construct('block_xp', $defaults);
        $this->defaults = $defaults;
    }

    /**
     * Get a value.
     *
     * @param string $name The name.
     * @return mixed
     */
    public function get($name) {
        $this->validate($name);

        // Legacy hack.
        if ($name === 'context') {
            $value = get_config('core', 'block_xp_context');
            if ($value === false) {
                return $this->defaults->get('context');
            }
            return $value;
        }

        return parent::get($name);
    }

    /**
     * Get all config.
     *
     * @return array
     */
    public function get_all() {
        global $CFG;

        $all = parent::get_all();
        if (isset($CFG->block_xp_context)) {
            $all['context'] = $CFG->block_xp_context;
        }

        return $all;
    }

    /**
     * Set a value.
     *
     * @param string $name Name of the config.
     * @param mixed $value The value.
     */
    public function set($name, $value) {
        $this->validate($name);

        // Legacy hack.
        if ($name === 'context') {
            return set_config('block_xp_context', $value);
        }

        return parent::set($name, $value);
    }

    /**
     * Set many.
     *
     * @param array $values Keys are config names, and values are values.
     */
    public function set_many(array $values) {
        if (isset($values['context'])) {
            $this->set('context', $values['context']);
            unset($values['context']);
        }
        parent::set_many($values);
    }

}
