<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Course and category data handler class for theme_nice.
 *
 * This class provides helper methods to retrieve detailed information
 * about courses and categories used by the Nice theme’s custom blocks
 * and UI components.
 *
 * @package     theme_nice
 * @copyright   2025 Nice Learning <support@docs.nicelearning.org>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

require_once($CFG->dirroot. '/course/renderer.php');
include_once($CFG->dirroot . '/course/lib.php');

class niceCourseHandler {

    /**
     * Fetches detailed course information by course ID.
     *
     * @param int $courseId The course ID.
     * @return stdClass|null Course details object, or null if course does not exist.
     */
    public function niceGetCourseDetails($courseId) {
        global $CFG, $COURSE, $USER, $DB, $SESSION, $SITE, $PAGE, $OUTPUT;

        $courseId = (int)$courseId;

        // Check if the course record exists.
        if ($DB->record_exists('course', array('id' => $courseId))) {
        
        // Prepare helper classes.
        $niceCourse = new \stdClass();
        $chelper = new coursecat_helper();
        $courseContext = context_course::instance($courseId);

        $courseRecord = $DB->get_record('course', array('id' => $courseId));
        $courseElement = new core_course_list_element($courseRecord);

        // Extract course details.
        $courseId = $courseRecord->id;
        $courseShortName = $courseRecord->shortname;
        $courseFullName = $courseRecord->fullname;
        $courseSummary = $chelper->get_course_formatted_summary($courseElement, array('noclean' => true, 'para' => false));
        $courseFormat = $courseRecord->format;
        $courseAnnouncements = $courseRecord->newsitems;
        $courseStartDate = $courseRecord->startdate;
        $courseEndDate = $courseRecord->enddate;
        $courseVisible = $courseRecord->visible;
        $courseCreated = $courseRecord->timecreated;
        $courseUpdated = $courseRecord->timemodified;
        $courseRequested = $courseRecord->requested;
        $courseEnrolmentCount = count_enrolled_users($courseContext);
        $course_is_enrolled = is_enrolled($courseContext, $USER->id, '', true);

        // Fetch category info.
        $categoryId = $courseRecord->category;

        try {
            $courseCategory = core_course_category::get($categoryId);
            $categoryName = $courseCategory->get_formatted_name();
            $categoryUrl = $CFG->wwwroot . '/course/index.php?categoryid='.$categoryId;
        } catch (Exception $e) {
            $courseCategory = "";
            $categoryName = "";
            $categoryUrl = "";
        }

        // Generate enrolment and course URLs.
        $enrolmentLink = $CFG->wwwroot . '/enrol/index.php?id=' . $courseId;
        $courseUrl = new moodle_url('/course/view.php', array('id' => $courseId));


        // Check for payment options (e.g. PayPal).
        $enrolInstances = enrol_get_instances($courseId, true);

        foreach($enrolInstances as $singleenrolInstances){
          if($singleenrolInstances->enrol == 'paypal'){
              $course_price = $singleenrolInstances->cost;
              $course_currency = $singleenrolInstances->currency;
          }else{
              $course_price = '';
              $course_currency = '';
          }
        }
        
        $niceCourseContacts = array();
        if ($courseElement->has_course_contacts()) {
            foreach ($courseElement->get_course_contacts() as $key => $courseContact) {
            $niceCourseContacts[$key] = new \stdClass();
            $niceCourseContacts[$key]->userId = $courseContact['user']->id;
            $niceCourseContacts[$key]->username = $courseContact['user']->username;
            $niceCourseContacts[$key]->name = $courseContact['user']->firstname . ' ' . $courseContact['user']->lastname;
            $niceCourseContacts[$key]->role = $courseContact['role']->displayname;
            $niceCourseContacts[$key]->profileUrl = new moodle_url('/user/view.php', array('id' => $courseContact['user']->id, 'course' => SITEID));
            }
        }


        // Process course overview image.
        $contentimages = $contentfiles = $CFG->wwwroot . '/theme/nice/images/niceBg.png';
        foreach ($courseElement->get_course_overviewfiles() as $file) {
            $isimage = $file->is_valid_image();
            $url = file_encode_url("{$CFG->wwwroot}/pluginfile.php",
                    '/'. $file->get_contextid(). '/'. $file->get_component(). '/'.
                    $file->get_filearea(). $file->get_filepath(). $file->get_filename(), !$isimage);
            if ($isimage) {
                $contentimages = $url;
            } 
        }

        // Map data to niceCourse object.
        $niceCourse->courseId = $courseId;
        $niceCourse->enrolments = $courseEnrolmentCount;
        $niceCourse->categoryId = $categoryId;
        $niceCourse->categoryName = $categoryName;
        $niceCourse->categoryUrl = $categoryUrl;
        $niceCourse->shortName = format_text($courseShortName, FORMAT_HTML, array('filter' => true));
        $niceCourse->fullName = format_text($courseFullName, FORMAT_HTML, array('filter' => true));
        $niceCourse->summary = $courseSummary;
        $niceCourse->imageUrl = $contentimages;
        $niceCourse->format = $courseFormat;
        $niceCourse->announcements = $courseAnnouncements;
        $niceCourse->startDate = userdate($courseStartDate, get_string('strftimedatefullshort', 'langconfig'));
        $niceCourse->endDate = userdate($courseEndDate, get_string('strftimedatefullshort', 'langconfig'));
        $niceCourse->visible = $courseVisible;
        $niceCourse->created = userdate($courseCreated, get_string('strftimedatefullshort', 'langconfig'));
        $niceCourse->updated = userdate($courseUpdated, get_string('strftimedatefullshort', 'langconfig'));
        $niceCourse->requested = $courseRequested;
        $niceCourse->enrolmentLink = $enrolmentLink;
        $niceCourse->url = $courseUrl;
        $niceCourse->teachers = $niceCourseContacts;
        $niceCourse->course_price = $course_price;
        $niceCourse->course_currency = $course_currency;
        $niceCourse->course_is_enrolled = $course_is_enrolled;

        // Prepare a render object with minimal HTML.
        $niceRender = new \stdClass();
        $niceRender->updatedDate        =     userdate($courseUpdated, get_string('strftimedatefullshort', 'langconfig'));
        $niceRender->title              =     '<h3><a href="'. $niceCourse->url .'">'. $niceCourse->fullName .'</a></h3>';
        $niceRender->fullName           = $niceCourse->fullName;
        $niceRender->url                = $niceCourse->url;
        $niceRender->courseImage        =     '<img class="" src="'. $contentimages .'" alt="'.$niceCourse->fullName.'">';
        $niceRender->ImageUrl = $contentimages;
        /* @niceBreak */
        $niceCourse->niceRender = $niceRender;
        return $niceCourse;
        }
        return null;
    }

      /**
     * Fetches a course's formatted summary text.
     *
     * @param int $courseId Course ID.
     * @param int|null $maxLength Optional maximum length of summary.
     * @return string|null Formatted summary text or null if unavailable.
     */
    public function niceGetCourseDescription($courseId, $maxLength){
        global $CFG, $COURSE, $USER, $DB, $SESSION, $SITE, $PAGE, $OUTPUT;
    
        if ($DB->record_exists('course', array('id' => $courseId))) {
        $chelper = new coursecat_helper();
        $courseContext = context_course::instance($courseId);
    
        $courseRecord = $DB->get_record('course', array('id' => $courseId));
        $courseElement = new core_course_list_element($courseRecord);
    
        if ($courseElement->has_summary()) {
            $courseSummary = $chelper->get_course_formatted_summary($courseElement, array('noclean' => false, 'para' => false));
            if($maxLength != null) {
            if (strlen($courseSummary) > $maxLength) {
                $courseSummary = wordwrap($courseSummary, $maxLength);
                $courseSummary = substr($courseSummary, 0, strpos($courseSummary, "\n")) . ' ...';
            }
            }
            return $courseSummary;
        }
    
        }
        return null;
    }

    /**
     * Lists all course categories and their names.
     *
     * @return array Array of category names indexed by category ID.
     */
    public function niceListCategories(){
        global $DB, $CFG;
        $topcategory = core_course_category::top();
        $topcategorykids = $topcategory->get_children();
        $areanames = array();
        foreach ($topcategorykids as $areaid => $topcategorykids) {
            $areanames[$areaid] = $topcategorykids->get_formatted_name();
            foreach($topcategorykids->get_children() as $k=>$child){
                $areanames[$k] = $child->get_formatted_name();
            }
        }
        return $areanames;
    }

    /**
     * Fetches detailed category information by category ID.
     *
     * @param int $categoryId Category ID.
     * @return stdClass|null Category details object or null if not found.
     */
    public function niceGetCategoryDetails($categoryId){
        global $CFG, $COURSE, $USER, $DB, $SESSION, $SITE, $PAGE, $OUTPUT;
    
        if ($DB->record_exists('course_categories', array('id' => $categoryId))) {
    
          $categoryRecord = $DB->get_record('course_categories', array('id' => $categoryId));
    
          $chelper = new coursecat_helper();
          $categoryObject = core_course_category::get($categoryId);
    
          $niceCategory = new \stdClass();
    
          $categoryId = $categoryRecord->id;
          $categoryName = format_text($categoryRecord->name, FORMAT_HTML, array('filter' => true));
          $categoryDescription = $chelper->get_category_formatted_description($categoryObject);
    
          $categorySummary = format_string($categoryRecord->description, $striplinks = true,$options = null);
          $isVisible = $categoryRecord->visible;
          $categoryUrl = $CFG->wwwroot . '/course/index.php?categoryid=' . $categoryId;
          $categoryCourses = $categoryObject->get_courses();
          $categoryCoursesCount = count($categoryCourses);
    
          // Fetch subcategories.
          $categoryGetSubcategories = [];
          $categorySubcategories = [];
          if (!$chelper->get_categories_display_option('nodisplay')) {
            $categoryGetSubcategories = $categoryObject->get_children($chelper->get_categories_display_options());
          }
          foreach($categoryGetSubcategories as $k=>$niceSubcategory) {
            $niceSubcat = new \stdClass();
            $niceSubcat->id = $niceSubcategory->id;
            $niceSubcat->name = $niceSubcategory->name;
            $niceSubcat->description = $niceSubcategory->description;
            $niceSubcat->depth = $niceSubcategory->depth;
            $niceSubcat->coursecount = $niceSubcategory->coursecount;
            $categorySubcategories[$niceSubcategory->id] = $niceSubcat;
          }
    
          $categorySubcategoriesCount = count($categorySubcategories);
    
          // Attempt to extract an image from the category description HTML.
          $outputimage = '';
          $description = $chelper->get_category_formatted_description($categoryObject);
          $src = "";
          if ($description) {
            $dom = new DOMDocument();
            $dom->loadHTML($description);
            $xpath = new DOMXPath($dom);
            $src = $xpath->evaluate("string(//img/@src)");
          }
          if ($src && $description){
            $outputimage = $src;
          } else {
            // Fallback: grab the first course overview image in the category.
            foreach($categoryCourses as $child_course) {
              if ($child_course === reset($categoryCourses)) {
                foreach ($child_course->get_course_overviewfiles() as $file) {
                  if ($file->is_valid_image()) {
                    $imagepath = '/' . $file->get_contextid() . '/' . $file->get_component() . '/' . $file->get_filearea() . $file->get_filepath() . $file->get_filename();
                    $imageurl = file_encode_url($CFG->wwwroot . '/pluginfile.php', $imagepath, false);
                    $outputimage  =  $imageurl;
                    // Use the first image found.
                    break;
                  }
                }
              }
            }
          }
    
          // Map category details to the object.
          $niceCategory->categoryId = $categoryId;
          $niceCategory->categoryName = $categoryName;
          $niceCategory->categoryDescription = $categoryDescription;
          $niceCategory->categorySummary = $categorySummary;
          $niceCategory->isVisible = $isVisible;
          $niceCategory->categoryUrl = $categoryUrl;
          $niceCategory->courseImage = $outputimage;
          $niceCategory->ImageUrl = $outputimage;
          $niceCategory->courses = $categoryCourses;
          $niceCategory->coursesCount = $categoryCoursesCount;
          $niceCategory->subcategories = $categorySubcategories;
          $niceCategory->subcategoriesCount = $categorySubcategoriesCount;
          return $niceCategory;
    
        }
      }

    /**
     * Fetches example categories and their details.
     *
     * @param int $maxNum Maximum number of categories to return.
     * @return array Array of category detail objects.
     */
    public function niceGetExampleCategories($maxNum) {
        global $CFG, $DB;
    
        $niceCategories = $DB->get_records('course_categories', array(), $sort='', $fields='*', $limitfrom=0, $limitnum=$maxNum);
    
        $niceReturn = array();
        foreach ($niceCategories as $niceCategory) {
        $niceReturn[] = $this->niceGetCategoryDetails($niceCategory->id);
        }
        return $niceReturn;
    }

    /**
     * Fetches example category IDs only.
     *
     * @param int $maxNum Maximum number of categories to return.
     * @return array Array of category IDs.
     */
    public function niceGetExampleCategoriesIds($maxNum) {
        global $CFG, $DB;
    
        $niceCategories = $this->niceGetExampleCategories($maxNum);
    
        $niceReturn = array();
        foreach ($niceCategories as $key => $niceCategory) {
          $niceReturn[] = $niceCategory->categoryId;
        }
        return $niceReturn;
      }
}
