<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Theme context and helper initialization for theme_nice.
 *
 * Prepares global variables, theme settings, and context data
 * shared across layout files and Mustache templates in theme_nice.
 *
 * @package     theme_nice
 * @copyright   2025 Nice Learning <support@docs.nicelearning.org>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

global $USER, $CFG, $SESSION, $OUTPUT, $COURSE, $DB;

require_once($CFG->dirroot. '/theme/nice/inc/nice_page_handler.php');

// Fetch navbar logo width from theme settings.
$navbar_logo_width = get_config('theme_nice', 'navbar_logo_width');
if (empty($navbar_logo_width)) {
  $navbar_logo_width = ''; 
}

// Check if the user can access admin settings.
$canaccessadmin = has_capability('moodle/site:config', context_system::instance(), $USER);
$templatecontext['canaccessadmin'] = $canaccessadmin;

// Site-wide search URL.
$searchurl = new moodle_url('/search/index.php');

// Determine global context and user status.
$context = context_system::instance();
$isguest = isguestuser(); // Checks if the current user is a guest.

// Add block button (for editors/admins).
$addblockbutton = $OUTPUT->addblockbutton();

// Determine page heading based on context (e.g. blog, activity).
$nicePageHandler = new nicePageHandler();
$pageheading = $nicePageHandler->niceGetPageTitle();

// Determine user role class for body tag styling.
if (is_siteadmin()) {$user_status = 'role-supreme';} else {$user_status = 'role-standard';}

// Prepare secondary navigation data if present.
$secondarynavigation = false;
$overflow = '';
if ($PAGE->has_secondary_navigation()) {
    $tablistnav = $PAGE->has_tablist_secondary_navigation();
    $moremenu = new \core\navigation\output\more_menu($PAGE->secondarynav, 'nav-tabs', true, $tablistnav);
    $secondarynavigation = $moremenu->export_for_template($OUTPUT);
    $overflowdata = $PAGE->secondarynav->get_overflow_menu_data();
    if (!is_null($overflowdata)) {
        $overflow = $overflowdata->export_for_template($OUTPUT);
    }
}

// Prepare primary navigation data.
$primary = new core\navigation\output\primary($PAGE);
$renderer = $PAGE->get_renderer('core');
$primarymenu = $primary->export_for_template($renderer);

// Check if main settings menu should appear in the header.
$buildregionmainsettings = !$PAGE->include_region_main_settings_in_header_actions()  && !$PAGE->has_secondary_navigation();
$regionmainsettingsmenu = $buildregionmainsettings ? $OUTPUT->region_main_settings_menu() : false;

// Activity header content for the page (if any).
$header = $PAGE->activityheader;
$headercontent = $header->export_for_template($renderer);

// Authentication details.
$login_url  = get_login_url();
$isloggedin = isloggedin();

// Prepare body classes.
$niceUserBodyClass = 'nice_body_class';
$extraclasses = array(
    'nice_no_class',
    $user_status,
    $niceUserBodyClass
);
// Retrieve blocks in side-pre (right sidebar) and left regions.
$blockshtml = $OUTPUT->blocks('side-pre');
$leftblocks = $OUTPUT->blocks('left');


// These variables will be deprecated in future versions:
$hasblocks = strpos($blockshtml, 'data-block=') !== false;
$hasleftblocks = strpos($leftblocks, 'data-block=') !== false;

// Use new sidebar booleans instead:
$sidebar_left = strpos($leftblocks, 'data-block=') !== false;
$sidebar_right = strpos($blockshtml, 'data-block=') !== false;

// Main settings menu again in case context changes.
$regionmainsettingsmenu = $OUTPUT->region_main_settings_menu();

// Prepare user name for profile icon.
if($isloggedin){
    $niceProfileIconUsername = $USER->firstname . ' '. $USER->lastname;
}else{
    $niceProfileIconUsername = '';
}

// Prepare full-width block regions.
$blocks_fullwidth_top = $OUTPUT->blocks('fullwidth-top');
$blocks_fullwidth_bottom = $OUTPUT->blocks('fullwidth-bottom');
$blocks_above_content = $OUTPUT->blocks('above-content');
$blocks_below_content = $OUTPUT->blocks('below-content');

// Check for specific pages where blocks should be hidden.
$userProfileFromCourseParticipants = strpos($_SERVER['REQUEST_URI'], "user/view.php") !== false && isset($_GET["course"]);
$courseSectionPage = strpos($_SERVER['REQUEST_URI'], "course/view.php") !== false && isset($_GET["section"]);

if ((strpos($_SERVER['REQUEST_URI'], "user/index.php") !== false || strpos($_SERVER['REQUEST_URI'], "course/edit.php") !== false || strpos($_SERVER['REQUEST_URI'], "course/completion.php") !== false || strpos($_SERVER['REQUEST_URI'], "course/admin.php") !== false || $courseSectionPage) || $userProfileFromCourseParticipants){
  $sidebar_left = false;
  $sidebar_right = false;
  $blocks_above_content = false;
  $blocks_below_content = false;
  $blocks_fullwidth_top = false;
  $blocks_fullwidth_bottom = false;
}

/**
 * Load theme assets such as logos and social media config.
 */
$themeconfig = theme_config::load('nice');

// Navbar logo URL.
$navbar_logo = '';
if (!empty($themeconfig->settings->navbar_logo)) {
    $navbar_logo = $themeconfig->setting_file_url('navbar_logo', 'navbar_logo');
}

// Footer logo URL.
$footer_logo = '';
if (!empty($themeconfig->settings->footer_logo)) {
    $footer_logo = $themeconfig->setting_file_url('footer_logo', 'footer_logo');
}

// Determine how social media links open (same tab or new tab).
$social_media_target = get_config('theme_nice', 'social_media_link_target');
if($social_media_target == 1) {
  $social_media_target_href = '_blank';
} else {
  $social_media_target_href = '_self';
}

// Load theme-specific feature toggles.
$scroll_to_top_visibility      = get_config('theme_nice', 'scroll_to_top_visibility');
$homepage_link_visibility      = get_config('theme_nice', 'homepagelink');
$dashboard_link_visibility     = get_config('theme_nice', 'dashboardlink');
$logo_url                      = get_config('theme_nice', 'logourl');
