// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Functions for the chat interface
 *
 * @module     block_exaaichat
 * @copyright  2025 GTN Solutions https://gtn-solutions.com
 * @copyright  based on work by Limekiller https://github.com/Limekiller/moodle-block_openai_chat
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

/* eslint-disable @babel/semi, no-undef */
var questionString = 'Ask a question...'
var errorString = 'An error occurred! Please try again later.'

var chatData;

export const init = (data) => {
    const blockId = data['blockId']
    const api_type = data['api_type']
    const persistConvo = data['persistConvo']

    // Initialize local data storage if necessary
    // If a thread ID exists for this block, make an API request to get existing messages
    if (api_type === 'assistant') {
        chatData = localStorage.getItem("block_exaaichat_data")
        if (chatData) {
            chatData = JSON.parse(chatData)
            if (chatData[blockId] && chatData[blockId]['threadId'] && persistConvo === "1") {
                fetch(`${M.cfg.wwwroot}/blocks/exaaichat/api/thread.php?thread_id=${chatData[blockId]['threadId']}`)
                .then(response => response.json())
                .then(data => {
                    for (let message of data) {
                        addToChatLog(message.role === 'user' ? 'user' : 'bot', message.message, blockId)
                    }
                })
                // Some sort of error in the API call. Probably the thread no longer exists, so lets reset it
                .catch(() => {
                    chatData[blockId] = {}
                    localStorage.setItem("block_exaaichat_data", JSON.stringify(chatData));
                })
            // The block ID doesn't exist in the chat data object, so let's create it
            } else {
                chatData[blockId] = {}
            }
        // We don't even have a chat data object, so we'll create one
        } else {
            chatData = {[blockId]: {}}
        }
        localStorage.setItem("block_exaaichat_data", JSON.stringify(chatData));
    }

  if (api_type === 'responses') {
    chatData = localStorage.getItem("block_exaaichat_data")
    if (chatData) {
      chatData = JSON.parse(chatData)
      if (chatData[blockId] && chatData[blockId]['threadId'] && persistConvo === "1") {
        // fetch(`${M.cfg.wwwroot}/blocks/exaaichat/api/thread.php?thread_id=${chatData[blockId]['threadId']}`)
        //   .then(response => response.json())
        //   .then(data => {
        //     for (let message of data) {
        //       addToChatLog(message.role === 'user' ? 'user' : 'bot', message.message, blockId)
        //     }
        //   })
        //   // Some sort of error in the API call. Probably the thread no longer exists, so lets reset it
        //   .catch(error => {
        //     chatData[blockId] = {}
        //     localStorage.setItem("block_exaaichat_data", JSON.stringify(chatData));
        //   })
        // The block ID doesn't exist in the chat data object, so let's create it
        chatData[blockId] = {}
      } else {
        chatData[blockId] = {}
      }
      // We don't even have a chat data object, so we'll create one
    } else {
      chatData = {[blockId]: {}}
    }
    localStorage.setItem("block_exaaichat_data", JSON.stringify(chatData));
  }

  // Prevent sidebar from closing when osk pops up (hack for MDL-77957)
    window.addEventListener('resize', event => {
        event.stopImmediatePropagation();
    }, true);

    document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #openai_input`).addEventListener('keyup', e => {
        if (e.which === 13 && e.target.value !== "") {
            addToChatLog('user', e.target.value, blockId)
            createCompletion(e.target.value, blockId, api_type)
            e.target.value = ''
        }
    })
    document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #go`).addEventListener('click', () => {
        const input = document.querySelector('#openai_input')
        if (input.value !== "") {
            addToChatLog('user', input.value, blockId)
            createCompletion(input.value, blockId, api_type)
            input.value = ''
        }
    })

    document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #refresh`).addEventListener('click', () => {
        clearHistory(blockId)
    })

    document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #popout`).addEventListener('click', () => {
        if (document.querySelector('.drawer.drawer-right')) {
            document.querySelector('.drawer.drawer-right').style.zIndex = '1041'
        }
        document.querySelector(`.block_exaaichat[data-instance-id='${blockId}']`).classList.toggle('expanded')
    })

    require(['core/str'], function(str) {
        var strings = [
            {
                key: 'askaquestion',
                component: 'block_exaaichat'
            },
            {
                key: 'erroroccurred',
                component: 'block_exaaichat'
            },
        ];
        str.get_strings(strings).then((results) => {
            questionString = results[0];
            errorString = results[1];
        });
    });
}

/**
 * Add a message to the chat UI
 * @param {string} type Which side of the UI the message should be on. Can be "user" or "bot"
 * @param {string} message The text of the message to add
 * @param {int} blockId The ID of the block to manipulate
 */
const addToChatLog = (type, message, blockId) => {
    let messageContainer = document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #exaaichat_log`)

    const messageElem = document.createElement('div')
    messageElem.classList.add('openai_message')
    for (let className of type.split(' ')) {
        messageElem.classList.add(className)
    }

    const messageText = document.createElement('span')
    messageText.innerHTML = message
    messageElem.append(messageText)

    messageContainer.append(messageElem)
    if (messageText.offsetWidth) {
        messageElem.style.width = (messageText.offsetWidth + 40) + "px"
    }
    messageContainer.scrollTop = messageContainer.scrollHeight
    messageContainer.closest('.block_exaaichat > div').scrollTop = messageContainer.scrollHeight
}

/**
 * Clears the thread ID from local storage and removes the messages from the UI in order to refresh the chat
 * @param {int} blockId The ID of the block to manipulate
 */
const clearHistory = (blockId) => {
    chatData = localStorage.getItem("block_exaaichat_data")
    if (chatData) {
        chatData = JSON.parse(chatData)
        if (chatData[blockId]) {
            chatData[blockId] = {}
            localStorage.setItem("block_exaaichat_data", JSON.stringify(chatData));
        }
    }
    document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #exaaichat_log`).innerHTML = ""
}

/**
 * Makes an API request to get a completion from GPT-3, and adds it to the chat log
 * @param {string} message The text to get a completion for
 * @param {int} blockId The ID of the block this message is being sent from -- used to override settings if necessary
 * @param {string} api_type "assistant" | "chat" The type of API to use
 */
const createCompletion = (message, blockId, api_type) => {
    let threadId = null
    let chatData

    // If the type is assistant, attempt to fetch a thread ID
    if (api_type === 'assistant' || api_type === 'responses') {
        chatData = localStorage.getItem("block_exaaichat_data")
        if (chatData) {
            chatData = JSON.parse(chatData)
            if (chatData[blockId]) {
                threadId = chatData[blockId]['threadId'] || null
            }
        } else {
            // create the chat data item if necessary
            chatData = {[blockId]: {}}
        }
    }

    const history = buildTranscript(blockId)

    document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #control_bar`).classList.add('disabled')
    document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #openai_input`).classList.remove('error')
    document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #openai_input`).placeholder = questionString
    document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #openai_input`).blur()
    addToChatLog('bot loading', '...', blockId);

    fetch(`${M.cfg.wwwroot}/blocks/exaaichat/api/completion.php`, {
        method: 'POST',
        body: JSON.stringify({
            message: message,
            history: history,
            blockId: blockId,
            threadId: threadId
        })
    })
    .then(response => {
        let messageContainer = document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #exaaichat_log`)
        messageContainer.removeChild(messageContainer.lastElementChild)
        document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #control_bar`).classList.remove('disabled')

        if (!response.ok) {
            throw Error(response.statusText)
        } else {
            return response.json()
        }
    })
    .then(data => {
        try {
            addToChatLog('bot', data.message, blockId)
            if (data.thread_id) {
                chatData[blockId]['threadId'] = data.thread_id
                localStorage.setItem("block_exaaichat_data", JSON.stringify(chatData));
            }
        } catch (error) {
            logError(error);
            addToChatLog('bot', data.error.message, blockId)
        }
        document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #openai_input`).focus()
    })
    .catch(error => {
        logError(error);
        document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #openai_input`).classList.add('error')
        document.querySelector(`.block_exaaichat[data-instance-id='${blockId}'] #openai_input`).placeholder = errorString
    })
}

/**
 * Log an error to the console and potentially to an external service in the future
 * @param {Error} error The error object to log
 */
function logError(error) {
  /* eslint-disable no-console */
  console.error(error);
}

/**
 * Using the existing messages in the chat history, create a string that can be used to aid completion
 * @param {int} blockId The block from which to build the history
 * @return {JSONObject} A transcript of the conversation up to this point
 */
const buildTranscript = (blockId) => {
    let transcript = []
    document.querySelectorAll(`.block_exaaichat[data-instance-id='${blockId}'] .openai_message`).forEach((message, index) => {
        if (index === document.querySelectorAll(`.block_exaaichat[data-instance-id='${blockId}'] .openai_message`).length - 1) {
            return
        }

        let user = userName
        if (message.classList.contains('bot')) {
            user = assistantName
        }
        transcript.push({"user": user, "message": message.innerText})
    })

    return transcript
}
