<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace local_datacurso_ratings\external;

defined('MOODLE_INTERNAL') || die();

require_once("$CFG->libdir/externallib.php");

use external_function_parameters;
use external_value;
use external_single_structure;
use external_api;
use context_module;
use aiprovider_datacurso\httpclient\ai_services_api;

/**
 * Web service to get AI analysis for activity comments.
 *
 * @package    local_datacurso_ratings
 * @copyright  2025 Industria Elearning
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class get_ai_analysis_comments extends external_api {
    /**
     * Input parameters.
     *
     * @return external_function_parameters
     */
    public static function execute_parameters() {
        return new external_function_parameters([
            'cmid' => new external_value(PARAM_INT, 'Course module ID'),
        ]);
    }

    /**
     * Execute main logic.
     *
     * @param int $cmid
     * @return array
     */
    public static function execute($cmid) {
        global $DB;

        $params = self::validate_parameters(self::execute_parameters(), [
            'cmid' => $cmid,
        ]);

        // Validate context.
        $cm = get_coursemodule_from_id('', $params['cmid'], 0, false, MUST_EXIST);
        $context = context_module::instance($cm->id);
        self::validate_context($context);
        require_capability('local/datacurso_ratings:viewcoursereport', $context);

        // Get course.
        $course = $DB->get_record('course', ['id' => $cm->course], '*', MUST_EXIST);

        // Collect comments.
        $records = $DB->get_records('local_datacurso_ratings', [
            'cmid' => $params['cmid'],
        ], 'timecreated DESC', 'id, feedback, rating');

        $comments = [];
        $likes = 0;

        foreach ($records as $r) {
            $islike = ((int)$r->rating === 1);
            if ($islike) {
                $likes++;
            }
            $comments[] = [
                'rating' => $islike,
                'feedback' => $r->feedback,
            ];
        }

        $total = count($records);
        $approvalpercent = $total > 0 ? round(($likes / $total) * 100, 2) : 0.0;

        // Body to send.
        $body = [
            'course' => $course->fullname,
            'activity' => $cm->name,
            'activity_type' => $cm->modname,
            'approvalpercent' => $approvalpercent,
            'comments' => $comments,
        ];

        // Call AI API.
        $client = new ai_services_api();
        $response = $client->request('POST', '/rating/query', $body);

        // Extract IA reply (assuming API always returns { reply: "...", meta: {...} }).
        $airesponse = '';
        if (is_array($response) && isset($response['reply'])) {
            $airesponse = $response['reply'];
        } else if (is_string($response)) {
            $airesponse = $response;
        }

        return [
            'cmid' => $params['cmid'],
            'ai_analysis_comment' => $airesponse,
        ];
    }

    /**
     * Return structure.
     *
     * @return external_single_structure
     */
    public static function execute_returns() {
        return new external_single_structure([
            'cmid' => new external_value(PARAM_INT, 'ID del course module'),
            'ai_analysis_comment' => new external_value(PARAM_RAW, 'RSummary generated by AI for comments'),
        ]);
    }
}
