<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

// phpcs:disable moodle.Files.RequireLogin.Missing

/**
 * Auto-login end-point, a user can be fully authenticated in the site providing a valid token.
 * The token is validated and resolved by the Identity Provider for AppCrue or by an external system using
 * a REST endpoint.
 * The user is redirected to a target URL after login.
 * The target URL is built from parameters in the request:
 * - urltogo: a relative URL in the Moodle site. Static URL.
 * - course search: finds the course executing a customizable SQL query against mdl_course.idnumber field. Supports
 * parameters: course, group), param1, param2.
 * - pattern: Use a list of configurable patterns to create URLs using parameters course, group, param1, param2.
 *
 *
 * @package    local_appcrue
 * @copyright  2021 Juan Pblo de Castro
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(__DIR__ . '/../../config.php');
require_once('locallib.php');

if (!get_config('local_appcrue', 'enable_autologin')) {
    header('HTTP/1.1 405 Method Not Allowed');
    die();
    // Better act as a service don't throw new moodle_exception('servicedonotexist', 'error').
}
$context = context_system::instance();
$PAGE->set_context($context);
$PAGE->set_url(new moodle_url('/local/appcrue/autologin.php'));

try {
    $token = local_appcrue_get_token_param();    // The key generated by the IDP for AppCrue.
    $fallback = optional_param('fallback', 'error', PARAM_ALPHA); // If token fails: error, logout, ignore.
    // If allow_continue is false then transform fallback mode to error.
    if ($fallback === 'continue' && get_config('local_appcrue', 'allow_continue') == false) {
        throw new moodle_exception('continue_not_allowed', 'local_appcrue');
    }
    // Check token and get user record.
    [$user, $diag] = local_appcrue_get_user_by_token($token);
    if ($user == null && $fallback == 'error') {
        // Set status code 401 and show error.
        echo $OUTPUT->header();
        echo $OUTPUT->box_start('generalbox centerpara');
        echo html_writer::tag('p', get_string('autologin:notauthenticated', 'local_appcrue'));
        echo $OUTPUT->error_text($diag->result);
        echo $OUTPUT->box_end();
        echo $OUTPUT->footer();
        // Log event.
        \local_appcrue\event\autologin_failed::create([
            'other' => [
                'ipaddress' => \local_appcrue\network_security_helper::getremoteaddr(),
                'token' => $token,
                'diagnosis' => $diag->result,
                ],
        ])->trigger();
        header('HTTP/1.1 401 Unauthorized');
        die();
    }

    if ($user == null && $fallback == 'logout') {
        require_logout();
        // Event log.
        \local_appcrue\event\autologin_failed::create([
            'other' => [
                'ipaddress' => network_security_helper::getremoteaddr(),
                'token' => $token,
                'diagnosis' => $diag->result,
                ],
        ])->trigger();
    } else if ($user == null && $fallback == 'continue') {
        // Do nothing with session.
        // If there is no session log in as guest to view the course page.
        if (!isset($USER->id) || $USER->id == 0) {
            $user = core_user::get_user_by_username('guest');
            complete_user_login($user);
            \core\session\manager::apply_concurrent_login_limit($user->id, session_id());
        }
        // Event log.
        \local_appcrue\event\autologin_failed::create([
            'other' => [
                'ipaddress' => network_security_helper::getremoteaddr(),
                'token' => $token,
                'diagnosis' => $diag->result . ' Continuing as guest or current user.',
                ],
        ])->trigger();
    } else if ($user != null) {
        // Token validated, now require an active user: not guest, not suspended.
        core_user::require_active_user($user, true, true);
        complete_user_login($user);
        \core\session\manager::apply_concurrent_login_limit($user->id, session_id());
        // Event log.
        \local_appcrue\event\autologin_login::create(['userid' => $user->id])->trigger();
    }

    // Get parameters to apply the redirection rules.
    $urltogo = optional_param('urltogo', null, PARAM_URL);    // Relative URL to redirect.
    $course = optional_param('course', null, PARAM_INT); // Course internal ID.
    $group = optional_param('group', 1, PARAM_INT); // Grupo docente.
    $year = optional_param('year', null, PARAM_INT); // Curso docente.
    $pattern = optional_param('pattern', null, PARAM_ALPHA);
    $param1 = optional_param('param1', null, PARAM_ALPHANUMEXT);
    $param2 = optional_param('param2', null, PARAM_ALPHANUMEXT);
    $param3 = optional_param('param3', null, PARAM_ALPHANUMEXT);

    $urltogo = local_appcrue_get_target_url($token, $urltogo, $course, $group, $year, $pattern, $param1, $param2, $param3);
    $usepageredirection = get_config('local_appcrue', 'use_redirection_page');
    if ($USER && $USER->id == 0) {
        // Unauthenticated user, do a plain redirect.
        $usepageredirection = false;
        $msg = null;
    } else if ($USER->username == 'guest') {
        $msg = get_string('autologin:loggedasguest', 'local_appcrue', ['sitename' => format_string(get_site()->fullname)]);
    } else {
        $msg = get_string(
            'autologin:loggedasuser',
            'local_appcrue',
            ['fullname' => fullname($USER), 'sitename' => format_string(get_site()->fullname)]
        );
    }
    // Redirect or show a redirection page.
    if (!$usepageredirection) {
        redirect($urltogo, $msg, 0, \core\output\notification::NOTIFY_SUCCESS);
    } else {
        // Show a redirection page with a link in case the direct redirection does not work.
        // For guest user we do not redirect automatically, so they need to click the link.
        echo $OUTPUT->header();
        echo $OUTPUT->box_start('generalbox centerpara');
        $PAGE->set_pagelayout('redirect');
        echo html_writer::tag('p', $msg);
        $link = html_writer::link($urltogo, get_string('continue'));
        echo html_writer::tag('p', $link);
        echo $OUTPUT->box_end();
        echo $OUTPUT->notification(get_string('autologin:redirecting', 'local_appcrue'), \core\output\notification::NOTIFY_INFO);
        if ($USER->username != 'guest') {
            // Add automatic redirection.
            echo html_writer::tag('script', "setTimeout(function() { window.location.href = '" . $urltogo . "'; }, 100);");
        }
        echo $OUTPUT->footer();
    }
} catch (moodle_exception $e) {
    header('HTTP/1.0 400 Bad Request: ' . $e->getMessage());
}
