<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Grade Now for solo plugin
 *
 * @package    mod_solo
 * @copyright  2019 Justin Hunt (poodllsupport@gmail.com)
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
 namespace mod_solo;
defined('MOODLE_INTERNAL') || die();

use mod_solo\constants;


/**
 * AI transcript Functions used generally across this mod
 *
 * @package    mod_solo
 * @copyright  2015 Justin Hunt (poodllsupport@gmail.com)
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class aitranscriptutils {



    // are we willing and able to transcribe submissions?
    public static function can_transcribe($instance) {
        // we default to true
        // but it only takes one no ....
        $ret = true;

        // The regions that can transcribe
        switch($instance->region){
            case "useast1":
            case "dublin":
            case "sydney":
            case "ottawa":
                break;
            default:
                $ret = false;
        }

        // if user disables ai, we do not transcribe
        if(!$instance->enableai){
            $ret = false;
        }

        return $ret;
    }

    // see if this is truly json or some error
    public static function is_json($string) {
        if(!$string){return false;
        }
        if(empty($string)){return false;
        }
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }


    /*
    * Turn a passage with text "lines" into html "brs"
    *
    * @param String The passage of text to convert
    * @param String An optional pad on each replacement (needed for processing when marking up words as spans in passage)
    * @return String The converted passage of text
    */
    public static function lines_to_brs($passage, $seperator='') {
        // see https://stackoverflow.com/questions/5946114/how-to-replace-newline-or-r-n-with-br
        return str_replace("\r\n", $seperator . '<br>' . $seperator, $passage);
        // this is better but we can not pad the replacement and we need that
        // return nl2br($passage);
    }

    public static function fetch_duration_from_transcript($jsontranscript) {
        // if we do not have the full transcript return 0
        if(!$jsontranscript || empty($jsontranscript)){
            return 0;
        }

        $transcript = json_decode($jsontranscript);
        if(isset($transcript->results)){
            $duration = self::fetch_duration_from_transcript_json($jsontranscript);
        }else{
            $duration = self::fetch_duration_from_transcript_gjson($jsontranscript);
        }
        return $duration;

    }

    public static function fetch_duration_from_transcript_json($jsontranscript) {
        // if we do not have the full transcript return 0
        if(!$jsontranscript || empty($jsontranscript)){
            return 0;
        }

        $transcript = json_decode($jsontranscript);
        $titems = $transcript->results->items;
        $twords = [];
        foreach($titems as $titem){
            if($titem->type == 'pronunciation'){
                $twords[] = $titem;
            }
        }
        $lastindex = count($twords);
        if($lastindex > 0){
            return round($twords[$lastindex - 1]->end_time, 0);
        }else{
            return 0;
        }
    }

    public static function fetch_duration_from_transcript_gjson($jsontranscript) {
        // if we do not have the full transcript return 0
        if(!$jsontranscript || empty($jsontranscript)){
            return 0;
        }

        $transcript = json_decode($jsontranscript);
        $twords = [];
        // create a big array of 'words' from gjson sentences
        foreach($transcript as $sentence) {
            $twords = array_merge($twords, $sentence->words);

        }//end of sentence
        $twordcount = count($twords);
        if($twordcount > 0){
            $tword = $twords[$twordcount - 1];
            $ms = round(floatval($tword->endTime->nanos * .000000001), 2);
            return round($tword->endTime->seconds + $ms, 0);
        }else{
            return 0;
        }
    }


    public static function fetch_audio_points($jsontranscript, $matches, $alternatives) {

        // first check if we have a jsontranscript (we might only have a transcript in some cases)
        // if not we just return dummy audio points. Que sera sera
        if (!self::is_json($jsontranscript)) {
            foreach ($matches as $matchitem) {
                $matchitem->audiostart = 0;
                $matchitem->audioend = 0;
            }
            return $matches;
        }
        $transcript = json_decode($jsontranscript);
        if(isset($transcript->results)){
            $matches = self::fetch_audio_points_json($transcript, $matches, $alternatives);
        }else{
            $matches = self::fetch_audio_points_gjson($transcript, $matches, $alternatives);
        }
        return $matches;
    }


    // fetch start-time and end-time points for each word
    public static function fetch_audio_points_json($transcript, $matches, $alternatives) {

        // get type 'pronunciation' items from full transcript. The other type is 'punctuation'.
        $titems = $transcript->results->items;
        $twords = [];
        foreach($titems as $titem){
            if($titem->type == 'pronunciation'){
                $twords[] = $titem;
            }
        }
        $twordcount = count($twords);

        // loop through matches and fetch audio start from word item
        foreach ($matches as $matchitem){
            if($matchitem->tposition <= $twordcount){
                // pull the word data object from the full transcript, at the index of the match
                $tword = $twords[$matchitem->tposition - 1];

                // trust or be sure by matching ...
                $trust = false;
                if($trust){
                    $matchitem->audiostart = $tword->start_time;
                    $matchitem->audioend = $tword->end_time;
                }else {
                    // format the text of the word to lower case no punc, to match the word in the matchitem
                    $twordtext = strtolower($tword->alternatives[0]->content);
                    $twordtext = preg_replace("#[[:punct:]]#", "", $twordtext);
                    // if we got it, fetch the audio position from the word data object
                    if ($matchitem->word == $twordtext) {
                        $matchitem->audiostart = $tword->start_time;
                        $matchitem->audioend = $tword->end_time;

                        // do alternatives search for match
                    }else if(diff::check_alternatives_for_match($matchitem->word,
                        $twordtext,
                        $alternatives)){
                        $matchitem->audiostart = $tword->start_time;
                        $matchitem->audioend = $tword->end_time;
                    }
                }
            }
        }
        return $matches;
    }

    // fetch start-time and end-time points for each word
    public static function fetch_audio_points_gjson($transcript, $matches, $alternatives) {
        $twords = [];
        // create a big array of 'words' from gjson sentences
        foreach($transcript as $sentence) {
            $twords = array_merge($twords, $sentence->words);

        }//end of sentence
        $twordcount = count($twords);

        // loop through matches and fetch audio start from word item
        foreach ($matches as $matchitem) {
            if ($matchitem->tposition <= $twordcount) {
                // pull the word data object from the full transcript, at the index of the match
                $tword = $twords[$matchitem->tposition - 1];
                // make startTime and endTime match the regular format
                $starttime = $tword->startTime->seconds + round(floatval($tword->startTime->nanos * .000000001), 2);
                $endtime = $tword->endTime->seconds + round(floatval($tword->endTime->nanos * .000000001), 2);

                // trust or be sure by matching ...
                $trust = false;
                if ($trust) {
                    $matchitem->audiostart = $starttime;
                    $matchitem->audioend = $endtime;
                } else {
                    // format the text of the word to lower case no punc, to match the word in the matchitem
                    $twordtext = strtolower($tword->word);
                    $twordtext = preg_replace("#[[:punct:]]#", "", $twordtext);
                    // if we got it, fetch the audio position from the word data object
                    if ($matchitem->word == $twordtext) {
                        $matchitem->audiostart = $starttime;
                        $matchitem->audioend = $endtime;

                        // do alternatives search for match
                    } else if (diff::check_alternatives_for_match($matchitem->word,
                            $twordtext,
                            $alternatives)) {
                        $matchitem->audiostart = $starttime;
                        $matchitem->audioend = $endtime;
                    }
                }
            }
        }//end of words

        return $matches;
    }

    // this is a server side implementation of the same name function in gradenowhelper.js
    // we need this when calculating adjusted grades(reports/machinegrading.php) and on making machine grades(aigrade.php)
    // the WPM adjustment based on accadjust only applies to machine grades, so it is NOT in gradenowhelper
    public static function processscores($sessiontime, $sessionendword, $errorcount, $activitydata) {

        // wpm score
        $wpmerrors = $errorcount;

        // target WPM is a ReadaLoud feature. We probably won't use it here.
        $targetwpm = 100;

        if($sessiontime > 0) {
            $wpmscore = round(($sessionendword - $wpmerrors) * 60 / $sessiontime);
        }else{
            $wpmscore = 0;
        }

        // accuracy score
        if($sessionendword > 0) {
            $accuracyscore = round(($sessionendword - $errorcount) / $sessionendword * 100);
        }else{
            $accuracyscore = 0;
        }

        // sessionscore
        $usewpmscore = $wpmscore;
        if($usewpmscore > $targetwpm){
            $usewpmscore = $targetwpm;
        }
        $sessionscore = round($usewpmscore / $targetwpm * 100);

        $scores = new \stdClass();
        $scores->wpmscore = $wpmscore;
        $scores->accuracyscore = $accuracyscore;
        $scores->sessionscore = $sessionscore;
        return $scores;

    }

    // take a json string of session errors, anmd count how many there are.
    public static function count_sessionerrors($sessionerrors) {
        $errors = json_decode($sessionerrors);
        if($errors){
            $errorcount = count(get_object_vars($errors));
        }else{
            $errorcount = 0;
        }
        return $errorcount;
    }

    // get all the aievaluations for a user
    public static function get_aieval_byuser($moduleid, $userid) {
        global $DB;
        $sql = "SELECT tai.*  FROM {" . constants::M_AITABLE . "} tai INNER JOIN  {" . constants::M_ATTEMPTSTABLE . "}" .
            " tu ON tu.id =tai.attemptid AND tu." . constants::M_AI_PARENTFIELDNAME . "=tai.moduleid WHERE tu." . constants::M_AI_PARENTFIELDNAME . "=? AND tu.userid=?";
        $result = $DB->get_records_sql($sql, [$moduleid, $userid]);
        return $result;
    }

    // get average difference between human graded attempt error count and AI error count
    // we only fetch if A) have machine grade and B) sessiontime> 0(has been manually graded)
    public static function estimate_errors($moduleid) {
        global $DB;
        $errorestimate = 0;
        $sql = "SELECT AVG(tai.errorcount - tu.errorcount) as errorestimate  FROM {" . constants::M_AITABLE . "} tai INNER JOIN  {" . constants::M_USERTABLE . "}" .
            " tu ON tu.id =tai.attemptid AND tu." . constants::M_AI_PARENTFIELDNAME . "=tai.moduleid WHERE tu.sessiontime > 0 AND tu." . constants::M_AI_PARENTFIELDNAME . "=?";
        $result = $DB->get_field_sql($sql, [$moduleid]);
        if($result !== false){
            $errorestimate = round($result);
        }
        return $errorestimate;
    }

    /*
    * Per passageword, an object with mistranscriptions and their frequency will be returned
    * To be consistent with how data is stored in matches/errors, we return a 1 based array of mistranscriptions
     * @return array an array of stdClass (1 item per passage word) with the passage index(1 based), passage word and array of mistranscription=>count
    */
    public static function fetch_all_mistranscriptions($moduleid) {
        global $DB;
        $attempts = $DB->get_records(constants::M_AITABLE , ['moduleid' => $moduleid]);
        $activity = $DB->get_record(constants::M_TABLE, ['id' => $moduleid]);
        $passagewords = diff::fetchWordArray($activity->passage);
        $passagecount = count($passagewords);
        // $alternatives = diff::fetchAlternativesArray($activity->alternatives);

        $results = [];
        $mistranscriptions = [];
        foreach ($attempts as $attempt) {
            $transcriptwords = diff::fetchWordArray($attempt->transcript);
            $matches = json_decode($attempt->sessionmatches);
            $mistranscriptions[] = self::fetch_attempt_mistranscriptions($passagewords, $transcriptwords, $matches);
        }
        // aggregate results
        for ($wordnumber = 1; $wordnumber <= $passagecount; $wordnumber++) {
            $aggregateset = [];
            foreach ($mistranscriptions as $mistranscript) {
                if (!$mistranscript[$wordnumber]) {
                    continue;
                }
                if (array_key_exists($mistranscript[$wordnumber], $aggregateset)) {
                    $aggregateset[$mistranscript[$wordnumber]]++;
                } else {
                    $aggregateset[$mistranscript[$wordnumber]] = 1;
                }
            }
            $result = new \stdClass();
            $result->mistranscriptions = $aggregateset;
            $result->passageindex = $wordnumber;
            $result->passageword = $passagewords[$wordnumber - 1];
            $results[] = $result;
        }//end of for loop
        return $results;
    }


    /*
    * This will return an array of mistranscript strings for a single attemot. 1 entry per passageword.
     * To be consistent with how data is stored in matches/errors, we return a 1 based array of mistranscriptions
     * @return array a 1 based array of mistranscriptions(string) or false. i item for each passage word
    */
    public static function fetch_attempt_mistranscriptions($passagewords, $transcriptwords, $matches) {
        $passagecount = count($passagewords);
        if(!$passagecount){return false;
        }
        $mistranscriptions = [];
        for($wordnumber = 1; $wordnumber <= $passagecount; $wordnumber++){
            $mistranscription = self::fetch_one_mistranscription($wordnumber, $transcriptwords, $matches);
            if($mistranscription){
                $mistranscriptions[$wordnumber] = $mistranscription;
            }else{
                $mistranscriptions[$wordnumber] = false;
            }
        }//end of for loop
        return $mistranscriptions;
    }

    /*
    * This will take a wordindex and find the previous and next transcript indexes that were matched and
    * return all the transcript words in between those.
     *
     * @return a string which is the transcript match of a passage word, or false if the transcript=passage
    */
    public static function fetch_one_mistranscription($passageindex, $transcriptwords, $matches) {

           // if we have a problem with matches (bad data?) just return
        if(!$matches){
            return false;
        }

            // count transcript words
            $transcriptlength = count($transcriptwords);
        if($transcriptlength == 0){
            return false;
        }

            // build a quick to search array of matched words
            $passagematches = [];
        foreach($matches as $match){
            $passagematches[$match->pposition] = $match->word;
        }

            // find startindex
            $startindex = -1;
        for($wordnumber = $passageindex; $wordnumber > 0; $wordnumber--){

            $ismatched = array_key_exists($wordnumber, $passagematches);
            if($ismatched){
                $startindex = $matches->{$wordnumber}->tposition + 1;
                break;
            }
        }//end of for loop

            // find endindex
            $endindex = -1;
        for($wordnumber = $passageindex; $wordnumber <= $transcriptlength; $wordnumber++){

            $ismatched = array_key_exists($wordnumber, $passagematches);
            // if we matched then the previous transcript word is the last unmatched one in the checkindex sequence
            if($ismatched){
                $endindex = $matches->{$wordnumber}->tposition - 1;
                break;
            }
        }//end of for loop --

            // if there was no previous matched word, we set start to 1
        if($startindex == -1){$startindex = 1;
        }
            // if there was no subsequent matched word we flag the end as the -1
        if($endindex == $transcriptlength){
            $endindex = -1;
            // an edge case is where the first word is not in transcript and first match is the second or later passage
            // word. It might not be possible for endindex to be lower than start index, but we don't want it anyway
        }else if($endindex == 0 || $endindex < $startindex){
            return false;
        }

            // up until this point the indexes have started from 1, since the passage word numbers start from 1
            // but the transcript array is 0 based so we adjust. array_slice function does not include item and endindex
            // so it needs to be one more then start index. hence we do not adjust that
            $startindex--;

            // finally we return the section of transcript
        if($endindex > 0) {
            $chunklength = $endindex - $startindex;
            $retarray = array_slice($transcriptwords, $startindex, $chunklength);
        }else{
            $retarray = array_slice($transcriptwords, $startindex);
        }

            $ret = implode(" ", $retarray);
        if(utils::super_trim($ret) == ''){
            return false;
        }else{
            return $ret;
        }
    }




    // for error estimate and accuracy adjustment, we can auto estimate errors, never estimate errors, or use a fixed error estimate, or ignore errors
    public static function get_accadjust_options() {
        return [
            constants::ACCMETHOD_NONE => get_string("accmethod_none", constants::M_COMPONENT),
            // constants::ACCMETHOD_AUTO  => get_string("accmethod_auto",constants::M_COMPONENT),
            constants::ACCMETHOD_FIXED  => get_string("accmethod_fixed", constants::M_COMPONENT),
            constants::ACCMETHOD_NOERRORS  => get_string("accmethod_noerrors", constants::M_COMPONENT),
        ];
    }

    public static function render_passage($passage, $markuptype='passage') {
        // load the HTML document
        $doc = new \DOMDocument;
        // it will assume ISO-8859-1  encoding, so we need to hint it:
        // see: http://stackoverflow.com/questions/8218230/php-domdocument-loadhtml-not-encoding-utf-8-correctly

        //The old way ... throws errors on PHP 8.2+
        //$safepassage = mb_convert_encoding($passage, 'HTML-ENTITIES', 'UTF-8');
        //@$doc->loadHTML($safepassage, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD | LIBXML_NOERROR | LIBXML_NOWARNING);

        //This could work, but on some occasions the doc has a meta header already .. hmm
        //$safepassage = mb_convert_encoding($passage, 'HTML-ENTITIES', 'UTF-8');
        //@$doc->loadHTML('<?xml encoding="utf-8" ? >' . $safepassage, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD | LIBXML_NOERROR | LIBXML_NOWARNING);

        //The new way .. incomprehensible but works
        $safepassage = htmlspecialchars($passage, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5, 'UTF-8');
        @$doc->loadHTML(mb_encode_numericentity($safepassage, [0x80, 0x10FFFF, 0, ~0], 'UTF-8'));

        // select all the text nodes
        $xpath = new \DOMXPath($doc);
        $nodes = $xpath->query('//text()');

        // base CSS class
        if($markuptype == 'passage') {
            $cssword = constants::M_CLASS . '_grading_passageword';
            $cssspace = constants::M_CLASS . '_grading_passagespace';
        }else{
            $cssword = constants::M_CLASS . '_grading_correctionsword';
            $cssspace = constants::M_CLASS . '_grading_correctionsspace';
        }

        // original CSS classes
        // The original classes are to show the original passage word before or after the corrections word
        // because of the layout, "rewritten/added words" [corrections] will show in green, after the original words [red]
        // but "removed(omitted) words" [corrections] will show as a green space  after the original words [red]
        // so the span layout for each word in the corrections is:
        // [original_preword][correctionsword][original_postword][correctionsspace]
        // suggested word: (original)He eat apples => (corrected)He eats apples =>
        // [original_preword: "eat->"][correctionsword: "eats"][original_postword][correctionsspace]
        // removed(omitted) word: (original)He eat devours the apples=> (corrected)He devours the apples =>
        // [original_preword: ][correctionsword: "He"][original_postword: "eat->" ][correctionsspace: " "]

        $cssoriginalpreword = constants::M_CLASS . '_grading_original_preword';
        $cssoriginalpostword = constants::M_CLASS . '_grading_original_postword';

        // init the text count
        $wordcount = 0;
        foreach ($nodes as $node) {
            $trimmednode = utils::super_trim($node->nodeValue);
            if (empty($trimmednode)) {
                continue;
            }

            // explode missed new lines that had been copied and pasted. eg A[newline]B was not split and was one word
            // This resulted in ai selected error words, having different index to their passage text counterpart
            $seperator = ' ';
            // $words = explode($seperator, $node->nodeValue);

            $nodevalue = self::lines_to_brs($node->nodeValue, $seperator);
            $words = preg_split('/\s+/', $nodevalue);

            foreach ($words as $word) {
                // if its a new line character from lines_to_brs we add it, but not as a word
                if ($word == '<br>') {
                    $newnode = $doc->createElement('br', $word);
                    $node->parentNode->appendChild($newnode);
                    continue;
                }

                $wordcount++;
                // wordnode
                $newnode = $doc->createElement('span', $word);
                $newnode->setAttribute('id', $cssword . '_' . $wordcount);
                $newnode->setAttribute('data-wordnumber', $wordcount);
                $newnode->setAttribute('class', $cssword);
                // space node
                $spacenode = $doc->createElement('span', $seperator);
                $spacenode->setAttribute('id', $cssspace . '_' . $wordcount);
                $spacenode->setAttribute('data-wordnumber', $wordcount);
                $spacenode->setAttribute('class', $cssspace);
                // original pre node
                if($markuptype !== 'passage'){
                    $originalprenode = $doc->createElement('span', '');
                    $originalprenode->setAttribute('id', $cssoriginalpreword . '_' . $wordcount);
                    $originalprenode->setAttribute('data-wordnumber', $wordcount);
                    $originalprenode->setAttribute('class', $cssoriginalpreword);

                }
                // original post node
                if($markuptype !== 'passage'){
                    $originalpostnode = $doc->createElement('span', '');
                    $originalpostnode->setAttribute('id', $cssoriginalpostword . '_' . $wordcount);
                    $originalpostnode->setAttribute('data-wordnumber', $wordcount);
                    $originalpostnode->setAttribute('class', $cssoriginalpostword);

                }
                // add nodes to doc
                if($markuptype == 'passage'){
                    $node->parentNode->appendChild($newnode);
                    $node->parentNode->appendChild($spacenode);
                }else{
                    $node->parentNode->appendChild($originalprenode);
                    $node->parentNode->appendChild($newnode);
                    $node->parentNode->appendChild($originalpostnode);
                    $node->parentNode->appendChild($spacenode);
                }
                // $newnode = $doc->createElement('span', $word);
            }
            $node->nodeValue = "";
        }

        $usepassage = $doc->saveHTML();
        // remove container 'p' tags, they mess up formatting in solo
        $usepassage = str_replace('<p>', '', $usepassage);
        $usepassage = str_replace('</p>', '', $usepassage);

        if($markuptype == 'passage') {
            $ret = \html_writer::div($usepassage, constants::M_CLASS . '_grading_passagecont ' . constants::M_CLASS . '_summarytranscriptplaceholder');
        }else{
            $ret = \html_writer::div($usepassage, constants::M_CLASS . '_corrections ');
        }
        return $ret;
    }

    public static function prepare_turn_markers($attempt) {
        $st = $attempt->selftranscript;
        if(empty($st)){
            return [];
        }
        $sentences = utils::fetch_selftranscript_parts($attempt);

        $markers = [];
        $nextstart = 1;
        foreach ($sentences as $sentence){
            $wordcount = self::count_turn_words($sentence);
            if($wordcount) {
                $turnend = $nextstart + $wordcount - 1;
                $markers[] = ['start' => $nextstart, 'end' => $turnend];
                $nextstart = $turnend + 1;
            }
        }
        return $markers;
    }

    /*
     * This function is v similar to render_passage because we need to have the same word count as render_passage would
     * for markup to be successful.
     *
     * TODO: remove the redundancy here with a helper function
     */
    public static function count_turn_words($turntext) {
        // load the HTML document
        $doc = new \DOMDocument;
        // it will assume ISO-8859-1  encoding, so we need to hint it:
        // see: http://stackoverflow.com/questions/8218230/php-domdocument-loadhtml-not-encoding-utf-8-correctly
        $safepassage = htmlspecialchars($turntext, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5, 'UTF-8');
        @$doc->loadHTML(mb_encode_numericentity($safepassage, [0x80, 0x10FFFF, 0, ~0], 'UTF-8'));

        // select all the text nodes
        $xpath = new \DOMXPath($doc);
        $nodes = $xpath->query('//text()');
        // init the text count
        $wordcount = 0;
        foreach ($nodes as $node) {
            $trimmednode = utils::super_trim($node->nodeValue);
            if (empty($trimmednode)) {
                continue;
            }

            // explode missed new lines that had been copied and pasted. eg A[newline]B was not split and was one word
            // This resulted in ai selected error words, having different index to their passage text counterpart
            $seperator = ' ';
            // $words = explode($seperator, $node->nodeValue);

            $nodevalue = self::lines_to_brs($node->nodeValue, $seperator);
            $words = preg_split('/\s+/', $nodevalue);

            foreach ($words as $word) {
                // if its a new line character from lines_to_brs we add it, but not as a word
                if ($word == '<br>') {
                    $newnode = $doc->createElement('br', $word);
                    $node->parentNode->appendChild($newnode);
                    continue;
                }

                $wordcount++;
            }
            $node->nodeValue = "";
        }
        return $wordcount;
    }

    public static function prepare_passage_amd($attempt, $aidata) {
        global $PAGE;

        // here we set up any info we need to pass into javascript
        $passageopts = [];
        $passageopts['sesskey'] = sesskey();
        $passageopts['activityid'] = $attempt->solo;
        $passageopts['attemptid'] = $attempt->id;
        $passageopts['sessiontime'] = $aidata->sessiontime;
        $passageopts['sessionerrors'] = $aidata->sessionerrors;
        $passageopts['sessionendword'] = $aidata->sessionendword;
        $passageopts['sessionmatches'] = $aidata->sessionmatches;
        $passageopts['aidata'] = $aidata;
        $passageopts['turns'] = self::prepare_turn_markers($attempt);
        $passageopts['opts_id'] = 'mod_solo_passageopts';

        $jsonstring = json_encode($passageopts);
        $optshtml =
                \html_writer::tag('input', '', ['id' => $passageopts['opts_id'], 'type' => 'hidden', 'value' => $jsonstring]);
        $PAGE->requires->js_call_amd("mod_solo/passagemarkup", 'init', [['id' => $passageopts['opts_id']]]);
        $PAGE->requires->strings_for_js(['heard'],
                'mod_solo');

        // these need to be returned and echo'ed to the page
        return $optshtml;
    }

    public static function prepare_corrections_amd($sessionerrors, $sessionmatches, $insertioncount) {
        global $PAGE;

        // here we set up any info we need to pass into javascript
        $correctionsopts = [];
        $correctionsopts['sessionerrors'] = $sessionerrors; // these are words different from those in original
        $correctionsopts['sessionmatches'] = $sessionmatches; // these are words matching the original
        $correctionsopts['insertioncount'] = $insertioncount; // these are words missing from the original
        $correctionsopts['opts_id'] = 'mod_solo_correctionopts';

        $jsonstring = json_encode($correctionsopts);
        $optshtml =
            \html_writer::tag('input', '', ['id' => $correctionsopts['opts_id'], 'type' => 'hidden', 'value' => $jsonstring]);
        $PAGE->requires->js_call_amd("mod_solo/correctionsmarkup", 'init', [['id' => $correctionsopts['opts_id']]]);

        // these need to be returned and echo'ed to the page
        return $optshtml;
    }

}
