<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Event triggered when a mitigation action is executed for a user.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\event;

defined('MOODLE_INTERNAL') || die();

/**
 * Event class representing a mitigation action executed for a user.
 *
 * This event is triggered whenever the Behaviour Analytics system
 * automatically runs a mitigation action (e.g., suspending a user,
 * terminating sessions, or notifying administrators) due to a
 * detected high-risk behavioural profile.
 *
 * The event is logged to Moodle’s standard event log for auditability.
 *
 * @package     local_behavioranalytics
 */
final class mitigation_executed extends \core\event\base {
    /**
     * Initialise event metadata.
     *
     * Sets CRUD type, education level, and associated database table.
     *
     * @return void
     */
    protected function init(): void {
        $this->data['crud'] = 'u';
        $this->data['edulevel'] = self::LEVEL_OTHER;
        $this->data['objecttable'] = 'user';
    }

    /**
     * Get the human-readable name of this event.
     *
     * @return string Localised event name.
     */
    public static function get_name(): string {
        return get_string('event_mitigation_executed', 'local_behavioranalytics');
    }

    /**
     * Generate a plain-text description of the event.
     *
     * @return string Description of the triggered event.
     */
    public function get_description(): string {
        $other = $this->other ?? [];
        $action = $other['action'] ?? 'unknown';
        $score = $other['score'] ?? '?';
        $level = $other['level'] ?? 'n/a';

        return "Mitigation action '{$action}' was executed for user ID {$this->relateduserid} "
            . "due to risk score {$score} ({$level}).";
    }

    /**
     * Get the URL associated with this event.
     *
     * Returns a link to the affected user’s profile page for review.
     *
     * @return \moodle_url Link to the user’s profile page.
     */
    public function get_url(): \moodle_url {
        return new \moodle_url('/user/profile.php', ['id' => $this->relateduserid]);
    }
}
