<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * External web service for retrieving a user's risk score.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\external;

use context_system;
use external_api;
use external_function_parameters;
use external_single_structure;
use external_value;
use local_behavioranalytics\local\risk\scorer;

/**
 * Provides a web service endpoint to retrieve a user's behaviour risk score.
 *
 * This service validates permissions, retrieves the persisted risk profile
 * if available, and falls back to computing the current score using
 * {@see scorer::score_user()} if no profile record exists.
 *
 * The function is defined as part of Moodle’s external API and can be exposed
 * via REST, SOAP, or the built-in mobile service.
 *
 * @package     local_behavioranalytics
 */
class get_user_risk extends external_api {
    /**
     * Define the expected input parameters for this external function.
     *
     * @return external_function_parameters Parameter definition.
     */
    public static function execute_parameters(): external_function_parameters {
        return new external_function_parameters([
            'userid' => new external_value(PARAM_INT, 'User ID'),
        ]);
    }

    /**
     * Execute the external function logic.
     *
     * Returns a user’s persisted or computed risk score and corresponding level.
     *
     * @param int $userid Moodle user ID.
     * @return array{score:int, level:string} Associative array with score and level.
     */
    public static function execute(int $userid): array {
        global $DB;

        $params = self::validate_parameters(self::execute_parameters(), ['userid' => $userid]);

        $context = context_system::instance();
        self::validate_context($context);
        require_capability('local/behavioranalytics:view', $context);

        $user = $DB->get_record('user', ['id' => $params['userid']], '*', MUST_EXIST);

        // Use stored risk profile if available, otherwise compute dynamically.
        if ($profile = $DB->get_record('local_behavioranalytics_profile', ['userid' => $user->id])) {
            return [
                'score' => (int) $profile->score,
                'level' => (string) $profile->level,
            ];
        }

        $computed = scorer::score_user($user);
        return [
            'score' => (int) $computed['score'],
            'level' => (string) $computed['level'],
        ];
    }

    /**
     * Define the structure of the return value for the external function.
     *
     * @return external_single_structure Output definition.
     */
    public static function execute_returns(): external_single_structure {
        return new external_single_structure([
            'score' => new external_value(PARAM_INT, 'Risk score (0–100)'),
            'level' => new external_value(PARAM_TEXT, 'Textual risk level label'),
        ]);
    }
}
