<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Mitigation action base class.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\local\action;

use admin_settingpage;
use stdClass;

/**
 * Abstract base class for mitigation actions.
 *
 * Each mitigation action represents a concrete response that can be executed
 * automatically when a user’s calculated risk score exceeds the configured
 * threshold (e.g., suspend user, notify admin, terminate sessions).
 *
 * Subclasses must:
 *  - Define a unique {@see self::IDENT} string.
 *  - Implement {@see self::get_name()} to provide a localised name.
 *  - Implement {@see self::execute()} to perform the actual action.
 *
 * Optionally, subclasses may override {@see self::add_settings()} to
 * contribute custom configuration controls to the admin settings page.
 *
 * @package     local_behavioranalytics
 */
abstract class base {
    /** @var string Unique identifier (lowercase snake_case). */
    public const IDENT = 'base';

    /**
     * Return a human-readable, localised name for this action.
     *
     * Implementations should use a language string reference such as:
     * `get_string('action_suspend_user', 'local_behavioranalytics')`
     *
     * @return string Localised display name.
     */
    abstract public static function get_name(): string;

    /**
     * Execute the mitigation logic for the specified user.
     *
     * Actions are invoked by the {@see \local_behavioranalytics\local\action\registry}
     * when a user’s risk score meets or exceeds the configured threshold.
     *
     * @param stdClass $user Moodle user record.
     * @param int $score Calculated risk score (0–100).
     * @param string $level Textual risk level (e.g. Low, Medium, High).
     * @return void
     */
    abstract public function execute(stdClass $user, int $score, string $level): void;

    /**
     * Optionally add custom settings for this action.
     *
     * This allows each action class to define its own admin settings, such as
     * notification recipients, thresholds, or toggles.
     *
     * @param admin_settingpage $settings The admin settings page instance.
     * @return void
     */
    public static function add_settings(admin_settingpage $settings): void {
        // Default: no additional settings.
    }
}
