<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Action: inform administrators via Moodle message.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\local\action;

use core_user;
use stdClass;

/**
 * Sends a message notification to site administrators when a high-risk
 * user profile is detected.
 *
 * This action uses Moodle’s core messaging API to send a plain-text
 * notification (`message_send()`) to all administrators returned by
 * {@see get_admins()}.
 *
 * The message provider must be registered in `db/messages.php` under the
 * name `risk_alert`.
 *
 * @package     local_behavioranalytics
 */
class inform_admin extends base {
    /** @var string Unique identifier for this action. */
    public const IDENT = 'inform_admin';

    /**
     * Returns the localised display name of this action.
     *
     * @return string Localised action name.
     */
    public static function get_name(): string {
        return get_string('action_inform_admin', 'local_behavioranalytics');
    }

    /**
     * Execute the administrator notification action.
     *
     * Sends a message to all site administrators informing them that a
     * particular user has exceeded the configured behaviour risk threshold.
     *
     * @param stdClass $user Moodle user record for the affected user.
     * @param int $score Computed risk score (0–100).
     * @param string $level Textual risk level (e.g. “Low”, “Medium”, “High”).
     * @return void
     */
    public function execute(stdClass $user, int $score, string $level): void {
        $admins = get_admins();
        foreach ($admins as $admin) {
            $msg = new \core\message\message();
            $msg->component = 'local_behavioranalytics';
            $msg->name = 'risk_alert';
            $msg->userfrom = core_user::get_noreply_user();
            $msg->userto = $admin;
            $msg->subject = 'Behavior analytics risk alert';
            $msg->fullmessage = "User {$user->username} reached risk score {$score} ({$level}).";
            $msg->fullmessageformat = FORMAT_PLAIN;
            $msg->notification = 1;

            message_send($msg);
        }
    }
}
