<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Action: terminate all active sessions for a user.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\local\action;

use core\session\manager;
use stdClass;

/**
 * Terminates all active sessions for the specified user.
 *
 * This mitigation action forcefully destroys every active Moodle session
 * belonging to a user whose behaviour risk score exceeds the configured
 * threshold. It uses {@see manager::destroy_user_sessions()} to ensure
 * compatibility with Moodle 4.5+.
 *
 * @package     local_behavioranalytics
 */
class kill_sessions extends base {
    /** @var string Unique identifier for this mitigation action. */
    public const IDENT = 'kill_sessions';

    /**
     * Return the localised name of this action.
     *
     * @return string Localised action name.
     */
    public static function get_name(): string {
        return get_string('action_kill_sessions', 'local_behavioranalytics');
    }

    /**
     * Execute the action by destroying all active sessions for a user.
     *
     * This ensures the affected user is immediately logged out from
     * all devices and cannot continue interacting with Moodle until
     * they log in again.
     *
     * @param stdClass $user Moodle user record.
     * @param int $score Computed risk score (0–100).
     * @param string $level Textual risk level (e.g. "Low", "Medium", "High").
     * @return void
     */
    public function execute(stdClass $user, int $score, string $level): void {
        manager::destroy_user_sessions($user->id);
    }
}
