<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Action registry for mitigation execution.
 *
 * Dynamically discovers all action classes under local_behavioranalytics\local\action
 * and executes the selected mitigations when a user exceeds the defined threshold.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @author      Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\local\action;

use context_system;
use local_behavioranalytics\event\mitigation_executed;
use ReflectionClass;
use stdClass;

/**
 * Action registry and executor.
 */
class registry {
    /**
     * Return all available mitigation action classes.
     *
     * @return string[] List of fully qualified class names.
     */
    public static function all(): array {
        $classes = [];
        $dir = __DIR__;
        $files = glob($dir . '/*.php');
        if ($files === false) {
            return [];
        }

        foreach ($files as $file) {
            $name = basename($file, '.php');
            if (in_array($name, ['registry'], true)) {
                continue;
            }
            $fqcn = __NAMESPACE__ . '\\' . $name;
            if (!class_exists($fqcn)) {
                // Lazy load in case it’s not yet declared.
                require_once($file);
            }
            if (class_exists($fqcn)) {
                try {
                    $reflection = new ReflectionClass($fqcn);
                    if ($reflection->isInstantiable() && $reflection->hasConstant('IDENT')) {
                        $classes[] = $fqcn;
                    }
                } catch (\Throwable $e) {
                    debugging('BehaviorAnalytics: failed reflection for ' . $fqcn . ' - ' . $e->getMessage(), DEBUG_DEVELOPER);
                }
            }
        }
        return $classes;
    }

    /**
     * Execute all enabled mitigation actions for a given user.
     *
     * @param stdClass $user The Moodle user object.
     * @param int $score The computed risk score.
     * @param string $level The computed risk level.
     * @return void
     */
    public static function execute_selected(stdClass $user, int $score, string $level): void {
        $enabledraw = (string) get_config('local_behavioranalytics', 'enabledactions');
        $enabled = self::parse_enabled_list($enabledraw);

        if (empty($enabled)) {
            debugging('BehaviorAnalytics: no enabled actions parsed from config.', DEBUG_DEVELOPER);
            return;
        }

        foreach (self::all() as $fqcn) {
            if (!class_exists($fqcn)) {
                debugging("BehaviorAnalytics: missing action class {$fqcn}", DEBUG_DEVELOPER);
                continue;
            }

            $ident = $fqcn::IDENT ?? null;
            if (!$ident || !in_array($ident, $enabled, true)) {
                continue;
            }

            try {
                $action = new $fqcn();
                if (!method_exists($action, 'execute')) {
                    debugging("BehaviorAnalytics: action {$fqcn} missing execute() method.", DEBUG_DEVELOPER);
                    continue;
                }

                // Execute action.
                $action->execute($user, $score, $level);

                // Log audit event.
                $event = mitigation_executed::create([
                    'context'       => context_system::instance(),
                    'objectid'      => $user->id,
                    'relateduserid' => $user->id,
                    'other'         => [
                        'action' => $ident,
                        'score'  => $score,
                        'level'  => $level,
                    ],
                ]);
                $event->trigger();
            } catch (\Throwable $e) {
                debugging("BehaviorAnalytics: error executing action {$fqcn}: " . $e->getMessage(), DEBUG_DEVELOPER);
            }
        }
    }

    /**
     * Parse enabled actions from config.
     *
     * Moodle stores multicheckbox results as a comma-separated string
     * (e.g. "inform_admin,suspend_user"), but we also accept JSON arrays
     * for forward compatibility.
     *
     * @param string $raw Raw config value from get_config().
     * @return string[] Array of enabled action identifiers.
     */
    private static function parse_enabled_list(string $raw): array {
        $raw = trim($raw);
        if ($raw === '') {
            return [];
        }

        // Try JSON first ({"inform_admin":1,"suspend_user":1} or ["inform_admin","suspend_user"]).
        $decoded = json_decode($raw, true);
        if (is_array($decoded)) {
            // Handle both associative and sequential JSON arrays.
            if (array_keys($decoded) === range(0, count($decoded) - 1)) {
                return array_values(array_map('strval', $decoded));
            }
            return array_values(array_map('strval', array_keys(array_filter($decoded))));
        }

        // Fallback: comma-separated list (Moodle default).
        $parts = array_map('trim', explode(',', $raw));
        return array_values(array_filter($parts, static function ($s) {
            return $s !== '';
        }));
    }
}
