<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Action: suspend a user account when risk threshold is exceeded.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\local\action;

use stdClass;

/**
 * Suspends a Moodle user account if not already suspended.
 *
 * This mitigation action updates the {@see user::$suspended} flag to `1`
 * for the specified user when their computed behaviour risk exceeds the
 * configured threshold. Suspension prevents login until manually re-enabled.
 *
 * @package     local_behavioranalytics
 */
class suspend_user extends base {
    /** @var string Unique identifier for this mitigation action. */
    public const IDENT = 'suspend_user';

    /**
     * Return the localised name of this action.
     *
     * @return string Localised action name.
     */
    public static function get_name(): string {
        return get_string('action_suspend_user', 'local_behavioranalytics');
    }

    /**
     * Execute the action by suspending the specified user account.
     *
     * This marks the user record as suspended (`suspended = 1`) in the database
     * if it is not already suspended.
     *
     * @param stdClass $user Moodle user record.
     * @param int $score Calculated risk score (0–100).
     * @param string $level Textual risk level (e.g. "Low", "Medium", "High").
     * @return void
     */
    public function execute(stdClass $user, int $score, string $level): void {
        global $DB;

        if (!empty($user->suspended)) {
            return; // Already suspended.
        }

        $user->suspended = 1;
        $DB->update_record('user', $user);
    }
}
