<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Activity burst detector (N events in M seconds).
 *
 * @package     local_behavioranalytics
 * @category    analytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\local\detector;

use admin_setting_configtext;
use admin_settingpage;
use stdClass;

/**
 * Detects bursts of user activity in a short time window.
 *
 * This detector identifies users performing a high number of actions
 * within a configurable time period, such as posting, logging in, or
 * viewing pages too rapidly. Such behaviour can indicate automated or
 * suspicious account activity.
 *
 * Configuration options:
 * - `activityburst_events`: minimum number of events to trigger detection.
 * - `activityburst_window`: duration of the time window (seconds).
 * - `activity_burst_weight`: relative impact in overall risk scoring.
 *
 * @package     local_behavioranalytics
 * @category    analytics
 */
class activity_burst extends base {
    /** @var string Unique detector identifier. */
    public const IDENT = 'activity_burst';

    /**
     * Get the human-readable detector name.
     *
     * @return string Localised name.
     */
    public static function get_name(): string {
        return get_string('detector_activity_burst', 'local_behavioranalytics');
    }

    /**
     * Detect abnormal activity bursts for a given user.
     *
     * Scans recent logstore entries and flags a user if more than the configured
     * number of events occur within the configured time window.
     *
     * @param stdClass $user Moodle user record.
     * @return array[] List of findings with keys 'risk' (0–100) and 'message'.
     */
    public function detect(stdClass $user): array {
        global $DB;

        if ($user->auth === 'webservice') {
            return [];
        }

        $events = max(1, (int)get_config('local_behavioranalytics', 'activityburst_events'));
        $window = max(10, (int)get_config('local_behavioranalytics', 'activityburst_window'));

        $recent = $DB->get_records(
            'logstore_standard_log',
            ['userid' => $user->id],
            'timecreated DESC',
            'id, timecreated',
            0,
            500
        );

        if (!$recent) {
            return [];
        }

        $times = array_map(static fn($r) => (int)$r->timecreated, $recent);
        sort($times);

        $left = 0;
        $n = count($times);

        for ($right = 0; $right < $n; $right++) {
            while ($times[$right] - $times[$left] > $window) {
                $left++;
            }
            $count = $right - $left + 1;
            if ($count >= $events) {
                return [[
                    'risk' => 60,
                    'message' => get_string(
                        'detector_activity_burst_flag',
                        'local_behavioranalytics',
                        ['events' => $count, 'seconds' => $window]
                    ),
                ]];
            }
        }

        return [];
    }

    /**
     * Add detector-specific admin settings.
     *
     * @param admin_settingpage $settings The settings page to append to.
     * @return void
     */
    public static function add_settings(admin_settingpage $settings): void {
        $settings->add(new admin_setting_configtext(
            'local_behavioranalytics/' . self::IDENT . '_weight',
            get_string('setting_activity_burst_weight', 'local_behavioranalytics'),
            '',
            1.0,
            PARAM_FLOAT
        ));
        $settings->add(new admin_setting_configtext(
            'local_behavioranalytics/activityburst_events',
            get_string('activityburst_events', 'local_behavioranalytics'),
            get_string('activityburst_events_desc', 'local_behavioranalytics'),
            30,
            PARAM_INT
        ));
        $settings->add(new admin_setting_configtext(
            'local_behavioranalytics/activityburst_window',
            get_string('activityburst_window', 'local_behavioranalytics'),
            get_string('activityburst_window_desc', 'local_behavioranalytics'),
            60,
            PARAM_INT
        ));
    }
}
