<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Base class for anomaly detectors.
 *
 * @package     local_behavioranalytics
 * @category    analytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\local\detector;

use admin_settingpage;
use stdClass;

/**
 * Abstract base class for all anomaly detector implementations.
 *
 * Each detector analyses a specific behavioral pattern (e.g., login velocity,
 * activity bursts, etc.) and returns a set of findings containing a risk score
 * and descriptive message. Concrete subclasses must implement {@see detect()}.
 *
 * @package     local_behavioranalytics
 * @category    analytics
 */
abstract class base {
    /** @var string Unique identifier (lowercase snake). */
    public const IDENT = 'base';

    /**
     * Detect anomalies for a single user.
     *
     * Must return items with a normalized 'risk' in range 0–100 and a 'message'.
     *
     * @param stdClass $user Moodle user record.
     * @return array[] Each finding: ['risk' => int 0–100, 'message' => string].
     */
    abstract public function detect(stdClass $user): array;

    /**
     * Human-readable detector name.
     *
     * @return string Localized detector name.
     */
    public static function get_name(): string {
        return static::IDENT;
    }

    /**
     * Optional: add detector-specific settings to the admin UI.
     *
     * @param admin_settingpage $settings The settings page to append to.
     * @return void
     */
    public static function add_settings(admin_settingpage $settings): void {
        // Default: none.
    }
}
