<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Detector: Consecutive failed login attempts.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\local\detector;

use admin_setting_configtext;
use admin_settingpage;
use stdClass;

/**
 * Detects consecutive failed logins indicating brute-force or credential-stuffing attempts.
 *
 * This detector analyses the standard logstore for recent failed login events
 * (`\core\event\user_login_failed`) per user. It counts failures since their
 * last successful login (`\core\event\user_loggedin`) within a configurable time
 * window. If the number of failures exceeds a defined threshold, a proportional
 * risk score is produced.
 */
final class failed_logins extends base {
    /** @var string Unique detector identifier. */
    public const IDENT = 'failed_logins';

    /**
     * Get the localized name for this detector.
     *
     * @return string Localized detector name.
     */
    public static function get_name(): string {
        return get_string('detector_failed_logins', 'local_behavioranalytics');
    }

    /**
     * Detect multiple consecutive failed login attempts for a given user.
     *
     * Queries the Moodle standard logstore for failed login events within the
     * configured time window, and counts consecutive failures since the last
     * successful login.
     *
     * @param stdClass $user Moodle user record (must contain id, username, auth).
     * @return array[] List of associative arrays describing risk findings.
     */
    public function detect(stdClass $user): array {
        global $DB;

        // ---------------------------------------------------------------------
        // Guard conditions (skip irrelevant users).
        // ---------------------------------------------------------------------
        if (
            empty($user->id) ||
            $user->username === 'guest' ||
            $user->auth === 'webservice' ||
            (!empty($user->deleted) && $user->deleted)
        ) {
            return [];
        }

        // ---------------------------------------------------------------------
        // Configuration.
        // ---------------------------------------------------------------------
        $threshold = max(3, (int) get_config('local_behavioranalytics', 'failedlogins_threshold'));
        $window = max(300, (int) get_config('local_behavioranalytics', 'failedlogins_window')); // Default: 5 minutes.
        $since = time() - $window;

        // ---------------------------------------------------------------------
        // Step 1: Get timestamp of the last successful login.
        // ---------------------------------------------------------------------
        $lastsuccess = (int) $DB->get_field_sql(
            "SELECT MAX(timecreated)
               FROM {logstore_standard_log}
              WHERE userid = :userid
                AND eventname = :eventname",
            [
                'userid' => $user->id,
                'eventname' => '\\core\\event\\user_loggedin',
            ]
        );

        // Only count failures that happened after last success.
        $since = max($since, $lastsuccess);

        // ---------------------------------------------------------------------
        // Step 2: Count failed logins since last success within the time window.
        // ---------------------------------------------------------------------
        $failures = $DB->get_records_sql(
            "SELECT id
               FROM {logstore_standard_log}
              WHERE eventname = :eventname
                AND userid = :userid
                AND timecreated >= :since",
            [
                'eventname' => '\\core\\event\\user_login_failed',
                'userid' => $user->id,
                'since' => $since,
            ]
        );

        $failcount = count($failures);

        // ---------------------------------------------------------------------
        // Step 3: Evaluate risk based on failure count.
        // ---------------------------------------------------------------------
        if ($failcount >= $threshold) {
            // Risk grows with count above threshold, capped at 100.
            $excess = min($failcount - $threshold + 1, 10);
            $risk = min(40 + ($excess * 6), 100);

            return [[
                'risk' => $risk,
                'message' => get_string(
                    'detector_failed_logins_flag',
                    'local_behavioranalytics',
                    (object) [
                        'count' => $failcount,
                        'minutes' => floor($window / 60),
                    ]
                ),
            ]];
        }

        return [];
    }

    /**
     * Add admin configuration settings for this detector.
     *
     * @param admin_settingpage $settings The settings page object.
     * @return void
     */
    public static function add_settings(admin_settingpage $settings): void {
        // Weight (importance in overall scoring).
        $settings->add(new admin_setting_configtext(
            'local_behavioranalytics/' . self::IDENT . '_weight',
            get_string('setting_failed_logins_weight', 'local_behavioranalytics'),
            get_string('setting_failed_logins_weight_desc', 'local_behavioranalytics'),
            1.0,
            PARAM_FLOAT
        ));

        // Threshold for failed attempts.
        $settings->add(new admin_setting_configtext(
            'local_behavioranalytics/failedlogins_threshold',
            get_string('failedlogins_threshold', 'local_behavioranalytics'),
            get_string('failedlogins_threshold_desc', 'local_behavioranalytics'),
            5,
            PARAM_INT
        ));

        // Time window in seconds.
        $settings->add(new admin_setting_configtext(
            'local_behavioranalytics/failedlogins_window',
            get_string('failedlogins_window', 'local_behavioranalytics'),
            get_string('failedlogins_window_desc', 'local_behavioranalytics'),
            600,
            PARAM_INT
        ));
    }
}
