<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * IP velocity detector (impossible travel).
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\local\detector;

use admin_setting_configcheckbox;
use admin_setting_configtext;
use admin_settingpage;
use local_behavioranalytics\local\geo\null_provider;
use local_behavioranalytics\local\geo\provider;
use stdClass;

/**
 * Detects suspicious login velocity between IP addresses.
 *
 * This detector estimates the travel speed between consecutive login IPs
 * for a user based on geolocation data. If the implied speed exceeds a
 * configurable threshold, it flags the event as "impossible travel".
 *
 * Configuration options:
 * - {@see enableipvelocity}: enable or disable the detector.
 * - {@see ipvelocity_kmph}: threshold in km/h above which travel is flagged.
 * - {@see ip_velocity_weight}: relative influence in total risk scoring.
 *
 * @package     local_behavioranalytics
 */
class ip_velocity extends base {
    /** @var string Unique detector identifier. */
    public const IDENT = 'ip_velocity';

    /**
     * @var provider|null Geolocation provider implementation used to calculate
     *      distances between IP addresses. Defaults to {@see null_provider} if not supplied.
     */
    private ?provider $geo;

    /**
     * Get the human-readable detector name.
     *
     * @return string Localised name.
     */
    public static function get_name(): string {
        return get_string('detector_ip_velocity', 'local_behavioranalytics');
    }

    /**
     * Constructor.
     *
     * Optionally injects a geolocation provider for distance computation.
     *
     * @param provider|null $geo Geolocation provider implementation (or null for default).
     */
    public function __construct(?provider $geo = null) {
        $this->geo = $geo ?? new null_provider();
    }

    /**
     * Detect improbable travel speeds between logins for the given user.
     *
     * Checks consecutive login records from the standard log store and estimates
     * the implied travel velocity between IP addresses using a geolocation provider.
     * Flags the user if the velocity exceeds the configured threshold.
     *
     * @param stdClass $user Moodle user record.
     * @return array[] List of findings with keys 'risk' (0–100) and 'message'.
     */
    public function detect(stdClass $user): array {
        global $DB;

        if (!get_config('local_behavioranalytics', 'enableipvelocity')) {
            return [];
        }

        $threshold = max(1.0, (float)get_config('local_behavioranalytics', 'ipvelocity_kmph'));

        $logs = $DB->get_records_sql(
            "SELECT id, timecreated, ip
               FROM {logstore_standard_log}
              WHERE userid = :userid AND action = :action
                AND ip IS NOT NULL AND ip <> ''
           ORDER BY timecreated DESC",
            ['userid' => $user->id, 'action' => 'loggedin']
        );

        $prev = null;
        foreach ($logs as $log) {
            if ($prev === null) {
                $prev = $log;
                continue;
            }

            $km = $this->geo->distance_km((string)$prev->ip, (string)$log->ip);
            $hours = max(0.0003, ((int)$prev->timecreated - (int)$log->timecreated) / 3600.0);
            $speed = $km / $hours;

            if ($speed > $threshold) {
                return [[
                    'risk' => 80,
                    'message' => get_string(
                        'detector_ip_velocity_flag',
                        'local_behavioranalytics',
                        [
                            'speed' => format_float($speed, 0),
                            'km' => format_float($km, 0),
                        ]
                    ),
                ]];
            }

            $prev = $log;
        }

        return [];
    }

    /**
     * Add detector-specific admin settings.
     *
     * @param admin_settingpage $settings The settings page to append to.
     * @return void
     */
    public static function add_settings(admin_settingpage $settings): void {
        $settings->add(new admin_setting_configtext(
            'local_behavioranalytics/' . self::IDENT . '_weight',
            get_string('setting_ip_velocity_weight', 'local_behavioranalytics'),
            '',
            1.0,
            PARAM_FLOAT
        ));
        $settings->add(new admin_setting_configcheckbox(
            'local_behavioranalytics/enableipvelocity',
            get_string('enableipvelocity', 'local_behavioranalytics'),
            get_string('enableipvelocity_desc', 'local_behavioranalytics'),
            1
        ));
        $settings->add(new admin_setting_configtext(
            'local_behavioranalytics/ipvelocity_kmph',
            get_string('ipvelocity_kmph', 'local_behavioranalytics'),
            get_string('ipvelocity_kmph_desc', 'local_behavioranalytics'),
            900,
            PARAM_FLOAT
        ));
    }
}
