<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Unusual login time detector.
 *
 * @package     local_behavioranalytics
 * @category    analytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\local\detector;

use admin_setting_configtext;
use admin_settingpage;
use stdClass;

/**
 * Detects users who frequently log in during atypical (nighttime) hours.
 *
 * This detector analyses the login times of users to determine whether a
 * significant proportion of their logins occur during a configurable
 * "night window". Frequent night activity may suggest compromised or
 * suspicious account usage patterns.
 *
 * Configuration options:
 * - `nightstarthour`: beginning of the night window (0–23).
 * - `nightendhour`: end of the night window (0–23).
 * - `unusual_login_time_weight`: relative influence on total risk.
 *
 * @package     local_behavioranalytics
 * @category    analytics
 */
class login_time extends base {
    /** @var string Unique detector identifier. */
    public const IDENT = 'unusual_login_time';

    /**
     * Get the human-readable detector name.
     *
     * @return string Localised name.
     */
    public static function get_name(): string {
        return get_string('detector_login_time', 'local_behavioranalytics');
    }

    /**
     * Detects users logging in predominantly at unusual hours.
     *
     * Evaluates the user's login records and calculates the percentage
     * that occur within the configured "night window". Assigns a risk
     * value if the ratio exceeds 50% or 75%.
     *
     * @param stdClass $user Moodle user record.
     * @return array[] Findings, each with 'risk' (0–100) and 'message'.
     */
    public function detect(stdClass $user): array {
        global $DB;

        $start = (int)get_config('local_behavioranalytics', 'nightstarthour');
        $end   = (int)get_config('local_behavioranalytics', 'nightendhour');

        $logs = $DB->get_records(
            'logstore_standard_log',
            ['userid' => $user->id, 'action' => 'loggedin'],
            'timecreated DESC',
            'id, timecreated'
        );

        $total = count($logs);
        if ($total < 10) {
            return [];
        }

        $night = 0;
        foreach ($logs as $log) {
            $hour = (int)userdate((int)$log->timecreated, '%H');
            if ($this->in_night_window($hour, $start, $end)) {
                $night++;
            }
        }

        $ratio = $night / $total;

        if ($ratio >= 0.75) {
            return [[
                'risk' => 70,
                'message' => get_string(
                    'detector_login_time_high',
                    'local_behavioranalytics',
                    format_float($ratio * 100, 1)
                ),
            ]];
        }

        if ($ratio >= 0.50) {
            return [[
                'risk' => 40,
                'message' => get_string(
                    'detector_login_time_medium',
                    'local_behavioranalytics',
                    format_float($ratio * 100, 1)
                ),
            ]];
        }

        return [];
    }

    /**
     * Determine if a given hour lies within the configured night window.
     *
     * Handles both contiguous ranges (e.g. 22–5) and wrap-around windows
     * that cross midnight.
     *
     * @param int $hour The hour (0–23) to check.
     * @param int $start Start of the night window (0–23).
     * @param int $end End of the night window (0–23).
     * @return bool True if within the night window.
     */
    private function in_night_window(int $hour, int $start, int $end): bool {
        $start = max(0, min(23, $start));
        $end   = max(0, min(23, $end));
        if ($start <= $end) {
            return $hour >= $start && $hour <= $end;
        }
        return $hour >= $start || $hour <= $end;
    }

    /**
     * Add detector-specific admin settings.
     *
     * @param admin_settingpage $settings The settings page to append to.
     * @return void
     */
    public static function add_settings(admin_settingpage $settings): void {
        // Weight configuration.
        $settings->add(new admin_setting_configtext(
            'local_behavioranalytics/' . self::IDENT . '_weight',
            get_string('setting_login_time_weight', 'local_behavioranalytics'),
            '',
            1.0,
            PARAM_FLOAT
        ));

        // Night window configuration.
        $settings->add(new admin_setting_configtext(
            'local_behavioranalytics/nightstarthour',
            get_string('nightstarthour', 'local_behavioranalytics'),
            get_string('nightstarthour_desc', 'local_behavioranalytics'),
            0,
            PARAM_INT
        ));

        $settings->add(new admin_setting_configtext(
            'local_behavioranalytics/nightendhour',
            get_string('nightendhour', 'local_behavioranalytics'),
            get_string('nightendhour_desc', 'local_behavioranalytics'),
            5,
            PARAM_INT
        ));
    }
}
