<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Detector registry (auto-discovery of anomaly detectors).
 *
 * @package     local_behavioranalytics
 * @category    analytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\local\detector;

use ReflectionClass;
use stdClass;

/**
 * Loads all detectors in classes/local/detector dynamically.
 *
 * Contributors can simply add a new detector class file extending
 * {@see \local_behavioranalytics\local\detector\base} and it will be
 * automatically picked up.
 */
class registry {
    /** @var array<string> static cache to avoid repeated scans. */
    private static array $cache = [];

    /**
     * Discover available detector classes.
     *
     * @return string[] Fully qualified class names of detectors.
     */
    public static function all(): array {
        if (self::$cache) {
            return self::$cache;
        }

        $dir = \core_component::get_component_directory('local_behavioranalytics') . '/classes/local/detector';
        if (!is_dir($dir)) {
            return [];
        }

        $detectors = [];
        foreach (glob($dir . '/*.php') as $file) {
            $basename = basename($file, '.php');
            // Skip base and registry.
            if (in_array($basename, ['base', 'registry'], true)) {
                continue;
            }

            $class = 'local_behavioranalytics\\local\\detector\\' . $basename;

            if (!class_exists($class)) {
                require_once($file);
            }

            try {
                $ref = new ReflectionClass($class);
                if ($ref->isAbstract() || !$ref->isSubclassOf(base::class)) {
                    continue;
                }
                $detectors[] = $class;
            } catch (\ReflectionException $e) {
                debugging('Detector reflection failed for ' . $class . ': ' . $e->getMessage(), DEBUG_DEVELOPER);
            }
        }

        self::$cache = $detectors;
        return $detectors;
    }

    /**
     * Run all detectors for a specific user.
     *
     * @param stdClass $user User record.
     * @return array[] List of findings with ['source','ident','risk','message'].
     */
    public static function run_for_user(stdClass $user): array {
        $findings = [];

        foreach (self::all() as $fqcn) {
            /** @var base $detector */
            $detector = new $fqcn();
            foreach ($detector->detect($user) as $finding) {
                $risk = max(0, min(100, (int)($finding['risk'] ?? 0)));
                $message = (string)($finding['message'] ?? '');
                if ($message === '') {
                    continue;
                }
                $findings[] = [
                    'source' => $fqcn::get_name(),
                    'ident'  => (string)$fqcn::IDENT,
                    'risk'   => $risk,
                    'message' => $message,
                ];
            }
        }

        return $findings;
    }
}
